"""Arcade Starter Tools for Github

DO NOT EDIT THIS MODULE DIRECTLY.

THIS MODULE WAS AUTO-GENERATED BY TRANSPILING THE API STARTER TOOL JSON DEFINITIONS
IN THE ../wrapper_tools DIRECTORY INTO PYTHON CODE. ANY CHANGES TO THIS MODULE WILL
BE OVERWRITTEN BY THE TRANSPILER.
"""
import asyncio
import json
import jsonschema
from enum import Enum
from typing import Annotated, Any

import httpx

from arcade_tdk import tool, ToolContext
from arcade_tdk.errors import RetryableToolError
from arcade_tdk.auth import GitHub

from .request_body_schemas import REQUEST_BODY_SCHEMAS

# Retry configuration
INITIAL_RETRY_DELAY = 0.5  # seconds

HTTP_CLIENT = httpx.AsyncClient(
    timeout=httpx.Timeout(60.0, connect=10.0),
    limits=httpx.Limits(max_keepalive_connections=20, max_connections=100),
    transport=httpx.AsyncHTTPTransport(retries=3),
    http2=True,
    follow_redirects=True,
)

class ToolMode(str, Enum):
    """Mode for tools with complex request bodies."""
    GET_REQUEST_SCHEMA = "get_request_schema"
    EXECUTE = "execute"



def remove_none_values(data: dict[str, Any]) -> dict[str, Any]:
    return {k: v for k, v in data.items() if v is not None}


async def make_request(
    url: str,
    method: str,
    params: dict[str, Any] | None = None,
    headers: dict[str, Any] | None = None,
    content: str | None = None,
    data: dict[str, Any] | None = None,
    auth: tuple[str, str] | None = None,
    max_retries: int = 3,
) -> httpx.Response:
    """Make an HTTP request with retry logic for 5xx server errors."""
    for attempt in range(max_retries):
        try:
            response = await HTTP_CLIENT.request(
                url=url,
                auth=auth,
                method=method,
                params=params,
                headers=headers,
                content=content,
            )
            response.raise_for_status()
        except httpx.HTTPStatusError as e:
            # Only retry on 5xx server errors
            if e.response.status_code >= 500 and attempt < max_retries - 1:
                # Exponential backoff: 0.5s, 1s, 2s
                await asyncio.sleep(INITIAL_RETRY_DELAY * (2 ** attempt))
                continue
            # Re-raise for 4xx errors or if max retries reached
            raise
        except httpx.RequestError as e:
            # Don't retry request errors (network issues are handled by transport)
            raise
        else:
            return response

    # This should never be reached, but satisfies type checker
    raise httpx.RequestError("Max retries exceeded")  # noqa: TRY003


async def make_request_with_schema_validation(
    url: str,
    method: str,
    request_data: dict[str, Any],
    schema: dict[str, Any] | str,
    auth: tuple[str, str] | None = None,
    params: dict[str, Any] | None = None,
    headers: dict[str, Any] | None = None,
    max_retries: int = 3,
) -> httpx.Response:
    """Make an HTTP request with schema validation on format errors."""
    # Parse schema if it's a string, skip validation if parsing fails
    parsed_schema = None
    if isinstance(schema, str):
        try:
            parsed_schema = json.loads(schema)
        except Exception:
            # If schema parsing fails, just skip validation
            parsed_schema = None
    else:
        parsed_schema = schema

    try:
        response = await make_request(
            url=url,
            auth=auth,
            method=method,
            params=params,
            headers=headers,
            content=json.dumps(request_data),
            max_retries=max_retries,
        )
    except httpx.HTTPStatusError as e:
        # Only provide schema validation for format-related errors
        if e.response.status_code in (400, 422):
            api_error_details = (
                f"API returned {e.response.status_code}: {e.response.text}"
            )

            # Only run validation if we have a valid parsed schema
            if parsed_schema is not None:
                # Run validation to provide additional context
                is_valid, validation_error = validate_json_against_schema(
                    request_data, parsed_schema
                )

                if not is_valid:
                    # Schema validation found issues - additional context
                    additional_context = (
                        f"{api_error_details}\n\n"
                        f"Schema validation found the following issues:\n"
                        f"{validation_error}"
                    )
                else:
                    # Schema validation passed - just show API error
                    additional_context = api_error_details
            else:
                # No valid schema - just show API error
                additional_context = api_error_details

            raise RetryableToolError(
                message=(
                    f"API request failed with validation error: "
                    f"{e.response.status_code}"
                ),
                developer_message=api_error_details,
                additional_prompt_content=additional_context,
            ) from e
        else:
            # For non-validation errors, re-raise as-is
            raise
    else:
        return response


def validate_json_against_schema(
    json_data: dict[str, Any], schema: dict[str, Any]
) -> tuple[bool, str | None]:
    """Validate JSON data against an OpenAPI/JSON Schema.

    This provides full JSON Schema Draft 7 validation including:
    - Required fields, types, enums
    - Pattern validation (regex)
    - Format validation (email, uuid, date-time, etc.)
    - Min/max length and values
    - oneOf, anyOf, allOf
    - And all other JSON Schema features

    Args:
        json_data: The JSON data to validate
        schema: The JSON Schema to validate against

    Returns:
        Tuple of (is_valid, error_messages). If valid, error_messages is None.
        If invalid, error_messages contains all validation errors.
    """
    try:
        validator = jsonschema.Draft7Validator(
            schema, format_checker=jsonschema.Draft7Validator.FORMAT_CHECKER
        )
        # Collect ALL validation errors
        errors = list(validator.iter_errors(json_data))
        if errors:
            # Format all errors with their paths
            error_messages = []
            for error in errors:
                error_path = (
                    ".".join(str(p) for p in error.path)
                    if error.path
                    else "root"
                )
                error_messages.append(f"{error.message} at {error_path}")
            # Join all errors with newlines
            return False, "\n".join(error_messages)
        else:
            return True, None
    except jsonschema.SchemaError as e:
        return False, f"Invalid schema: {e.message}"
    except Exception as e:
        return False, f"Validation error: {e!s}"

@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_root_links(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'meta/root'."]:
    """Retrieve Hypermedia links to GitHub's REST API resources.

    Call this tool to obtain Hypermedia links that provide access to various resources in GitHub's REST API. Useful for navigating the capabilities and endpoints available in GitHub's API."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_global_webhooks(
    context: ToolContext,
    page_number: Annotated[int | None, "Specify the page number of the results to fetch. This is used to navigate through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to include in each page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-global-webhooks'."]:  # noqa: E501
    """Retrieve a list of global webhooks from GitHub Enterprise.

    Use this tool to obtain a list of all global webhooks configured in a GitHub Enterprise environment. Ideal for administrators managing hooks across the entire enterprise."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_global_webhook(
    context: ToolContext,
    payload_delivery_url: Annotated[str, "The destination URL where the webhook payloads will be delivered. Ensure this URL is accessible and properly configured to handle incoming requests."],  # noqa: E501
    webhook_type: Annotated[str, "Specifies the type of webhook. Must be set to \"web\"."],
    hmac_key_for_signature: Annotated[str | None, "The key for generating the HMAC hex digest in the X-Hub-Signature header. Optional."] = None,  # noqa: E501
    payload_content_type: Annotated[str | None, "Specifies the media type for payload serialization. Options: 'json', 'form'. Default is 'form'."] = None,  # noqa: E501
    send_notifications: Annotated[bool | None, "Set to `true` to send notifications when the webhook is triggered."] = True,  # noqa: E501
    ssl_verification: Annotated[str | None, "Set '0' to verify SSL certificate of the host for the URL; '1' to skip verification. Default is '0'. Avoid setting to '1' to prevent security risks."] = None,  # noqa: E501
    trigger_events: Annotated[list[str] | None, "List of events that trigger the webhook. Default events are `user` and `organization`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-global-webhook'."]:  # noqa: E501
    """Create a global webhook in GitHub Enterprise Admin.

    Use this tool to create a global webhook for GitHub Enterprise. It should be called when you need to set up notifications for all repositories across the organization."""  # noqa: E501
    request_data: Any = {"active": send_notifications, "config": {"content_type": payload_content_type, "insecure_ssl": ssl_verification, "secret": hmac_key_for_signature, "url": payload_delivery_url}, "events": trigger_events, "name": webhook_type}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_global_webhook(
    context: ToolContext,
    webhook_id: Annotated[int, "The unique identifier of the global webhook to be deleted."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-global-webhook'."]:  # noqa: E501
    """Delete a global webhook in GitHub Enterprise.

    Use this tool to delete a global webhook in a GitHub Enterprise setup. It should be called when there's a need to remove a webhook that is no longer required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), hook_id=webhook_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_global_webhook(
    context: ToolContext,
    global_webhook_id: Annotated[int, "The unique identifier for the global webhook in GitHub Enterprise. Provide this ID to retrieve specific webhook details."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-global-webhook'."]:  # noqa: E501
    """Retrieve details of a specific global webhook in GitHub Enterprise.

    Use this tool to obtain detailed information about a specific global webhook in a GitHub Enterprise setup by providing the hook ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), hook_id=global_webhook_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_global_webhook(
    context: ToolContext,
    webhook_identifier: Annotated[int, "The unique integer identifier for the webhook that needs to be updated."],  # noqa: E501
    hmac_secret_key: Annotated[str | None, "Secret key for generating HMAC hex digest value in `X-Hub-Signature` header."] = None,  # noqa: E501
    payload_delivery_url: Annotated[str | None, "The URL where webhook payloads will be delivered for processing."] = None,  # noqa: E501
    payload_media_type: Annotated[str | None, "The media type for payload serialization. Supported values: `json`, `form`. Default is `form`."] = None,  # noqa: E501
    send_notifications_on_trigger: Annotated[bool | None, "Set to `true` to send notifications when the webhook is triggered."] = True,  # noqa: E501
    verify_ssl_certificate: Annotated[str | None, "Determines SSL certificate verification for payload delivery. Use '0' for verification and '1' to skip (not recommended). Default is '0'."] = None,  # noqa: E501
    webhook_trigger_events: Annotated[list[str] | None, "The events that trigger the global webhook. Can include 'user', 'organization'. Defaults to both if not specified."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-global-webhook'."]:  # noqa: E501
    """Update a GitHub enterprise global webhook.

    Use this tool to modify settings of a global webhook in GitHub Enterprise. Parameters not provided will default or be removed if no default exists."""  # noqa: E501
    request_data: Any = {"active": send_notifications_on_trigger, "config": {"content_type": payload_media_type, "insecure_ssl": verify_ssl_certificate, "secret": hmac_secret_key, "url": payload_delivery_url}, "events": webhook_trigger_events}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), hook_id=webhook_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def trigger_github_webhook_ping(
    context: ToolContext,
    webhook_hook_id: Annotated[int, "The unique identifier of the GitHub webhook to ping."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/ping-global-webhook'."]:  # noqa: E501
    """Trigger a ping event to a GitHub webhook.

    This tool sends a ping event to the specified GitHub webhook to test connectivity or configuration. It should be called when it's necessary to verify if the webhook setup is correct."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/hooks/{hook_id}/pings".format(git_server_url=context.get_secret("GIT_SERVER_URL"), hook_id=webhook_hook_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_public_keys(
    context: ToolContext,
    filter_keys_accessed_since: Annotated[str | None, "Specify a timestamp to only list public keys accessed after this time. Use ISO 8601 format."] = None,  # noqa: E501
    page_number_to_fetch: Annotated[int | None, "The specific page number of results to retrieve."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page. Maximum allowed is 100. Use to limit the amount of data retrieved per call."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "The direction to sort the results: 'asc' for ascending or 'desc' for descending."] = "desc",  # noqa: E501
    sort_order: Annotated[str | None, "Criteria for sorting results. Options: 'created', 'updated', 'accessed'."] = "created",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-public-keys'."]:  # noqa: E501
    """Retrieve GitHub Enterprise Admin public keys.

    Use this tool to obtain a list of public keys available for GitHub Enterprise Admin. Ideal for administrators needing to access or audit public keys."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/keys".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number_to_fetch, "direction": sort_direction, "sort": sort_order, "since": filter_keys_accessed_since}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_public_key(
    context: ToolContext,
    public_key_identifier: Annotated[str, "The unique identifier of the public key to delete from GitHub Enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-public-key'."]:  # noqa: E501
    """Delete a public key from GitHub Enterprise.

    Use this tool to delete a specific public key from a GitHub Enterprise account. This should be called when an obsolete or compromised key needs to be removed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/keys/{key_ids}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), key_ids=public_key_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_ldap_mapping_for_team(
    context: ToolContext,
    ldap_distinguished_name: Annotated[str, "The distinguished name (DN) of the LDAP entry to map to a team. This should be a string following the LDAP DN format."],  # noqa: E501
    team_id: Annotated[int, "The unique identifier of the GitHub team to update LDAP mapping for."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-ldap-mapping-for-team'."]:  # noqa: E501
    """Update the LDAP mapping for a GitHub team.

    Use this tool to update the distinguished name (DN) of an LDAP entry mapped to a GitHub team. LDAP synchronization must be enabled for this operation."""  # noqa: E501
    request_data: Any = {"ldap_dn": ldap_distinguished_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/ldap/teams/{team_id}/mapping".format(git_server_url=context.get_secret("GIT_SERVER_URL"), team_id=team_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def queue_ldap_sync_for_team(
    context: ToolContext,
    team_id: Annotated[int, "The unique identifier of the GitHub team for which the LDAP sync job should be queued."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/sync-ldap-mapping-for-team'."]:  # noqa: E501
    """Queue an LDAP sync job for a specified team.

    This tool is used to queue an LDAP sync job for a specific GitHub team. It should be called when you want to sync LDAP mappings for a team. A successful response indicates the job has been queued, not that it has been completed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/ldap/teams/{team_id}/sync".format(git_server_url=context.get_secret("GIT_SERVER_URL"), team_id=team_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_ldap_mapping_for_user(
    context: ToolContext,
    github_user_handle: Annotated[str, "The handle for the GitHub user account to update the LDAP mapping for."],  # noqa: E501
    ldap_distinguished_name: Annotated[str, "The distinguished name (DN) of the LDAP entry to map to a team. It should be in a string format as specified [here](https://www.ldap.com/ldap-dns-and-rdns)."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-ldap-mapping-for-user'."]:  # noqa: E501
    """Update LDAP mapping for a user in GitHub Enterprise Admin.

    This tool is used to update the LDAP mapping for a specific user in the GitHub Enterprise environment. It should be called when there's a need to modify or correct the LDAP information associated with a user's account."""  # noqa: E501
    request_data: Any = {"ldap_dn": ldap_distinguished_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/ldap/users/{username}/mapping".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def sync_github_ldap_user_mapping(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle to queue LDAP sync for."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/sync-ldap-mapping-for-user'."]:  # noqa: E501
    """Queue a sync job for LDAP mapping of a GitHub user.

    Use this tool to queue a job to sync LDAP mapping for a specific GitHub user. This does not initiate immediate synchronization but queues it for when the instance is ready. A response of `201` indicates successful queuing."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/ldap/users/{username}/sync".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_organization(
    context: ToolContext,
    admin_user_login: Annotated[str, "The login username of the user designated to manage the new GitHub organization."],  # noqa: E501
    organization_username: Annotated[str, "The username for the organization in GitHub."],
    organization_display_name: Annotated[str | None, "The display name for the organization to be created on GitHub."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-org'."]:
    """Create a new organization on GitHub.

    This tool is used to create a new organization on GitHub Enterprise. Call it when you need to set up a new organization within a GitHub Enterprise setup."""  # noqa: E501
    request_data: Any = {"admin": admin_user_login, "login": organization_username, "profile_name": organization_display_name}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_org_name(
    context: ToolContext,
    current_organization_name: Annotated[str, "The current name of the organization to be updated. It is not case sensitive."],  # noqa: E501
    new_organization_name: Annotated[str, "The new name for the organization on GitHub Enterprise. This will be the name you want to update to."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-org-name'."]:  # noqa: E501
    """Updates the organization name on GitHub Enterprise.

    Use this tool to update the name of an organization in GitHub Enterprise Administration. Ideal for managing organizational details in GitHub Enterprise settings."""  # noqa: E501
    request_data: Any = {"login": new_organization_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/organizations/{org}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=current_organization_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_pre_receive_environments(
    context: ToolContext,
    page_number: Annotated[int | None, "Specifies the page number of results to fetch. Use this to navigate through paginated data."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
    sort: Annotated[str | None, "Criteria to sort the results: 'created', 'updated', or 'name'."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "Specifies the order to sort results: 'asc' for ascending or 'desc' for descending."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-pre-receive-environments'."]:  # noqa: E501
    """Retrieve a list of pre-receive environments for GitHub Enterprise.

    This tool is used to fetch and list all pre-receive environments in a GitHub Enterprise instance. It should be called when you need to inspect or manage pre-receive environments."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number, "direction": sort_direction, "sort": sort}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_pre_receive_environment(
    context: ToolContext,
    pre_receive_environment_name: Annotated[str, "The name of the new pre-receive environment to be created."],  # noqa: E501
    tarball_download_url: Annotated[str, "URL to download the tarball for the pre-receive environment setup."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-pre-receive-environment'."]:  # noqa: E501
    """Create a new pre-receive environment on GitHub Enterprise.

    This tool is used to create a pre-receive environment in GitHub Enterprise. It should be called when you need to set up an environment for pre-receive hooks."""  # noqa: E501
    request_data: Any = {"image_url": tarball_download_url, "name": pre_receive_environment_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_pre_receive_environment(
    context: ToolContext,
    pre_receive_environment_identifier: Annotated[int, "The unique integer identifier of the pre-receive environment to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-pre-receive-environment'."]:  # noqa: E501
    """Delete a specified pre-receive environment in GitHub Enterprise.

    This tool deletes a specified pre-receive environment by its ID in GitHub Enterprise. If the environment cannot be deleted, a `422 Unprocessable Entity` response is returned with potential error messages indicating that the default environment cannot be modified or deleted, hooks are present, or a download is in progress."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments/{pre_receive_environment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_environment_id=pre_receive_environment_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_pre_receive_environment(
    context: ToolContext,
    pre_receive_environment_id: Annotated[int, "The unique identifier for the GitHub pre-receive environment."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-pre-receive-environment'."]:  # noqa: E501
    """Retrieve a GitHub pre-receive environment by ID.

    Use this tool to get detailed information about a specific pre-receive environment in GitHub for enterprise administration purposes."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments/{pre_receive_environment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_environment_id=pre_receive_environment_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_pre_receive_environment(
    context: ToolContext,
    pre_receive_environment_id: Annotated[int, "The unique identifier of the pre-receive environment to update."],  # noqa: E501
    new_environment_name: Annotated[str | None, "The new name for the pre-receive environment."] = None,  # noqa: E501
    tarball_download_url: Annotated[str | None, "The URL to download the tarball for the environment update."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-pre-receive-environment'."]:  # noqa: E501
    """Update a pre-receive environment in GitHub Enterprise.

    Use this tool to update a specific pre-receive environment in a GitHub Enterprise setup. Note that the default environment cannot be modified."""  # noqa: E501
    request_data: Any = {"image_url": tarball_download_url, "name": new_environment_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments/{pre_receive_environment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_environment_id=pre_receive_environment_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def trigger_environment_download(
    context: ToolContext,
    pre_receive_environment_id: Annotated[int, "The unique identifier for the pre-receive environment to trigger the download for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/start-pre-receive-environment-download'."]:  # noqa: E501
    """Start a new download of the environment tarball.

    This tool triggers a new download of the environment tarball based on the specified `image_url`. Once completed, the downloaded tarball replaces the existing environment. It should be called when a new version of the environment is needed. If a download cannot be started due to constraints like ongoing downloads or protected environments, an error will be provided."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments/{pre_receive_environment_id}/downloads".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_environment_id=pre_receive_environment_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_pre_receive_env_download_status(
    context: ToolContext,
    pre_receive_environment_identifier: Annotated[int, "The unique identifier for the pre-receive environment to retrieve its download status."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-download-status-for-pre-receive-environment'."]:  # noqa: E501
    """Retrieve the latest download status for a pre-receive environment.

    Use this tool to check the most recent download status for a pre-receive environment within GitHub Enterprise Administration. It's useful for monitoring or verifying the status separately from other environment data."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-environments/{pre_receive_environment_id}/downloads/latest".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_environment_id=pre_receive_environment_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_pre_receive_hooks(
    context: ToolContext,
    results_page_number: Annotated[int | None, "The specific page number of results to retrieve, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "The direction to sort the results by. Options are 'asc' for ascending or 'desc' for descending."] = "desc",  # noqa: E501
    sort_results_by: Annotated[str | None, "Specify the property to sort the results by. Options are 'created', 'updated', or 'name'."] = "created",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-pre-receive-hooks'."]:  # noqa: E501
    """Retrieve the list of pre-receive hooks in GitHub Enterprise.

    Use this tool to get a list of all the pre-receive hooks configured in the GitHub Enterprise admin settings."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "direction": sort_direction, "sort": sort_results_by}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_pre_receive_hook(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-pre-receive-hook'."]:  # noqa: E501
    """Create a pre-receive hook for GitHub enterprise administration.

    Use this tool when you need to create a pre-receive hook for managing GitHub enterprise repositories. It's crucial for setting up checks before code is accepted.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEPRERECEIVEHOOK"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters

    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEPRERECEIVEHOOK"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEPRERECEIVEHOOK"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/admin/pre-receive-hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEPRERECEIVEHOOK"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_pre_receive_hook(
    context: ToolContext,
    pre_receive_hook_identifier: Annotated[int, "The unique identifier for the pre-receive hook to delete."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-pre-receive-hook'."]:  # noqa: E501
    """Delete a pre-receive hook from GitHub Enterprise Admin.

    This tool deletes a pre-receive hook from a GitHub Enterprise Server instance. Use it when you need to remove a specific pre-receive hook, identified by its ID, from the administrative settings."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_hook_id=pre_receive_hook_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_pre_receive_hook(
    context: ToolContext,
    pre_receive_hook_id: Annotated[int, "The unique identifier of the pre-receive hook to retrieve details for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-pre-receive-hook'."]:  # noqa: E501
    """Retrieve details of a specific pre-receive hook in GitHub Enterprise Admin.

    This tool is used to get information about a specific pre-receive hook by its ID in GitHub Enterprise. It is helpful for administrators managing repositories who need to view hook configurations."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_hook_id=pre_receive_hook_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_pre_receive_hook(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    pre_receive_hook_id: Annotated[int | None, "The unique identifier of the pre-receive hook in the GitHub enterprise environment. This is required for updating hook details.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-pre-receive-hook'."]:  # noqa: E501
    """Update a GitHub enterprise pre-receive hook.

    Use this tool to update details of a pre-receive hook in a GitHub enterprise environment. This is useful for modifying hook settings or configurations.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBPRERECEIVEHOOK"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not pre_receive_hook_id:
        missing_params.append(("pre_receive_hook_id", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBPRERECEIVEHOOK"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBPRERECEIVEHOOK"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/admin/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), pre_receive_hook_id=pre_receive_hook_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBPRERECEIVEHOOK"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_personal_access_tokens(
    context: ToolContext,
    results_page_number: Annotated[int | None, "The page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results to return per page. Maximum value is 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-personal-access-tokens'."]:  # noqa: E501
    """Retrieve personal access tokens for all users including admins.

    This tool calls the GitHub API to list personal access tokens for all users in an enterprise, including admin users. It's useful for administrators wanting to audit or manage user access tokens."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/tokens".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_personal_access_token(
    context: ToolContext,
    github_token_id: Annotated[int, "The unique identifier of the GitHub personal access token to delete."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-personal-access-token'."]:  # noqa: E501
    """Delete a GitHub personal access token.

    Deletes a specified GitHub personal access token. Use when you need to remove a token. If the token is in use, such as being accessed by the token itself, a '403 - Forbidden' status will be returned."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/tokens/{token_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), token_id=github_token_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_enterprise_user(
    context: ToolContext,
    user_username: Annotated[str, "The user's username for the GitHub enterprise account. It will be normalized to contain only alphanumeric characters or single hyphens."],  # noqa: E501
    user_email: Annotated[str | None, "The email address of the user. Required for built-in authentication but optional for CAS, LDAP, or SAML auth methods."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-user'."]:
    """Creates a new user in GitHub enterprise with external authentication.

    This tool should be called to create a new user in GitHub Enterprise when using external authentication mechanisms like LDAP. It ensures the login name is normalized and manages existing account conflicts."""  # noqa: E501
    request_data: Any = {"email": user_email, "login": user_username}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/users".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_enterprise_user(
    context: ToolContext,
    github_user_handle: Annotated[str, "The handle for the GitHub user account to be deleted. Ensure it's the correct user, as this action is irreversible."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-user'."]:
    """Delete a GitHub Enterprise user and their data.

    Use this tool to permanently delete a user on GitHub Enterprise, including all their data such as repositories and personal settings. Consider if suspending the user would be a better option before using this tool."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/users/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_username(
    context: ToolContext,
    current_github_username: Annotated[str, "The current handle of the GitHub user account to be updated."],  # noqa: E501
    new_github_username: Annotated[str, "The new username for the GitHub user account."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-username-for-user'."]:  # noqa: E501
    """Update a GitHub user's username.

    Use this tool to change the username for a user in a GitHub enterprise environment."""
    request_data: Any = {"login": new_github_username}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/users/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=current_github_username),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_impersonation_oauth_token(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle for which the impersonation OAuth token will be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-impersonation-o-auth-token'."]:  # noqa: E501
    """Deletes an impersonation OAuth token for a user in GitHub Enterprise Admin.

    Use this tool to remove an OAuth token for a specific user's impersonation in GitHub Enterprise Admin. This is useful for security and access management in organization environments."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/users/{username}/authorizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_impersonation_oauth_token(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle for which to create the impersonation OAuth token."],  # noqa: E501
    oauth_scopes_list: Annotated[list[str], "A list of scopes defining the permissions for the OAuth token. Each scope is a string representing a specific set of access rights. Refer to [scopes documentation](https://docs.github.com/enterprise-server@3.8/apps/building-oauth-apps/understanding-scopes-for-oauth-apps/) for valid options."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-impersonation-o-auth-token'."]:  # noqa: E501
    """Create an impersonation OAuth token for a GitHub user.

    This tool is used to generate an impersonation OAuth token for a specified GitHub user, typically for enterprise administration purposes. It should be called when there is a need to access a user's resources on their behalf in an enterprise GitHub environment."""  # noqa: E501
    request_data: Any = {"scopes": oauth_scopes_list}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/admin/users/{username}/authorizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_app_info(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-authenticated'."]:
    """Retrieve details about the authenticated GitHub App.

    This tool returns information about the GitHub App associated with the authentication credentials. It provides details such as the number of installations linked to the app. It's useful to check app configuration and installation status."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def complete_github_app_handshake(
    context: ToolContext,
    temporary_code_for_github_app: Annotated[str, "The temporary code provided during the GitHub App Manifest flow to retrieve the app's id, pem, and webhook_secret."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/create-from-manifest'."]:
    """Complete the GitHub App Manifest handshake to retrieve app details.

    Use this tool to finalize the creation of a GitHub App via the manifest flow, retrieving the app's id, pem (private key), and webhook_secret using a temporary code."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app-manifests/{code}/conversions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), code=temporary_code_for_github_app),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_app_webhook_config(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-webhook-config-for-app'."]:
    """Fetches the webhook configuration for a GitHub App.

    This tool retrieves the webhook configuration details for a specific GitHub App. It should be called when you need to review or manage the webhook settings of your GitHub App. Note that access requires authentication using a JWT."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/hook/config".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_app_webhook_config(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/update-webhook-config-for-app'."]:  # noqa: E501
    """Update the webhook configuration for a GitHub App.

    Use this tool to update the webhook configuration settings for a GitHub App. This requires authentication with a JSON Web Token (JWT).

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBAPPWEBHOOKCONFIG"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters

    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBAPPWEBHOOKCONFIG"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBAPPWEBHOOKCONFIG"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/app/hook/config".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBAPPWEBHOOKCONFIG"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_webhook_deliveries(
    context: ToolContext,
    only_redeliveries: Annotated[bool | None, "Set to true to include only redeliveries in the results."] = None,  # noqa: E501
    pagination_cursor: Annotated[str | None, "Starting point for pagination to fetch a page of deliveries. Use the `link` header to find next and previous page cursors."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of webhook deliveries to return per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/list-webhook-deliveries'."]:
    """Retrieve webhook deliveries for a GitHub App.

    Use this tool to obtain a list of webhook deliveries associated with a GitHub App. Authentication requires a JWT."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/hook/deliveries".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "cursor": pagination_cursor, "redelivery": only_redeliveries}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_app_webhook_delivery(
    context: ToolContext,
    webhook_delivery_id: Annotated[int, "The ID of the webhook delivery to retrieve. Must be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-webhook-delivery'."]:
    """Retrieve delivery details for a GitHub App webhook.

    Fetches the delivery information for a specific webhook configured in a GitHub App. This tool should be called when you need to obtain details about a specific delivery event linked to a webhook. Ensure you are authenticated with a JWT to access this information."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/hook/deliveries/{delivery_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), delivery_id=webhook_delivery_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def redeliver_github_webhook_delivery(
    context: ToolContext,
    webhook_delivery_id: Annotated[int, "The unique integer ID of the webhook delivery to be redelivered."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/redeliver-webhook-delivery'."]:
    """Redeliver a GitHub App webhook delivery.

    Use this tool to redeliver a webhook delivery for a GitHub App. Authentication with a JWT is required for access."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/hook/deliveries/{delivery_id}/attempts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), delivery_id=webhook_delivery_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_app_installations(
    context: ToolContext,
    include_outdated: Annotated[str | None, "Include or exclude outdated installations in the results. Pass 'true' to include them."] = None,  # noqa: E501
    notifications_updated_since: Annotated[str | None, "Specify a timestamp in ISO 8601 format to filter installations updated after this time. Format: `YYYY-MM-DDTHH:MM:SSZ`."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of the results you want to retrieve, allowing pagination through the results. Useful for fetching specific subsets of data."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page. Maximum is 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/list-installations'."]:
    """Retrieve installations of a GitHub app using a JWT.

    Use this tool to get a list of all installations for a GitHub app. The response includes details about the permissions granted to each installation. This is useful for managing and auditing app installations."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/installations".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "since": notifications_updated_since, "outdated": include_outdated}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def uninstall_github_app(
    context: ToolContext,
    installation_id: Annotated[int, "The unique identifier of the GitHub App installation to uninstall."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/delete-installation'."]:
    """Uninstall a GitHub App from an account.

    This tool is used to uninstall a GitHub App from a user, organization, or business account. It should be called when you want to permanently remove an app's access. For temporary suspension, consider using the suspend endpoint instead. Authentication requires a JWT."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/installations/{installation_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), installation_id=installation_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_app_installation_info(
    context: ToolContext,
    installation_id: Annotated[int, "The unique identifier of the GitHub App installation to fetch information for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-installation'."]:
    """Fetch information of a GitHub App installation by ID.

    Use this tool to retrieve detailed information about a specific GitHub App installation using its installation ID. Useful for scenarios where you need to know the configuration or settings of a particular installation."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/installations/{installation_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), installation_id=installation_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_app_installation_token(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    installation_id: Annotated[int | None, "The unique identifier for the GitHub App installation. Required to create the access token.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/create-installation-access-token'."]:  # noqa: E501
    """Create an installation access token for a GitHub App.

    Use this tool to generate an installation access token for a GitHub App, allowing it to make authenticated API requests for its installation on an organization or individual account. Tokens expire in one hour and can be restricted to specific repositories by providing `repository_ids`. Requires JWT authentication.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBAPPINSTALLATIONTOKEN"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not installation_id:
        missing_params.append(("installation_id", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBAPPINSTALLATIONTOKEN"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBAPPINSTALLATIONTOKEN"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/app/installations/{installation_id}/access_tokens".format(git_server_url=context.get_secret("GIT_SERVER_URL"), installation_id=installation_id),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBAPPINSTALLATIONTOKEN"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_github_app_suspension(
    context: ToolContext,
    installation_id: Annotated[int, "The unique identifier for the GitHub App installation to be unsuspended."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/unsuspend-installation'."]:
    """Unsuspend a GitHub App installation.

    This tool removes the suspension of a GitHub App installation. Use this when you need to reactivate an app on GitHub that has been suspended. Requires authentication with a JSON Web Token (JWT) as a GitHub App."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/installations/{installation_id}/suspended".format(git_server_url=context.get_secret("GIT_SERVER_URL"), installation_id=installation_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def suspend_github_app_installation(
    context: ToolContext,
    installation_id: Annotated[int, "The unique identifier of the GitHub App installation to suspend."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/suspend-installation'."]:
    """Suspend a GitHub App's installation for specified accounts.

    Use this tool to suspend a GitHub App on a user, organization, or business account. This blocks the app from accessing the account's resources via the GitHub Enterprise Server API or webhook events. Note: A JWT is required for authentication."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/app/installations/{installation_id}/suspended".format(git_server_url=context.get_secret("GIT_SERVER_URL"), installation_id=installation_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def revoke_github_oauth_grant(
    context: ToolContext,
    github_app_client_id: Annotated[str, "The unique client ID for the GitHub app, used for Basic Authentication."],  # noqa: E501
    oauth_access_token: Annotated[str, "The OAuth access token for authenticating the GitHub API and revoking the grant."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/delete-authorization'."]:
    """Revoke OAuth grant for a GitHub application and user.

    This tool is used to revoke a grant for a specified user's OAuth application on GitHub. It requires Basic Authentication using the application's client_id and client_secret, along with the user's access_token. Once executed, it deletes all OAuth tokens for the user associated with the application, removing the application's access to the user's GitHub account."""  # noqa: E501
    request_data: Any = {"access_token": oauth_access_token}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/applications/{client_id}/grant".format(git_server_url=context.get_secret("GIT_SERVER_URL"), client_id=github_app_client_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def revoke_github_oauth_token(
    context: ToolContext,
    github_app_client_id: Annotated[str, "The client ID of the GitHub OAuth application to identify the app during token revocation."],  # noqa: E501
    oauth_access_token: Annotated[str, "The OAuth access token used to authenticate to the GitHub API. Required for token revocation."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/delete-token'."]:
    """Revoke a GitHub OAuth application's token.

    Use this tool to revoke a single token for a GitHub OAuth application. Requires basic authentication with the application's client ID and client secret."""  # noqa: E501
    request_data: Any = {"access_token": oauth_access_token}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/applications/{client_id}/token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), client_id=github_app_client_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def reset_github_oauth_token(
    context: ToolContext,
    github_app_client_id: Annotated[str, "The client ID of the GitHub application required for resetting the OAuth token."],  # noqa: E501
    oauth_access_token: Annotated[str, "The access token of the OAuth application to be reset."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/reset-token'."]:
    """Reset an OAuth token for a GitHub application.

    Use this tool to reset a valid OAuth token for a GitHub application using Basic Authentication with the application's client_id and client_secret. This operation invalidates the previous token immediately. A 404 error indicates the token was invalid."""  # noqa: E501
    request_data: Any = {"access_token": oauth_access_token}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/applications/{client_id}/token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), client_id=github_app_client_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_github_token_validity(
    context: ToolContext,
    github_app_client_id: Annotated[str, "The unique client ID of the GitHub application for OAuth authentication."],  # noqa: E501
    oauth_access_token: Annotated[str, "The OAuth access token to verify its validity with the GitHub API."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/check-token'."]:
    """Check GitHub OAuth token validity with reduced rate limits.

"""
    request_data: Any = {"access_token": oauth_access_token}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/applications/{client_id}/token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), client_id=github_app_client_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_scoped_token(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    github_client_id: Annotated[str | None, "The client ID of your GitHub app used for authentication.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/scope-token'."]:
    """Create a GitHub repository and permission scoped token.

    This tool generates a repository and/or permission scoped user-to-server access token for GitHub. It requires a non-scoped token and the client's Basic Authentication details. Use this when specific repository access or permission is needed for an app.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBSCOPEDTOKEN"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not github_client_id:
        missing_params.append(("github_client_id", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBSCOPEDTOKEN"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBSCOPEDTOKEN"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/applications/{client_id}/token/scoped".format(git_server_url=context.get_secret("GIT_SERVER_URL"), client_id=github_client_id),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBSCOPEDTOKEN"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_app_details_by_slug(
    context: ToolContext,
    github_app_slug: Annotated[str, "The URL-friendly name of the GitHub App to retrieve details for. Found on the GitHub App settings page."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-by-slug'."]:
    """Retrieve GitHub App details using its slug.

    Use this tool to obtain details about a GitHub App by providing its URL-friendly slug. This can be accessed without authentication for public apps, but requires authentication for private ones using a personal or installation access token."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/apps/{app_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), app_slug=github_app_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_all_github_codes_of_conduct(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'codes-of-conduct/get-all-codes-of-conduct'."]:  # noqa: E501
    """Retrieve all GitHub codes of conduct.

    Call this tool to get a list of all codes of conduct available on GitHub. Useful for reviewing or selecting a code of conduct for a project."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/codes_of_conduct".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_code_of_conduct(
    context: ToolContext,
    conduct_code_key: Annotated[str, "The unique identifier for the specific code of conduct you want to retrieve."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'codes-of-conduct/get-conduct-code'."]:  # noqa: E501
    """Retrieve a specific code of conduct from GitHub.

    This tool retrieves the details of a specific code of conduct using its unique key from GitHub. It should be called when you need information about a particular community guideline."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/codes_of_conduct/{key}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), key=conduct_code_key),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_emojis(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'emojis/get'."]:
    """Lists all available GitHub emojis.

    Use this tool to retrieve a list of all emojis that can be used on GitHub Enterprise Server."""
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/emojis".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_enterprise_announcement(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-announcement'."]:  # noqa: E501
    """Removes the global announcement banner in your enterprise.

    This tool removes the global announcement banner in a GitHub enterprise setup. Use it to clear announcements when they are no longer needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_enterprise_announcement_banner(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-announcement'."]:  # noqa: E501
    """Retrieve the global announcement banner for your enterprise.

    This tool fetches the current message and expiration date of the global announcement banner in your enterprise on GitHub. It is useful for administrators who need to check or display the announcement details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_announcement(
    context: ToolContext,
    announcement_text_gfm: Annotated[str, "The announcement text in GitHub Flavored Markdown. Use for global messages in GitHub Enterprise."],  # noqa: E501
    announcement_expiration_time: Annotated[str | None, "Timestamp for when the announcement expires in ISO 8601 format. Use `null` or an empty string for no expiration."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-announcement'."]:  # noqa: E501
    """Set the announcement banner message and expiration in GitHub Enterprise.

    Use this tool to configure the global announcement banner message and expiration time in your GitHub Enterprise. This is useful for communicating important messages to all users in the enterprise."""  # noqa: E501
    request_data: Any = {"announcement": announcement_text_gfm, "expires_at": announcement_expiration_time}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_license_information(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-license-information'."]:  # noqa: E501
    """Retrieve GitHub Enterprise license information.

    Use this tool to fetch information about your GitHub Enterprise license settings. It is helpful for administrators who need to manage license details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/settings/license".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_enterprise_stats(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-all-stats'."]:
    """Retrieve all statistics for GitHub Enterprise.

    This tool fetches comprehensive statistics related to GitHub Enterprise. It should be used when you need detailed operational insights into the GitHub Enterprise environment."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/all".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_comment_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-comment-stats'."]:  # noqa: E501
    """Retrieve comment statistics from GitHub Enterprise.

    Use this tool to get detailed statistics about comments in a GitHub Enterprise instance. Useful for analyzing comment activity and engagement across repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/comments".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_git_gist_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-gist-stats'."]:
    """Retrieve gist statistics from GitHub Enterprise.

    Use this tool to obtain statistics for gists from GitHub Enterprise. It should be called when you need detailed data about gists usage and distribution."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/gists".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_hooks_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-hooks-stats'."]:  # noqa: E501
    """Retrieves statistics about enterprise webhooks on GitHub.

    Use this tool to obtain detailed statistics on webhook usage and performance within a GitHub Enterprise environment. It helps administrators monitor and analyze webhook activity."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_issue_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-issue-stats'."]:  # noqa: E501
    """Retrieve statistics on GitHub issues for an enterprise.

    This tool fetches statistical data related to issues in a GitHub enterprise account. It should be called when you need insights on issue counts, trends, or other related metrics."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/issues".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_milestone_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-milestone-stats'."]:  # noqa: E501
    """Retrieve GitHub enterprise milestone statistics.

    Use this tool to get statistics about milestones in a GitHub enterprise environment. It is useful for understanding project progress and tracking milestones."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/milestones".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_org_stats(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-org-stats'."]:
    """Retrieve organization statistics from GitHub Enterprise.

    Use this tool to get detailed statistics about organizations in a GitHub Enterprise setup. It's useful for monitoring and analyzing organizational data."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/orgs".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_pages_stats(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-pages-stats'."]:  # noqa: E501
    """Retrieve statistics for GitHub Pages in an enterprise account.

    Use this tool to obtain detailed statistics about GitHub Pages usage and performance within an enterprise account."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/pages".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_pull_request_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-pull-request-stats'."]:  # noqa: E501
    """Retrieve pull request statistics from GitHub Enterprise.

    Use this tool to obtain statistics related to pull requests from a GitHub Enterprise instance."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/pulls".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_repository_statistics(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-repo-stats'."]:
    """Retrieve statistics for GitHub repositories.

    This tool is used to get statistical information about repositories in a GitHub enterprise setting. It should be called when detailed repository stats are needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/repos".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_user_stats(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-user-stats'."]:
    """Retrieve user statistics from GitHub Enterprise.

    This tool retrieves user statistics from a GitHub Enterprise instance. It should be called when needing detailed user data to analyze usage and activity metrics."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprise/stats/users".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_cache_usage_for_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Can be either the enterprise name in slug format or the enterprise ID."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage-for-enterprise'."]:  # noqa: E501
    """Retrieve GitHub Actions cache usage for an enterprise.

    Fetches the total cache usage for GitHub Actions in a specified enterprise. Useful for monitoring and managing enterprise-level actions resources. The data is refreshed every 5 minutes and requires an access token with the 'admin:enterprise' scope for authentication."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/cache/usage".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_cache_policy(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version or ID of the enterprise for GitHub Actions cache policy retrieval."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage-policy-for-enterprise'."]:  # noqa: E501
    """Retrieve the GitHub Actions cache usage policy for an enterprise.

    Fetches the cache usage policy for GitHub Actions associated with a specific enterprise. Requires authentication with an access token that has the `admin:enterprise` scope or a GitHub App with the `enterprise_administration:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/cache/usage-policy".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_actions_cache_policy(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise for which to set the cache policy."],  # noqa: E501
    default_repo_cache_size_limit_gb: Annotated[int | None, "Default size limit for all caches in a repository, specified in gigabytes."] = None,  # noqa: E501
    maximum_repository_cache_size_limit_in_gb: Annotated[int | None, "Maximum cache size limit for all repository caches in an enterprise, in gigabytes."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-actions-cache-usage-policy-for-enterprise'."]:  # noqa: E501
    """Set GitHub Actions cache usage policy for an enterprise.

    This tool sets the cache usage policy for GitHub Actions within a specific enterprise. Requires authentication with an access token having `admin:enterprise` scope or an appropriate GitHub App permission."""  # noqa: E501
    request_data: Any = {"max_repo_cache_size_limit_in_gb": maximum_repository_cache_size_limit_in_gb, "repo_cache_size_limit_in_gb": default_repo_cache_size_limit_gb}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/cache/usage-policy".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_permissions(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "Identifier for the enterprise, either as a slug version of the name or the enterprise ID."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-github-actions-permissions-enterprise'."]:  # noqa: E501
    """Get GitHub Actions permissions for an enterprise.

    Retrieve the GitHub Actions permissions policy and allowed actions for organizations within a specified enterprise. Requires authentication with an access token that has the `admin:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_actions_permissions(
    context: ToolContext,
    enabled_organizations_policy: Annotated[str, "Specifies which organizations can run GitHub Actions: 'all', 'none', or 'selected'."],  # noqa: E501
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise for setting GitHub Actions permissions."],  # noqa: E501
    actions_permission_policy: Annotated[str | None, "Specifies the actions allowed to run in the enterprise. Possible values: 'all', 'local_only', 'selected'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-github-actions-permissions-enterprise'."]:  # noqa: E501
    """Set GitHub Actions permissions for an enterprise.

    Set the permissions policy for GitHub Actions within an enterprise. Requires admin authentication with the `admin:enterprise` scope."""  # noqa: E501
    request_data: Any = {"allowed_actions": actions_permission_policy, "enabled_organizations": enabled_organizations_policy}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_actions_enabled_orgs_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise to identify it for listing organizations with GitHub Actions enabled."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specifies which page of results to retrieve, helpful for navigating through multiple pages."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-selected-organizations-enabled-github-actions-enterprise'."]:  # noqa: E501
    """List organizations with GitHub Actions enabled in an enterprise.

    This tool lists organizations selected to have GitHub Actions enabled within a specified enterprise. It requires the enterprise permission policy for `enabled_organizations` to be set to `selected`. Authentication with an access token with the `admin:enterprise` scope is necessary."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_actions_enabled_orgs(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug version or ID of the enterprise for which to update enabled GitHub Actions organizations."],  # noqa: E501
    organization_ids_for_github_actions: Annotated[list[int], "An array of organization IDs to enable GitHub Actions for specific organizations in the enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-selected-organizations-enabled-github-actions-enterprise'."]:  # noqa: E501
    """Replace organizations enabled for GitHub Actions in an enterprise.

    Use this tool to update the list of organizations enabled for GitHub Actions within a specific enterprise. It requires the enterprise's permission policy to be set to 'selected'. Ensure authentication with an access token having the 'admin:enterprise' scope."""  # noqa: E501
    request_data: Any = {"selected_organization_ids": organization_ids_for_github_actions}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def disable_org_github_actions(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID to identify it for the operation."],  # noqa: E501
    organization_unique_identifier: Annotated[int, "The unique identifier of the organization to disable GitHub Actions for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/disable-selected-organization-github-actions-enterprise'."]:  # noqa: E501
    """Disable GitHub Actions for an organization in an enterprise.

    This tool removes a specified organization from the list enabled for GitHub Actions within an enterprise. It requires the enterprise permission policy for `enabled_organizations` to be set to `selected`, and an access token with `admin:enterprise` scope is needed for authentication. Use this when you need to disable GitHub Actions for an organization in your enterprise."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/organizations/{org_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, org_id=organization_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def enable_github_actions_for_org(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug or ID of the enterprise to identify it for GitHub Actions enablement. Accepts slug version or enterprise ID."],  # noqa: E501
    organization_id: Annotated[int, "The unique identifier for the organization to enable GitHub Actions. Must be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/enable-selected-organization-github-actions-enterprise'."]:  # noqa: E501
    """Enable GitHub Actions for a selected organization in an enterprise.

    This tool adds an organization to the list enabled for GitHub Actions within an enterprise. The enterprise's permissions must allow 'selected' organizations. Requires an access token with 'admin:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/organizations/{org_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id, org_id=organization_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_allowed_actions_for_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version or ID of the enterprise to fetch allowed actions for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-allowed-actions-enterprise'."]:  # noqa: E501
    """Retrieve the actions allowed in a GitHub enterprise.

    This tool gets the selected actions permitted in a specified enterprise. The enterprise's permission policy must be configured to allow selected actions. Authentication with an access token having the `admin:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_allowed_actions_enterprise(
    context: ToolContext,
    allow_github_owned_actions: Annotated[bool, "Set to true to allow GitHub-owned actions in the enterprise, such as those in the 'actions' organization."],  # noqa: E501
    allowed_action_patterns: Annotated[list[str], "List of patterns to match specific GitHub Actions to allow. Use wildcards, tags, and SHAs for specification."],  # noqa: E501
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-allowed-actions-enterprise'."]:  # noqa: E501
    """Configure allowed GitHub Actions for an enterprise.

    This tool sets the allowed GitHub Actions for a given enterprise. It is used when the enterprise's permission policy for `allowed_actions` is set to `selected`. Requires authentication with an `admin:enterprise` scope token."""  # noqa: E501
    request_data: Any = {"github_owned_allowed": allow_github_owned_actions, "patterns_allowed": allowed_action_patterns}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_token_workflow_permissions(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-github-actions-default-workflow-permissions-enterprise'."]:  # noqa: E501
    """Retrieve GitHub Actions default workflow permissions for an enterprise.

    This tool retrieves the default permissions for the `GITHUB_TOKEN` in workflows within a specified enterprise, including settings for pull request review approvals. Use this tool to understand and manage workflow access within your GitHub enterprise."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_enterprise_workflow_permissions(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use the slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    allow_approve_pull_request_reviews: Annotated[bool | None, "Indicate if GitHub Actions can approve pull requests. Enabling this may pose a security risk."] = None,  # noqa: E501
    default_workflow_permissions: Annotated[str | None, "Specify 'read' or 'write' to set the permissions for the GITHUB_TOKEN when running workflows."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-github-actions-default-workflow-permissions-enterprise'."]:  # noqa: E501
    """Set default GitHub Actions permissions for an enterprise.

    Use this tool to set the default workflow permissions for the `GITHUB_TOKEN` in GitHub enterprise workflows, and determine if GitHub Actions can approve pull request reviews. Authentication with an `admin:enterprise` scoped token is required."""  # noqa: E501
    request_data: Any = {"can_approve_pull_request_reviews": allow_approve_pull_request_reviews, "default_workflow_permissions": default_workflow_permissions}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_self_hosted_runner_groups(
    context: ToolContext,
    enterprise_slug: Annotated[str, "The slug or ID of the enterprise for which to list self-hosted runner groups. This identifies the enterprise by name or numeric ID."],  # noqa: E501
    organization_filter: Annotated[str | None, "Filter results to show runner groups usable by the specified organization."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Specifies which page of the self-hosted runner groups results to retrieve. Useful for paginating through large sets of data."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The maximum number of results to return per page, up to 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-self-hosted-runner-groups-for-enterprise'."]:  # noqa: E501
    """Retrieve all self-hosted runner groups for an enterprise.

    Fetches a list of all self-hosted runner groups for a specified enterprise. Requires authentication with an access token having the 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "visible_to_organization": organization_filter}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_self_hosted_runner_group(
    context: ToolContext,
    enterprise_name_or_id: Annotated[str, "The slug version of the enterprise name or the enterprise ID to identify the enterprise."],  # noqa: E501
    runner_group_name: Annotated[str, "Name of the runner group to be created."],
    allow_public_repository_use: Annotated[bool | None, "Set to true to allow the runner group to be used by public repositories."] = False,  # noqa: E501
    organization_ids_for_access: Annotated[list[int] | None, "List of IDs for organizations allowed to access the runner group."] = None,  # noqa: E501
    restrict_to_selected_workflows: Annotated[bool | None, "Set to true to restrict the runner group to only the workflows in 'selected_workflows'."] = False,  # noqa: E501
    runner_group_visibility: Annotated[str | None, "Specifies the visibility of the runner group: 'selected' for individual organizations or 'all' for all organizations."] = None,  # noqa: E501
    runner_ids: Annotated[list[int] | None, "List of runner IDs to be added to the new runner group."] = None,  # noqa: E501
    workflows_allowed_for_runner_group: Annotated[list[str] | None, "List of workflows the runner group can run. Ignored unless `restricted_to_workflows` is `true`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-self-hosted-runner-group-for-enterprise'."]:  # noqa: E501
    """Create a self-hosted runner group for an enterprise.

    Use this tool to create a new self-hosted runner group for an enterprise on GitHub. Requires authentication with an access token having the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = {"allows_public_repositories": allow_public_repository_use, "name": runner_group_name, "restricted_to_workflows": restrict_to_selected_workflows, "runners": runner_ids, "selected_organization_ids": organization_ids_for_access, "selected_workflows": workflows_allowed_for_runner_group, "visibility": runner_group_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_name_or_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_self_hosted_runner_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID to identify which enterprise the runner group belongs to."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier for the self-hosted runner group to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-self-hosted-runner-group-from-enterprise'."]:  # noqa: E501
    """Delete a self-hosted runner group for an enterprise.

    Use this tool to delete a self-hosted runner group from an enterprise. It requires authentication with an access token that has the `manage_runners:enterprise` scope. Call this tool when you need to manage enterprise runner groups on GitHub."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_self_hosted_runner_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug of the enterprise name or substitute with the enterprise ID."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Unique identifier of the self-hosted runner group, required to retrieve specific group details within an enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-self-hosted-runner-group-for-enterprise'."]:  # noqa: E501
    """Retrieve a specific self-hosted runner group for an enterprise.

    Use this tool to get details of a particular self-hosted runner group within an enterprise. Authentication with an access token is required, including the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_runner_group_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise to identify which enterprise the runner group belongs to."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group to update."],
    allow_public_repositories: Annotated[bool | None, "Set to true if the runner group can be used by public repositories."] = False,  # noqa: E501
    allowable_workflow_list: Annotated[list[str] | None, "List of workflow names the runner group is allowed to run. Ignored unless `restricted_to_workflows` is set to `true`."] = None,  # noqa: E501
    restrict_to_workflows: Annotated[bool | None, "Set to true to restrict the runner group to only the workflows in the selected_workflows array."] = False,  # noqa: E501
    runner_group_name: Annotated[str | None, "Name of the self-hosted runner group to be updated."] = None,  # noqa: E501
    runner_group_visibility: Annotated[str | None, "Specifies whether the runner group is visible to all organizations or only selected ones. Valid values are 'selected' or 'all'."] = "all",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-self-hosted-runner-group-for-enterprise'."]:  # noqa: E501
    """Update the name and visibility of a self-hosted runner group in an enterprise.

    Use this tool to update the name and visibility settings of a self-hosted runner group within a GitHub enterprise. This requires authentication with a token that has the 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = {"allows_public_repositories": allow_public_repositories, "name": runner_group_name, "restricted_to_workflows": restrict_to_workflows, "selected_workflows": allowable_workflow_list, "visibility": runner_group_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_access_runner_group(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug or ID of the enterprise. Used to identify the specific enterprise context for the runner group."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier for the self-hosted runner group."],
    page_number: Annotated[int | None, "Specify the page number of results to fetch. Useful for pagination in large datasets."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-org-access-to-self-hosted-runner-group-in-enterprise'."]:  # noqa: E501
    """List organizations with access to a self-hosted runner group.

    Retrieves a list of organizations that have access to a specific self-hosted runner group in an enterprise. Requires authentication with a token that has 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_gh_org_access_to_runner_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Specify either for identifying the enterprise."],  # noqa: E501
    organization_ids_for_runner_access: Annotated[list[int], "List of organization IDs permitted to access the self-hosted runner group."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Unique integer identifier of the self-hosted runner group within the enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-org-access-to-self-hosted-runner-group-in-enterprise'."]:  # noqa: E501
    """Update organization access for a GitHub runner group.

    This tool allows updating which organizations have access to a self-hosted runner group within a GitHub enterprise. It requires authentication with a token that has the 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = {"selected_organization_ids": organization_ids_for_runner_access}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_org_access_runner_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise to identify it for the runner group operation."],  # noqa: E501
    organization_id: Annotated[int, "The unique identifier for the organization to be removed from the self-hosted runner group."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group to modify access for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-org-access-to-self-hosted-runner-group-in-enterprise'."]:  # noqa: E501
    """Removes an organization's access to a self-hosted runner group.

    Use this tool to remove an organization from accessing a specific self-hosted runner group in an enterprise. Requires authentication with a token having `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/organizations/{org_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id, org_id=organization_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_org_access_to_runner_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use either the enterprise name slug or its ID."],  # noqa: E501
    organization_id: Annotated[int, "The unique identifier of the organization to add access to the self-hosted runner group."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier for the self-hosted runner group. Required for adding organization access to it."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/add-org-access-to-self-hosted-runner-group-in-enterprise'."]:  # noqa: E501
    """Add organization access to a self-hosted runner group in an enterprise.

    Use this tool to add an organization to a selected list that can access a self-hosted runner group. The runner group's visibility must be set to 'selected'. Requires an access token with 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/organizations/{org_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id, org_id=organization_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_self_hosted_runners_in_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use the slug version of the enterprise name or substitute with the enterprise ID."],  # noqa: E501
    runner_group_id: Annotated[int, "The unique identifier of the self-hosted runner group within the enterprise."],  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch."] = 1,
    results_per_page: Annotated[int | None, "The number of results to display per page (maximum 100)."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-self-hosted-runners-in-group-for-enterprise'."]:  # noqa: E501
    """Retrieve self-hosted runners in an enterprise group.

    Use this tool to list the self-hosted runners associated with a specific group in an enterprise on GitHub. Authentication with an access token that has the 'manage_runners:enterprise' scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_self_hosted_runners_in_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Unique identifier for the self-hosted runner group. This integer ID specifies which group to update."],  # noqa: E501
    runner_ids_to_add: Annotated[list[int], "Array of runner IDs to be added to the specified runner group in the enterprise. Each runner ID should be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-self-hosted-runners-in-group-for-enterprise'."]:  # noqa: E501
    """Update self-hosted runners in an enterprise group.

    This tool replaces the list of self-hosted runners in a specified enterprise runner group. It requires authentication with an access token having the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = {"runners": runner_ids_to_add}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_runner_from_enterprise_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use either the slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group to remove a runner from."],  # noqa: E501
    runner_identifier: Annotated[int, "The unique integer identifier of the self-hosted runner to be removed from the enterprise group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-self-hosted-runner-from-group-for-enterprise'."]:  # noqa: E501
    """Remove a self-hosted runner from an enterprise group.

    This tool removes a self-hosted runner from a specified group in an enterprise and returns it to the default group. It requires authentication using an access token with the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_id, runner_id=runner_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_self_hosted_runner_to_group(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "Slug version or ID of the enterprise for adding the runner."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Unique identifier of the self-hosted runner group."],
    self_hosted_runner_id: Annotated[int, "Unique identifier of the self-hosted runner to be added to the group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/add-self-hosted-runner-to-group-for-enterprise'."]:  # noqa: E501
    """Add a self-hosted runner to an enterprise group in GitHub.

    Use this tool to add a self-hosted runner to a specified runner group within an enterprise on GitHub. Authentication with an access token having the 'manage_runners:enterprise' scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runner-groups/{runner_group_id}/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_group_id=runner_group_identifier, runner_id=self_hosted_runner_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_self_hosted_runners_for_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise for which to list self-hosted runners."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of the results to fetch for pagination purposes."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-self-hosted-runners-for-enterprise'."]:  # noqa: E501
    """Retrieve all self-hosted runners for a GitHub enterprise.

    Use this tool to obtain a list of all self-hosted runners configured for a specified GitHub enterprise. Authentication with an access token having the `manage_runners:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_runner_binaries_for_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID for the enterprise to obtain runner binaries."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-runner-applications-for-enterprise'."]:  # noqa: E501
    """Retrieve download links for runner application binaries.

    Use this tool to list available binaries for GitHub runner applications that can be downloaded and run for a specified enterprise. Authentication with an access token having the `manage_runners:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/downloads".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_enterprise_registration_token(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug version of the enterprise name or the enterprise ID for GitHub Enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-registration-token-for-enterprise'."]:  # noqa: E501
    """Generate a registration token for GitHub Enterprise runners.

    Use this tool to obtain a registration token for configuring self-hosted GitHub Enterprise runners. The token expires after one hour and requires authentication with a token having the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/registration-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_remove_token_for_enterprise_runner(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use this to specify the enterprise for which you want to generate a removal token."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/create-remove-token-for-enterprise'."]:  # noqa: E501
    """Generates a token to remove a self-hosted runner from an enterprise.

    Use this tool to obtain a token for removing a self-hosted runner from an enterprise on GitHub. The token is valid for one hour and requires authentication with an access token having the 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/remove-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_self_hosted_runner_from_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Use the slug version of the name or the enterprise ID for identification."],  # noqa: E501
    runner_id: Annotated[int, "Unique identifier for the self-hosted runner to be removed."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/delete-self-hosted-runner-from-enterprise'."]:  # noqa: E501
    """Remove a self-hosted runner from an enterprise.

    Use this tool to forcefully remove a self-hosted runner from an enterprise when the machine no longer exists. Authentication with an access token having the `manage_runners:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_self_hosted_runner_info(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    runner_id: Annotated[int, "The unique identifier of the self-hosted runner to retrieve details for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Retrieve details of a self-hosted runner in an enterprise.

    Fetches information about a specific self-hosted runner configured within an enterprise on GitHub. Requires authentication with an access token having `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_custom_labels_from_runner(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID for identifying the enterprise."],  # noqa: E501
    runner_id: Annotated[int, "The unique identifier of the self-hosted runner from which to remove custom labels."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-all-custom-labels-from-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Remove all custom labels from an enterprise's self-hosted runner.

    This tool removes all custom labels from a self-hosted runner configured in an enterprise using GitHub's API. It is used when you need to reset or clear custom labels from a runner, and it returns the runner’s remaining read-only labels. Authentication with a token that has `manage_runners:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_labels_for_runner(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug or ID of the enterprise for identifying self-hosted runner."],  # noqa: E501
    runner_id: Annotated[int, "Unique identifier of the self-hosted runner for which to list labels."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-labels-for-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Retrieve all labels for a self-hosted runner in an enterprise.

    Use this tool to get a list of all labels associated with a specific self-hosted runner within an enterprise on GitHub. Authentication with an access token having `manage_runners:enterprise` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_custom_labels_to_runner(
    context: ToolContext,
    custom_labels: Annotated[list[str], "An array of names for the custom labels to add to the self-hosted runner."],  # noqa: E501
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise for the self-hosted runner. It identifies the enterprise name or ID in GitHub."],  # noqa: E501
    runner_unique_id: Annotated[int, "The unique integer identifier for the self-hosted runner."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/add-custom-labels-to-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Add custom labels to a self-hosted runner in an enterprise.

    Use this tool to attach custom labels to a self-hosted runner within an enterprise on GitHub. Requires authentication with a token that has the `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = {"labels": custom_labels}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=runner_unique_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_custom_labels_for_self_hosted_runner(
    context: ToolContext,
    custom_labels: Annotated[list[str], "List of new custom labels for the runner. Use an empty list to remove all labels."],  # noqa: E501
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise where the runner is configured. Use the slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    self_hosted_runner_id: Annotated[int, "The unique identifier of the self-hosted runner to update labels for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/set-custom-labels-for-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Set custom labels for a self-hosted runner in an enterprise.

    This tool updates the custom labels for a specific self-hosted runner in a GitHub enterprise, removing all previous labels. Requires an access token with `manage_runners:enterprise` scope."""  # noqa: E501
    request_data: Any = {"labels": custom_labels}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=self_hosted_runner_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_custom_label_from_runner(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "Slug or ID of the enterprise. Use the slug version of the enterprise name or the enterprise ID as an alternative."],  # noqa: E501
    runner_custom_label_name: Annotated[str, "The name of the custom label to be removed from the self-hosted runner."],  # noqa: E501
    runner_identifier: Annotated[int, "Unique identifier for the self-hosted runner."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-custom-label-from-self-hosted-runner-for-enterprise'."]:  # noqa: E501
    """Remove a custom label from a self-hosted runner in an enterprise.

    This tool removes a specified custom label from a self-hosted runner configured in an enterprise and returns the remaining labels. It should be called when you need to update runner labels, and requires authentication with an access token having 'manage_runners:enterprise' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/actions/runners/{runner_id}/labels/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, runner_id=runner_identifier, name=runner_custom_label_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_enterprise_audit_log(
    context: ToolContext,
    enterprise_slug_or_id: Annotated[str, "The slug or ID of the enterprise to fetch the audit log for. Either the slug version of the enterprise name or the enterprise ID can be used."],  # noqa: E501
    after_cursor: Annotated[str | None, "A cursor from the Link header to search for events after this point."] = None,  # noqa: E501
    audit_log_event_order: Annotated[str | None, "Specify 'desc' for newest events first or 'asc' for oldest events first. Default is 'desc'."] = None,  # noqa: E501
    event_types_to_include: Annotated[str | None, "Specify event types to include: 'web' for web events, 'git' for Git events, or 'all' for both. Defaults to 'web'."] = None,  # noqa: E501
    result_page_number: Annotated[int | None, "The page number of audit log results to fetch."] = 1,
    results_per_page: Annotated[int | None, "Specify the number of results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
    search_events_before_cursor: Annotated[str | None, "A cursor to filter events occurring before the specified position in the audit log."] = None,  # noqa: E501
    search_phrase: Annotated[str | None, "A search phrase to filter audit log entries. Refer to [GitHub Docs](https://docs.github.com/enterprise-server@3.8/admin/monitoring-activity-in-your-enterprise/reviewing-audit-logs-for-your-enterprise/searching-the-audit-log-for-your-enterprise#searching-the-audit-log) for more details."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-audit-log'."]:
    """Retrieve the audit log for a specified enterprise.

    This tool fetches the audit log for an enterprise on GitHub. It is intended for use by enterprise admins with proper access rights. The audit log provides insights into various activities and changes within the enterprise."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/audit-log".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_slug_or_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"phrase": search_phrase, "include": event_types_to_include, "after": after_cursor, "before": search_events_before_cursor, "order": audit_log_event_order, "page": result_page_number, "per_page": results_per_page}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_enterprise_code_scanning_alerts(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version or ID of the enterprise. Used to specify which enterprise's alerts to list."],  # noqa: E501
    alert_state_filter: Annotated[str | None, "Filter code scanning alerts by their state. Valid values are 'open', 'closed', 'dismissed', 'fixed'."] = None,  # noqa: E501
    code_scanning_tool_guid: Annotated[str | None, "The GUID of a code scanning tool to filter alerts. Use either `code_scanning_tool_guid` or `code_scanning_tool_name`, but not both."] = None,  # noqa: E501
    code_scanning_tool_name: Annotated[str | None, "The name of a code scanning tool to filter results. Use either this or `tool_guid`, but not both."] = None,  # noqa: E501
    query_results_before_cursor: Annotated[str | None, "The cursor from the Link header to search for alerts before this point."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number to fetch results from. Used for pagination of code scanning alerts."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of alerts to return per page, with a maximum of 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "Specifies the sorting order of the results, either ascending ('asc') or descending ('desc')."] = "desc",  # noqa: E501
    sort_property: Annotated[str | None, "Specify the property to sort the results by. Valid values are 'created' or 'updated'."] = "created",  # noqa: E501
    start_after_cursor: Annotated[str | None, "Specifies the cursor to return results after this point. Utilize the cursor from the Link header."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'code-scanning/list-alerts-for-enterprise'."]:  # noqa: E501
    """Retrieve code scanning alerts for enterprise repositories.

    This tool is used to list code scanning alerts for the default branch of all eligible repositories within an enterprise. It requires the user to be a member of the enterprise and have an access token with the necessary scopes (`repo` or `security_events`). Use this tool to monitor security alerts across repositories owned by organizations you manage as a security manager."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/code-scanning/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"tool_name": code_scanning_tool_name, "tool_guid": code_scanning_tool_guid, "before": query_results_before_cursor, "after": start_after_cursor, "page": results_page_number, "per_page": results_per_page, "direction": sort_direction, "state": alert_state_filter, "sort": sort_property}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_security_analysis_settings(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version or ID of the enterprise. Use the enterprise's slug name or its ID to specify it."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'secret-scanning/get-security-analysis-settings-for-enterprise'."]:  # noqa: E501
    """Get security analysis settings for an enterprise.

    Retrieve code security and analysis settings for a specified enterprise. Requires admin access to the enterprise and an access token with the `admin:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/code_security_and_analysis".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_security_settings_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. Accepts the enterprise's slug name or ID for identification."],  # noqa: E501
    enable_dependabot_alerts_for_new_repositories: Annotated[bool | None, "Set to true to automatically enable Dependabot alerts for new repositories."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'secret-scanning/patch-security-analysis-settings-for-enterprise'."]:  # noqa: E501
    """Update security and scanning settings for enterprise repositories.

    Use this tool to update the advanced security, secret scanning, and push protection settings for new repositories within an enterprise. Ensure you have admin access to the enterprise and an access token with the `admin:enterprise` scope."""  # noqa: E501
    request_data: Any = {"dependabot_alerts_enabled_for_new_repositories": enable_dependabot_alerts_for_new_repositories}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/code_security_and_analysis".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_enterprise_dependabot_alerts(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. This identifies the enterprise for which alerts are listed."],  # noqa: E501
    alert_severities: Annotated[str | None, "A comma-separated list of alert severities to filter results. Options: `low`, `medium`, `high`, `critical`."] = None,  # noqa: E501
    alert_state_filter: Annotated[str | None, "Comma-separated list of alert states to filter by: `dismissed`, `fixed`, `open`. Only alerts with these states will be returned."] = None,  # noqa: E501
    before_cursor: Annotated[str | None, "Specify a cursor to fetch results before this point. Use the format provided in the Link header."] = None,  # noqa: E501
    cursor_after: Annotated[str | None, "A cursor to return results after a specific point. Use as given in the Link header for pagination."] = None,  # noqa: E501
    ecosystem_list: Annotated[str | None, "A comma-separated list of ecosystems to filter alerts. Options include: `composer`, `go`, `maven`, `npm`, `nuget`, `pip`, `pub`, `rubygems`, `rust`."] = None,  # noqa: E501
    number_of_results_per_page_starting_first: Annotated[int | None, "**Deprecated**. Specifies the number of results per page (maximum 100), beginning with the first matching result. Avoid using with `number_of_results_per_page_starting_last`. Use `results_per_page` with `paging_after_cursor` instead."] = 30,  # noqa: E501
    package_filter: Annotated[str | None, "A comma-separated list of package names. Specify to filter alerts by these packages."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
    results_per_page_starting_from_last: Annotated[int | None, "**Deprecated**. Number of results per page (max 100), starting from the last matching result. Avoid using with 'results_per_page_starting_from_first'."] = None,  # noqa: E501
    sort_direction: Annotated[str | None, "Specifies the sorting order of the results, either ascending ('asc') or descending ('desc')."] = "desc",  # noqa: E501
    sort_property_for_alerts: Annotated[str | None, "Specifies the property to sort Dependabot alerts by. Options are 'created' (when the alert was created) or 'updated' (when the alert's state last changed)."] = "created",  # noqa: E501
    vulnerable_dependency_scope: Annotated[str | None, "The scope of the vulnerable dependency to filter alerts by. Options: 'development', 'runtime'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/list-alerts-for-enterprise'."]:  # noqa: E501
    """Get Dependabot alerts for enterprise-owned repositories.

    This tool retrieves a list of Dependabot alerts for repositories owned by a specified enterprise on GitHub. It is used by members of the enterprise with appropriate access tokens. Alerts are restricted to organizations where the user is an owner or security manager."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/dependabot/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"state": alert_state_filter, "severity": alert_severities, "ecosystem": ecosystem_list, "package": package_filter, "scope": vulnerable_dependency_scope, "sort": sort_property_for_alerts, "direction": sort_direction, "before": before_cursor, "after": cursor_after, "first": number_of_results_per_page_starting_first, "last": results_per_page_starting_from_last, "per_page": results_per_page}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_secret_scanning_alerts_for_enterprise(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID of the enterprise. This identifies the specific enterprise to list alerts for."],  # noqa: E501
    alert_resolution_filters: Annotated[str | None, "Comma-separated list of alert resolutions to filter by: false_positive, wont_fix, revoked, pattern_edited, pattern_deleted, used_in_tests."] = None,  # noqa: E501
    alert_state_filter: Annotated[str | None, "Specify if only 'open' or 'resolved' secret scanning alerts should be listed."] = None,  # noqa: E501
    cursor_after: Annotated[str | None, "A cursor for retrieving results after this point, as specified in the Link header."] = None,  # noqa: E501
    cursor_before: Annotated[str | None, "A cursor to fetch results before this point, as specified by the link header."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page. Maximum is 100."] = 30,
    secret_types_to_return: Annotated[str | None, "A comma-separated list of secret types to return. By default, all secret types are returned. Refer to GitHub's secret scanning patterns documentation for supported types."] = None,  # noqa: E501
    sort_by: Annotated[str | None, "Specify 'created' to sort by alert creation date or 'updated' to sort by the last update or resolution date."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "Specify the direction ('asc' or 'desc') to sort the secret scanning alerts results."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'secret-scanning/list-alerts-for-enterprise'."]:  # noqa: E501
    """Retrieve secret scanning alerts for enterprise repositories.

    This tool fetches secret scanning alerts for eligible repositories within an enterprise, presented from newest to oldest. Users must be members of the enterprise and possess the appropriate access token with `repo` or `security_events` scope. Alerts are only available for organizations where the user has ownership or security management permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/secret-scanning/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"state": alert_state_filter, "secret_type": secret_types_to_return, "resolution": alert_resolution_filters, "sort": sort_by, "direction": sort_direction, "per_page": results_per_page, "before": cursor_before, "after": cursor_after}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_security_billing_info(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug or ID representing the enterprise name for security billing info."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of results to fetch. Use to navigate through paginated data."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page to return. Maximum is 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'billing/get-github-advanced-security-billing-ghe'."]:  # noqa: E501
    """Retrieve GitHub Advanced Security billing details for an enterprise.

    This tool retrieves data on active committers using GitHub Advanced Security for an enterprise, listed per repository. It provides information on the number of distinct user logins and the total repositories with committer information."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/settings/billing/advanced-security".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def manage_enterprise_security_feature(
    context: ToolContext,
    enterprise_identifier: Annotated[str, "The slug version of the enterprise name or the enterprise ID."],  # noqa: E501
    security_feature: Annotated[str, "Specify the security feature to enable or disable. Options: 'advanced_security', 'secret_scanning', 'secret_scanning_push_protection'."],  # noqa: E501
    set_enablement_status: Annotated[str, "Specify 'enable_all' to activate or 'disable_all' to deactivate the security feature for all repositories in the enterprise."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'secret-scanning/post-security-product-enablement-for-enterprise'."]:  # noqa: E501
    """Enable or disable a security feature for an enterprise.

    Use this tool to enable or disable a specified security feature across all repositories within an enterprise. Requires administrative access and an access token with the `admin:enterprise` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/enterprises/{enterprise}/{security_product}/{enablement}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), enterprise=enterprise_identifier, security_product=security_feature, enablement=set_enablement_status),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_recent_github_events(
    context: ToolContext,
    page_number: Annotated[int | None, "The page number of the GitHub public event results to fetch."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/list-public-events'."]:
    """Retrieve recent public events from GitHub.

    Obtain the most recent public events from GitHub, with events delayed by at least five minutes."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/events".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_feeds(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/get-feeds'."]:
    """Retrieve available GitHub feeds for an authenticated user.

    This tool retrieves a list of timeline resources available on GitHub Enterprise Server in Atom format. It includes global public timelines, public and private timelines for users, private timelines for user activities and organizations, and security advisories. Private feeds require Basic Auth for retrieval."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/feeds".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_user_gists(
    context: ToolContext,
    page_number_to_fetch: Annotated[int | None, "Specify the page number of the results to fetch."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of gists to return per page. Maximum is 100."] = 30,  # noqa: E501
    show_gists_since: Annotated[str | None, "Show gists updated after the specified time in ISO 8601 format: `YYYY-MM-DDTHH:MM:SSZ`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list'."]:
    """Lists a user's gists or public gists if unauthenticated.

    This tool retrieves the list of gists for the authenticated user. If no authentication is provided, it returns all public gists. Useful for gathering a user's code snippets or public contributions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"since": show_gists_since, "per_page": results_per_page, "page": page_number_to_fetch}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_gist(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/create'."]:
    """Create a new gist with one or more files on GitHub.

    Use this tool to add a new gist containing one or more code files on GitHub. Avoid naming files with the 'gistfile' format followed by a number, as it may conflict with GitHub's internal naming scheme.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGIST"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters

    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGIST"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGIST"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/gists".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGIST"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_recent_public_gists(
    context: ToolContext,
    result_page_number: Annotated[int | None, "Page number to fetch the results from. Use for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of gists to display per page. Maximum allowed is 100."] = 30,  # noqa: E501
    updated_since: Annotated[str | None, "A timestamp in ISO 8601 format to filter gists updated after this time."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list-public'."]:
    """Retrieve the most recently updated public gists.

    Call this tool to get a list of public gists on GitHub sorted by the most recent updates. Useful for accessing the latest shared code snippets or projects made publicly available on GitHub."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/public".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"since": updated_since, "per_page": results_per_page, "page": result_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_starred_gists(
    context: ToolContext,
    results_page_number: Annotated[int | None, "Specifies which page of results to fetch for the user's starred gists."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page. Maximum allowed is 100."] = 30,  # noqa: E501
    updated_since_time: Annotated[str | None, "Only show gists updated after this time. Use ISO 8601 format: `YYYY-MM-DDTHH:MM:SSZ`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list-starred'."]:
    """Retrieve the authenticated user's starred gists.

    Call this tool to get the authenticated user's starred gists on GitHub. Useful for accessing gists that the user has marked as starred."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/starred".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"since": updated_since_time, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_gist(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier for the GitHub gist to be deleted."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/delete'."]:
    """Delete a GitHub gist by its ID.

    Use this tool to delete a specific GitHub gist by providing its gist ID. Useful for managing personal gists on GitHub."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_gist(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier for the GitHub gist you want to retrieve."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/get'."]:
    """Retrieve details of a specific GitHub gist using its ID.

    Use this tool to obtain information about a particular GitHub gist by providing the gist ID. It is helpful when you need to access the content, description, or other metadata of a GitHub gist."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_gist(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    gist_unique_identifier: Annotated[str | None, "The unique identifier of the GitHub gist to be updated.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/update'."]:
    """Update a GitHub gist's description and files.

    Use this tool to change the description, update, delete, or rename files in an existing GitHub gist. Unchanged files remain the same.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBGIST"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not gist_unique_identifier:
        missing_params.append(("gist_unique_identifier", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBGIST"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBGIST"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/gists/{gist_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBGIST"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_gist_comments(
    context: ToolContext,
    gist_identifier: Annotated[str, "Provide the unique identifier of the gist for which comments are to be listed."],  # noqa: E501
    result_page_number: Annotated[int | None, "Specifies the page number to fetch from the list of gist comments."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of comments to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list-comments'."]:
    """Retrieve comments for a specific GitHub gist.

    This tool is used to fetch comments associated with a specified GitHub gist, identified by its gist ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/comments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_gist_comment(
    context: ToolContext,
    comment_text: Annotated[str, "The text content of the comment to be added to the gist."],
    gist_unique_identifier: Annotated[str, "The unique identifier of the gist to comment on."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/create-comment'."]:
    """Create a comment on a GitHub gist.

    This tool is used to add a comment to a specific GitHub gist by providing the necessary gist ID and comment content."""  # noqa: E501
    request_data: Any = {"body": comment_text}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/comments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_gist_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "The unique identifier of the comment to be deleted."],
    gist_identifier: Annotated[str, "The unique identifier for the specific GitHub gist from which the comment is to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/delete-comment'."]:
    """Delete a comment from a GitHub gist.

    This tool deletes a specific comment from a GitHub gist when provided with the gist and comment IDs. It should be called when a user needs to remove an undesired or outdated comment."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/comments/{comment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier, comment_id=comment_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_gist_comment(
    context: ToolContext,
    comment_id: Annotated[int, "Unique identifier for the gist comment to retrieve details for."],
    gist_unique_id: Annotated[str, "The unique identifier for the gist needed to retrieve a specific comment."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/get-comment'."]:
    """Retrieve a specific comment from a GitHub gist.

    Use this tool to obtain details about a specific comment on a GitHub gist using the gist and comment IDs."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/comments/{comment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_id, comment_id=comment_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_gist_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "The unique identifier of the comment to update."],
    comment_text: Annotated[str, "The text content of the gist comment to be updated."],
    gist_identifier: Annotated[str, "The unique identifier for the GitHub gist you want to update a comment on."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/update-comment'."]:
    """Update an existing comment on a GitHub gist.

    Use this tool to update a comment on a specific GitHub gist. Ideal for modifying content of a gist comment when changes are necessary."""  # noqa: E501
    request_data: Any = {"body": comment_text}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/comments/{comment_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier, comment_id=comment_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_gist_commits(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier of the gist for which to list commits."],
    results_page_number: Annotated[int | None, "Specify the page number of results to fetch for pagination. Useful for accessing more than the default result set."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list-commits'."]:
    """Retrieve the commit history of a specified GitHub gist.

    Use this tool to obtain a list of all commits associated with a specific gist on GitHub. Ideal for tracking changes and updates made to the gist content over time."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/commits".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_gist_forks(
    context: ToolContext,
    gist_unique_identifier: Annotated[str, "The unique identifier of the gist to retrieve forks information from."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of the results you want to fetch. Used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100 allowed."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/list-forks'."]:
    """Retrieve a list of forks for a specific gist.

    Use this tool to get information about the forks of a specific gist on GitHub by providing the gist ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/forks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fork_gist(
    context: ToolContext,
    gist_unique_identifier: Annotated[str, "The unique identifier of the GitHub gist to be forked. This is required to specify which gist to duplicate."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/fork'."]:
    """Fork a GitHub gist to your account.

    Use this tool to create a fork of an existing GitHub gist into your account. This can be useful for modifying or duplicating code snippets while keeping track of the original version."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/forks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def unstar_github_gist(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier for the GitHub gist to be unstarred."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/unstar'."]:
    """Unstar a GitHub gist by its ID.

    This tool is used to remove a star from a specific GitHub gist identified by its ID. Call this tool when you need to unstar a gist to update its starred status. It confirms the action has been completed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/star".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_if_gist_is_starred(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier for the gist to check if it is starred."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/check-is-starred'."]:
    """Determine if a specific gist is starred on GitHub.

    This tool checks the starred status of a specific GitHub gist using its gist ID."""
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/star".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def star_github_gist(
    context: ToolContext,
    gist_unique_id: Annotated[str, "The unique identifier of the gist to be starred on GitHub."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/star'."]:
    """Star a gist on GitHub using its gist ID.

    Use this tool to star a specific gist on GitHub by providing its gist ID. This action marks the gist as starred for the authenticated user."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/star".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_unique_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_gist_revision(
    context: ToolContext,
    gist_identifier: Annotated[str, "The unique identifier of the GitHub gist to retrieve a specific revision."],  # noqa: E501
    revision_sha: Annotated[str, "The SHA hash of the specific gist revision to retrieve."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gists/get-revision'."]:
    """Retrieve a specific revision of a GitHub gist.

    Use this tool to get details about a specific revision of a GitHub gist using the gist ID and SHA."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gists/{gist_id}/{sha}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), gist_id=gist_identifier, sha=revision_sha),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_gitignore_templates(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gitignore/get-all-templates'."]:
    """Retrieve all available .gitignore templates from GitHub.

    Use this tool to obtain a list of all .gitignore templates that can be used when creating a new repository on GitHub. This can help in setting up repositories with the appropriate exclusions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gitignore/templates".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fetch_gitignore_template(
    context: ToolContext,
    gitignore_template_name: Annotated[str, "The name of the .gitignore template to fetch from GitHub. This is required to specify which template's raw content to retrieve."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'gitignore/get-template'."]:
    """Fetches the raw .gitignore template by name.

    Use this tool to retrieve the raw source of a specified .gitignore template from GitHub. It is useful when you need to include or reference specific ignore rules for different development environments."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/gitignore/templates/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), name=gitignore_template_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_app_accessible_repos(
    context: ToolContext,
    results_page_number: Annotated[int | None, "Page number of the results to fetch."] = 1,
    results_per_page: Annotated[int | None, "The number of repositories to include on each page of results. Maximum is 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/list-repos-accessible-to-installation'."]:  # noqa: E501
    """List repositories accessible to a GitHub app installation.

    Use this tool to retrieve a list of repositories that a GitHub app installation can access. Requires an installation access token for authentication."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/installation/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def revoke_github_installation_token(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/revoke-installation-access-token'."]:  # noqa: E501
    """Revoke your GitHub installation access token.

    This tool revokes the GitHub installation token currently being used for authentication. Once revoked, the token becomes invalid and cannot be used for further authentication. This is useful when you need to invalidate an access token immediately."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/installation/token".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_issues(
    context: ToolContext,
    include_collaborative_repositories: Annotated[bool | None, "Include issues from collaborative repositories. Set to true to filter issues where you have collaborative access."] = None,  # noqa: E501
    include_owned_repositories: Annotated[bool | None, "Set to true to include issues from repositories owned by the authenticated user."] = None,  # noqa: E501
    include_pull_requests: Annotated[bool | None, "Set to true to include pull requests in the issues list."] = None,  # noqa: E501
    issue_filter_type: Annotated[str | None, "Specifies the type of issues to return: assigned, created, mentioned, subscribed, repos, or all."] = "assigned",  # noqa: E501
    issue_labels: Annotated[str | None, "Comma-separated list of label names to filter issues. Example: 'bug,ui,@high'."] = None,  # noqa: E501
    issue_state: Annotated[str | None, "Specifies the state of issues to retrieve: `open`, `closed`, or `all`."] = "open",  # noqa: E501
    organization_repositories: Annotated[bool | None, "Include issues from organization repositories when set to true. If false, include issues from all repositories."] = None,  # noqa: E501
    page_number: Annotated[int | None, "The specific page of results to fetch. Use this for pagination to navigate through large list of issues."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page, maximum is 100."] = 30,
    sort_direction: Annotated[str | None, "Specifies the order of sorting for the results. Use 'asc' for ascending and 'desc' for descending."] = "desc",  # noqa: E501
    sort_issues_by: Annotated[str | None, "Specify the criteria to sort issues by: 'created', 'updated', or 'comments'."] = "created",  # noqa: E501
    updated_since_timestamp: Annotated[str | None, "Show issues updated after this timestamp in ISO 8601 format: YYYY-MM-DDTHH:MM:SSZ."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'issues/list'."]:
    """Get issues assigned to you across all GitHub repositories.

    This tool retrieves a list of issues that are assigned to the authenticated user from all visible repositories on GitHub, including owned, member, and organization repositories. It can be used to monitor workload and track tasks across projects. Note that the results may include pull requests as well, identified by a `pull_request` key."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/issues".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"filter": issue_filter_type, "state": issue_state, "labels": issue_labels, "sort": sort_issues_by, "direction": sort_direction, "since": updated_since_timestamp, "collab": include_collaborative_repositories, "orgs": organization_repositories, "owned": include_owned_repositories, "pulls": include_pull_requests, "per_page": results_per_page, "page": page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_commonly_used_licenses(
    context: ToolContext,
    only_featured_licenses: Annotated[bool | None, "Set to true to return only featured licenses."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch. Use to navigate through pages of results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of licenses to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'licenses/get-all-commonly-used'."]:
    """Fetch a list of commonly used software licenses.

    Retrieves a collection of the most commonly used open-source software licenses from GitHub."""
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/licenses".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"featured": only_featured_licenses, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_license(
    context: ToolContext,
    license_key: Annotated[str, "The key of the GitHub license to retrieve information about."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'licenses/get'."]:
    """Retrieve a specific GitHub license by its key.

    Use this tool to get details about a specific license on GitHub by specifying its key. Useful when you need information about license terms and conditions for a repository."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/licenses/{license}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), license=license_key),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def render_markdown(
    context: ToolContext,
    markdown_text: Annotated[str, "The Markdown text to convert into HTML format."],
    rendering_mode: Annotated[str | None, "Specifies the rendering mode: 'markdown' for plain Markdown or 'gfm' for GitHub Flavored Markdown."] = "markdown",  # noqa: E501
    repository_context: Annotated[str | None, "The repository context for linking references in `gfm` mode (e.g., 'octo-org/octo-repo')."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'markdown/render'."]:
    """Convert Markdown content to HTML rendering.

    Use this tool to convert Markdown text into HTML format, suitable for displaying on web pages or applications."""  # noqa: E501
    request_data: Any = {"context": repository_context, "mode": rendering_mode, "text": markdown_text}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/markdown".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def render_markdown_plain(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'markdown/render-raw'."]:
    """Convert Markdown text to rendered plain text format.

    Use this tool to convert Markdown text to a plain-text format without GitHub Flavored Markdown support. Ideal for rendering Markdown as it appears in README.md files. The content size must be 400 KB or less."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/markdown/raw".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_enterprise_meta_info(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'meta/get'."]:
    """Retrieve GitHub Enterprise Server meta information.

    This tool is used to obtain meta information about a GitHub Enterprise Server. It should be called when details about the server's configuration and capabilities are needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/meta".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repo_network_public_events(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This value is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The GitHub account owner of the repository. This value is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of results to retrieve. Use this to navigate through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/list-public-events-for-repo-network'."]:  # noqa: E501
    """Retrieve public events for a network of repositories.

    Use this tool to get the latest public events associated with a network of repositories on GitHub. It is useful for monitoring activity across multiple related repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/networks/{owner}/{repo}/events".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_user_notifications(
    context: ToolContext,
    filter_notifications_before_date: Annotated[str | None, "Only show notifications updated before the specified ISO 8601 timestamp (`YYYY-MM-DDTHH:MM:SSZ`)."] = None,  # noqa: E501
    include_read_notifications: Annotated[bool | None, "Set to `true` to include notifications marked as read in the results."] = False,  # noqa: E501
    notifications_since_timestamp: Annotated[str | None, "Return notifications updated after this timestamp in ISO 8601 format: YYYY-MM-DDTHH:MM:SSZ."] = None,  # noqa: E501
    only_show_participating_notifications: Annotated[bool | None, "If true, only shows notifications where the user is directly participating or mentioned."] = False,  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of the notification results to fetch."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of notifications to display per page, with a maximum limit of 50."] = 50,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/list-notifications-for-authenticated-user'."]:  # noqa: E501
    """Retrieve notifications for the authenticated GitHub user.

    This tool retrieves all notifications for the authenticated GitHub user, sorted by most recently updated. It should be called when the user wants to check their latest notifications."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"all": include_read_notifications, "participating": only_show_participating_notifications, "since": notifications_since_timestamp, "before": filter_notifications_before_date, "page": results_page_number, "per_page": results_per_page}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def mark_github_notifications_as_read(
    context: ToolContext,
    mark_notifications_as_read: Annotated[bool | None, "A boolean flag to set notifications as read. True marks notifications as read."] = None,  # noqa: E501
    notifications_last_read_timestamp: Annotated[str | None, "A timestamp marking the last time notifications were checked. Notifications updated since this time won't be marked as read. Use ISO 8601 format `YYYY-MM-DDTHH:MM:SSZ`. Leave empty to mark all as read."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/mark-notifications-as-read'."]:  # noqa: E501
    """Mark all GitHub notifications as read for the current user.

    Use this tool to mark all notifications as 'read' in GitHub for the authenticated user. If there are too many notifications, the process may run asynchronously, and any remaining 'unread' notifications can be checked using a separate endpoint."""  # noqa: E501
    request_data: Any = {"last_read_at": notifications_last_read_timestamp, "read": mark_notifications_as_read}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_notification_thread(
    context: ToolContext,
    notification_thread_id: Annotated[int, "The unique identifier of the notification thread, returned in the `id` field when retrieving notifications."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/get-thread'."]:
    """Retrieve information about a GitHub notification thread.

    Use this tool to get detailed information about a specific notification thread on GitHub. It's helpful for users who need to track updates or manage their notifications."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications/threads/{thread_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), thread_id=notification_thread_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def mark_github_thread_as_read(
    context: ToolContext,
    notification_thread_id: Annotated[int, "The unique identifier for the GitHub notification thread to be marked as read. Obtain this from the `id` field when retrieving notifications."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/mark-thread-as-read'."]:
    """Mark a GitHub thread notification as read.

    Use this tool to mark a GitHub notification thread as read. It mirrors the action of clicking a notification in the GitHub notification inbox."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications/threads/{thread_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), thread_id=notification_thread_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def mute_github_thread_notifications(
    context: ToolContext,
    notification_thread_id: Annotated[int, "The unique identifier for the specific GitHub notification thread to mute. Obtain from the `id` field of fetched notifications."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/delete-thread-subscription'."]:  # noqa: E501
    """Mute all future notifications for a GitHub thread.

    Use this tool to mute notifications for a specific GitHub conversation thread. Notifications will be silenced until you comment on the thread or receive an @mention. This tool is helpful if you want to focus on relevant updates without distractions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications/threads/{thread_id}/subscription".format(git_server_url=context.get_secret("GIT_SERVER_URL"), thread_id=notification_thread_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_thread_subscription(
    context: ToolContext,
    notification_thread_id: Annotated[int, "The unique identifier of the notification thread, retrieved from notification data."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/get-thread-subscription-for-authenticated-user'."]:  # noqa: E501
    """Check if the authenticated user is subscribed to a thread.

    This tool determines if the current user is subscribed to a specific thread on GitHub. Subscriptions are present if the user is participating in the conversation, such as replying, being @mentioned, or manually subscribing to the thread."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications/threads/{thread_id}/subscription".format(git_server_url=context.get_secret("GIT_SERVER_URL"), thread_id=notification_thread_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def manage_github_thread_notifications(
    context: ToolContext,
    notification_thread_id: Annotated[int, "The unique ID of the notification thread, as retrieved from the GitHub notifications API."],  # noqa: E501
    ignore_thread_notifications: Annotated[bool | None, "Set to true to block all notifications from a thread. Use false to allow notifications."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/set-thread-subscription'."]:  # noqa: E501
    """Manage GitHub notifications for specific threads.

    Use this tool to ignore future notifications for GitHub threads, or to subscribe to threads you aren't receiving notifications for. Ideal for managing notifications on watched repositories by ignoring or subscribing to specific threads."""  # noqa: E501
    request_data: Any = {"ignored": ignore_thread_notifications}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/notifications/threads/{thread_id}/subscription".format(git_server_url=context.get_secret("GIT_SERVER_URL"), thread_id=notification_thread_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_octocat_ascii_art(
    context: ToolContext,
    speech_bubble_text: Annotated[str | None, "Text to display in Octocat's speech bubble. Provide a string with the desired message."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'meta/get-octocat'."]:
    """Retrieve the octocat as ASCII art.

    Use this tool to get the octocat character depicted in ASCII art from GitHub."""
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/octocat".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"s": speech_bubble_text}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_organizations(
    context: ToolContext,
    organization_id_since: Annotated[int | None, "Only return organizations with an ID greater than this value to paginate results."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of results to return per page, with a maximum value of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list'."]:
    """Retrieve a list of GitHub organizations.

    Fetches all organizations from GitHub Enterprise Server in the order they were created. Supports pagination using the 'since' parameter."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/organizations".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({"since": organization_id_since, "per_page": results_per_page}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_custom_roles(
    context: ToolContext,
    organization_id: Annotated[str, "The unique identifier of the GitHub organization to list custom roles for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-custom-roles'."]:
    """Retrieve custom repository roles for a GitHub organization.

    Retrieve a list of available custom repository roles in a GitHub organization if authenticated as an organization owner or administrator with the appropriate permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/organizations/{organization_id}/custom_roles".format(git_server_url=context.get_secret("GIT_SERVER_URL"), organization_id=organization_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_org_info(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization to retrieve information for. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get'."]:
    """Retrieve detailed information about a GitHub organization.

    This tool retrieves information about a specific GitHub organization, such as its plan and two-factor authentication requirements. It requires authentication as an organization owner or a GitHub App with the appropriate permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_organization(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The GitHub organization name. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/update'."]:
    """Update a GitHub organization's profile and member privileges.

    Allows an authenticated organization owner to update the organization's profile and set member privileges using the `admin:org` scope. New parameters enable more granular permissions for repository creation.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGANIZATION"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGANIZATION"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGANIZATION"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGANIZATION"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fetch_org_actions_cache_usage(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It should not be case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage-for-org'."]:  # noqa: E501
    """Fetches GitHub Actions cache usage for a specified organization.

    This tool retrieves the total cache usage of GitHub Actions for a specified organization. The data is updated every 5 minutes. Authentication with the 'read:org' scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/cache/usage".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_cache_usage_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization to retrieve cache usage for. This is not case sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "Specify the page number of the results to fetch. This is useful for paginated data retrieval."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum of 100 entries."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage-by-repo-for-org'."]:  # noqa: E501
    """Retrieve GitHub Actions cache usage for an organization's repositories.

    Use this tool to get a list of repositories and their GitHub Actions cache usage for a specified organization. The data is refreshed every 5 minutes. Requires authentication with an access token with the `read:org` scope or a GitHub App with `organization_administration:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/cache/usage-by-repository".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_oidc_custom_sub_template_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization for which you want to retrieve the OIDC customization template. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'oidc/get-oidc-custom-sub-template-for-org'."]:  # noqa: E501
    """Retrieves the OIDC subject claim customization template for an organization.

    Use this tool to get the customization template for an OpenID Connect (OIDC) subject claim for a specific organization. This requires authentication with a GitHub access token with `read:org` scope or a GitHub App with `organization_administration:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/oidc/customization/sub".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_oidc_template(
    context: ToolContext,
    include_claim_keys: Annotated[list[str], "Array of unique strings for OIDC claim keys with alphanumeric characters and underscores."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. Case insensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'oidc/update-oidc-custom-sub-template-for-org'."]:  # noqa: E501
    """Update OIDC custom subject claim template for GitHub organization.

    This tool updates or creates the customization template for an OpenID Connect (OIDC) subject claim for a GitHub organization. Requires an access token with the `write:org` scope or GitHub Apps with `admin:org` permission."""  # noqa: E501
    request_data: Any = {"include_claim_keys": include_claim_keys}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/oidc/customization/sub".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_permissions_for_organization(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name for which to retrieve GitHub Actions permissions. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-github-actions-permissions-organization'."]:  # noqa: E501
    """Retrieve GitHub Actions permissions for an organization.

    Fetches the GitHub Actions permissions policy for repositories and allowed actions within a specified organization. Requires authentication with an access token having the `admin:org` scope or a GitHub App with `administration` organization permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def configure_github_actions_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_execution_policy: Annotated[str, "Specifies which repositories in the organization are allowed to run GitHub Actions. Options: 'all', 'none', 'selected'."],  # noqa: E501
    allowed_actions_policy: Annotated[str | None, "Specifies the permissions policy for actions: 'all', 'local_only', or 'selected'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-github-actions-permissions-organization'."]:  # noqa: E501
    """Configure GitHub Actions permissions for an organization.

    Use this tool to set the GitHub Actions permissions policy for repositories and allowed actions within an organization. Note that enterprise-level restrictions cannot be overridden. Requires authentication with an 'admin:org' token or 'administration' organization permission for GitHub Apps."""  # noqa: E501
    request_data: Any = {"allowed_actions": allowed_actions_policy, "enabled_repositories": repository_execution_policy}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_actions_enabled_repos(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization for which to list GitHub Actions-enabled repositories. This field is not case sensitive."],  # noqa: E501
    result_page_number: Annotated[int | None, "Page number of the results to fetch. Use for paginating through results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The maximum number of repositories to return per page (max 100)."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-selected-repositories-enabled-github-actions-organization'."]:  # noqa: E501
    """Retrieve repositories enabled for GitHub Actions in an organization.

    Use this tool to list all repositories where GitHub Actions are enabled within a specific organization. Ensure the organization permission policy for `enabled_repositories` is set to `selected`. Authentication with a token having `admin:org` scope or a GitHub App with `administration` permission is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_actions_repos_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    repository_ids_for_github_actions: Annotated[list[int], "An array of repository IDs to enable for GitHub Actions within the organization. Each ID should be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-selected-repositories-enabled-github-actions-organization'."]:  # noqa: E501
    """Configure selected repositories for GitHub Actions in an organization.

    This tool replaces the list of repositories with GitHub Actions enabled in a specified organization. Use it when you need to manage which repositories have access to GitHub Actions, provided the organization policy allows it. Requires authentication with an access token granting `admin:org` scope or a GitHub App with `administration` permission."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": repository_ids_for_github_actions}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def disable_github_actions_repo(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_unique_id: Annotated[int, "The unique integer identifier of the repository to be disabled for GitHub Actions."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/disable-selected-repository-github-actions-organization'."]:  # noqa: E501
    """Disable GitHub Actions for a specific repo in an organization.

    Use this tool to remove a repository from the selected list of repositories enabled for GitHub Actions within an organization. This is applicable when the organization's repository permission policy is set to 'selected'. Authentication with an access token having 'admin:org' scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, repository_id=repository_unique_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def enable_github_actions_for_repo(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization to enable GitHub Actions for, not case sensitive."],  # noqa: E501
    repository_unique_identifier: Annotated[int, "The unique identifier of the repository to enable for GitHub Actions."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/enable-selected-repository-github-actions-organization'."]:  # noqa: E501
    """Enable a repository for GitHub Actions in an organization.

    Adds a repository to the list of selected repositories enabled for GitHub Actions within an organization. Requires organization permission policy for `enabled_repositories` set to `selected`. Authentication with `admin:org` scope is necessary."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, repository_id=repository_unique_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_allowed_actions_for_organization(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-allowed-actions-organization'."]:  # noqa: E501
    """Retrieve the allowed GitHub Actions for an organization.

    Call this tool to get the list of selected GitHub Actions permitted for an organization, provided the organization policy is set to allow selected actions only. Requires an authenticated access token with `admin:org` scope or GitHub App with `administration` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_allowed_actions_for_organization(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The name of the GitHub organization. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-allowed-actions-organization'."]:  # noqa: E501
    """Set allowed GitHub Actions for an organization.

    This tool sets which GitHub Actions are permitted in an organization, given the organization's permission policy is configured to 'selected'. It requires an access token with the 'admin:org' scope or GitHub App permissions for 'administration'. Use it when you need to update allowed actions based on organization or enterprise policies.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETALLOWEDACTIONSFORORGANIZATION"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETALLOWEDACTIONSFORORGANIZATION"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETALLOWEDACTIONSFORORGANIZATION"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETALLOWEDACTIONSFORORGANIZATION"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_default_github_actions_workflow_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name for which to fetch the workflow permissions. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-github-actions-default-workflow-permissions-organization'."]:  # noqa: E501
    """Fetches default workflow permissions for an organization's GitHub Actions.

    This tool retrieves the default workflow permissions granted to the GITHUB_TOKEN for running workflows in a specified organization, and checks if GitHub Actions can submit approving pull request reviews. Requires authentication with an access token having 'admin:org' scope or 'administration' permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_default_github_actions_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. Not case sensitive."],
    allow_actions_to_approve_pull_requests: Annotated[bool | None, "Allow GitHub Actions to approve pull requests. Enabling this may pose a security risk."] = None,  # noqa: E501
    default_github_token_permissions: Annotated[str | None, "The default permissions granted to the GITHUB_TOKEN when running workflows. Options are 'read' or 'write'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-github-actions-default-workflow-permissions-organization'."]:  # noqa: E501
    """Configure default GitHub Actions permissions for an organization.

    This tool sets the default workflow permissions for the `GITHUB_TOKEN` in an organization and determines if GitHub Actions can submit approving pull request reviews. It's essential to authenticate using an access token with `admin:org` scope, or GitHub Apps must have the `administration` permission. Use it to manage organizational GitHub Actions settings."""  # noqa: E501
    request_data: Any = {"can_approve_pull_request_reviews": allow_actions_to_approve_pull_requests, "default_workflow_permissions": default_github_token_permissions}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_required_workflows(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. Note: The name is not case sensitive. This identifies which organization's workflows to list."],  # noqa: E501
    results_page_number: Annotated[int | None, "The specific page number of required workflow results to fetch."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-required-workflows'."]:
    """Retrieve all required workflows in a GitHub organization.

    Use this tool to get a list of all workflows that are required within a specific GitHub organization. Authentication with an access token having the `read:org` scope is necessary."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_identifier: Annotated[str, "The ID of the repository that contains the workflow file. Use this to specify which repository's workflow file should be used."],  # noqa: E501
    workflow_file_path: Annotated[str, "Path of the workflow file to set as required for the organization."],  # noqa: E501
    repository_ids_for_selected_scope: Annotated[list[int] | None, "List of repository IDs to enable the workflow when `scope` is 'selected'."] = None,  # noqa: E501
    workflow_scope: Annotated[str | None, "Specify whether to enable the required workflow for all repositories or only selected ones within the organization. Use 'all' for all repositories and 'selected' when specifying particular repositories."] = "all",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-required-workflow'."]:  # noqa: E501
    """Create a required workflow in a GitHub organization.

    This tool creates a required workflow within a specified GitHub organization. It requires an access token with 'admin:org' scope for authentication. Use this tool when setting up or enforcing organizational workflow standards on GitHub."""  # noqa: E501
    request_data: Any = {"repository_id": repository_identifier, "scope": workflow_scope, "selected_repository_ids": repository_ids_for_selected_scope, "workflow_file_path": workflow_file_path}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This is case insensitive."],
    workflow_identifier: Annotated[int, "The unique identifier of the required workflow to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-required-workflow'."]:  # noqa: E501
    """Deletes a required workflow in a GitHub organization.

    Use this tool to delete a specific required workflow configured in a GitHub organization. Requires authentication with an `admin:org` scope access token."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=workflow_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This name is not case sensitive."],  # noqa: E501
    workflow_id: Annotated[int, "The unique identifier of the workflow to retrieve for the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-required-workflow'."]:
    """Retrieve a required workflow for a GitHub organization.

    This tool retrieves information about a required workflow configured in a GitHub organization. It requires authentication with a token having the `read:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=workflow_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    required_workflow_identifier: Annotated[int, "The unique identifier for the required workflow to update in the organization."],  # noqa: E501
    repository_id_for_workflow: Annotated[str | None, "The ID of the repository containing the workflow file to be updated."] = None,  # noqa: E501
    repository_ids_for_workflow: Annotated[list[int] | None, "List of repository IDs to enable the required workflow. Applicable only if `scope` is `selected`."] = None,  # noqa: E501
    workflow_file_path: Annotated[str | None, "Path to the workflow file to be set as a required workflow in the organization."] = None,  # noqa: E501
    workflow_scope: Annotated[str | None, "Specify the repository scope for enabling the workflow: 'all' for all repositories or 'selected' for specific ones within the organization."] = "all",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/update-required-workflow'."]:  # noqa: E501
    """Update a required workflow in a GitHub organization.

    This tool updates a specified required workflow within a GitHub organization. It requires an access token with `admin:org` scope for authentication."""  # noqa: E501
    request_data: Any = {"repository_id": repository_id_for_workflow, "scope": workflow_scope, "selected_repository_ids": repository_ids_for_workflow, "workflow_file_path": workflow_file_path}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=required_workflow_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_required_workflow_repositories(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive and identifies the organization within GitHub."],  # noqa: E501
    workflow_unique_identifier: Annotated[int, "The unique identifier of the required workflow for which the repositories are to be listed."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-selected-repositories-required-workflow'."]:  # noqa: E501
    """List repositories configured for a required workflow.

    This tool retrieves a list of repositories that are configured to use a specific required workflow within an organization. It should be called when you need to know which repositories are set up to run a required workflow. Authentication with an access token having `read:org` scope or a GitHub App with `administration` permission is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=workflow_unique_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_repos_for_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. Input is not case sensitive."],
    repository_ids_for_required_workflow: Annotated[list[int], "An array of repository IDs for which the workflow is required. Provide each repository's ID as an integer."],  # noqa: E501
    required_workflow_identifier: Annotated[int, "The unique identifier for the required workflow you want to set for the repositories."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-selected-repos-to-required-workflow'."]:  # noqa: E501
    """Set repositories for a GitHub required workflow.

    This tool sets the repositories associated with a GitHub required workflow. Use it when you need to configure or update which repositories are subject to a required workflow. Authentication with an access token and the `admin:org` scope is required."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": repository_ids_for_required_workflow}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=required_workflow_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_repo_from_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    repository_identifier: Annotated[int, "The unique identifier of the repository to be removed from the required workflow."],  # noqa: E501
    required_workflow_identifier: Annotated[int, "The unique identifier of the required workflow to be removed."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-selected-repo-from-required-workflow'."]:  # noqa: E501
    """Removes a repository from a GitHub required workflow.

    Use this tool to remove a repository from a GitHub required workflow. Ensure the workflow is set for selected repositories and authenticate with an 'admin:org' scoped token."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=required_workflow_identifier, repository_id=repository_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_repo_to_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name for GitHub. It is not case sensitive."],  # noqa: E501
    repository_unique_identifier: Annotated[int, "The unique identifier for the GitHub repository to be added to the required workflow."],  # noqa: E501
    required_workflow_identifier: Annotated[int, "The unique integer ID of the required workflow to which the repository will be added."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-selected-repo-to-required-workflow'."]:  # noqa: E501
    """Adds a repository to a GitHub required workflow.

    Use this tool to add a repository to a GitHub required workflow. Ensure the workflow is set to run on selected repositories. Authentication with an access token having `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/required_workflows/{required_workflow_id}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, required_workflow_id=required_workflow_identifier, repository_id=repository_unique_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_runner_groups_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. This is not case sensitive."],  # noqa: E501
    repository_visibility_filter: Annotated[str | None, "Specify the repository to filter runner groups that they are allowed to be used by."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of the results to fetch. Use an integer value."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-self-hosted-runner-groups-for-org'."]:  # noqa: E501
    """Retrieve self-hosted runner groups for a GitHub organization.

    This tool lists all self-hosted runner groups configured in a specified GitHub organization, including those inherited from an enterprise. Authentication with an access token having the `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "visible_to_repository": repository_visibility_filter}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_runner_group_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    runner_group_name: Annotated[str, "Name of the self-hosted runner group to be created. This should be a descriptive and distinct name within your organization."],  # noqa: E501
    accessible_repository_ids: Annotated[list[int] | None, "List of repository IDs that can access the runner group."] = None,  # noqa: E501
    allow_public_repositories: Annotated[bool | None, "Set to true to allow the runner group to be used by public repositories."] = False,  # noqa: E501
    allowed_workflows: Annotated[list[str] | None, "List of workflows names the runner group is permitted to run. Considered only if 'restricted_to_workflows' is true."] = None,  # noqa: E501
    restrict_to_selected_workflows: Annotated[bool | None, "Set to true to restrict the runner group to run only the workflows in the selected_workflows array."] = False,  # noqa: E501
    runner_group_visibility: Annotated[str | None, "Specify the visibility of the runner group: 'selected' for individual repositories, 'all' for all repositories, or 'private' for private repositories only."] = "all",  # noqa: E501
    runner_ids_to_add: Annotated[list[int] | None, "List of runner IDs to include in the newly created runner group for the organization."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-self-hosted-runner-group-for-org'."]:  # noqa: E501
    """Create a self-hosted runner group for an organization.

    This tool creates a new self-hosted runner group for an organization on GitHub. Requires authentication with an access token having the `admin:org` scope."""  # noqa: E501
    request_data: Any = {"allows_public_repositories": allow_public_repositories, "name": runner_group_name, "restricted_to_workflows": restrict_to_selected_workflows, "runners": runner_ids_to_add, "selected_repository_ids": accessible_repository_ids, "selected_workflows": allowed_workflows, "visibility": runner_group_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_runner_group_from_organization(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization on GitHub. This is not case-sensitive."],  # noqa: E501
    runner_group_id: Annotated[int, "The unique integer ID of the self-hosted runner group to delete."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-self-hosted-runner-group-from-org'."]:  # noqa: E501
    """Delete a self-hosted runner group from an organization.

    Use this tool to delete a self-hosted runner group from a specified organization on GitHub. Requires authentication with an access token with `admin:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_self_hosted_runner_group_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name on GitHub. It is not case sensitive."],
    runner_group_identifier: Annotated[int, "Unique identifier for the self-hosted runner group. It should be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-self-hosted-runner-group-for-org'."]:  # noqa: E501
    """Retrieve a specific self-hosted runner group for an organization.

    Use this tool to obtain details about a particular self-hosted runner group within an organization on GitHub. Requires authentication with an access token having 'admin:org' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_runner_group_settings(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. The name is not case sensitive."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group being updated."],  # noqa: E501
    runner_group_name: Annotated[str, "The new name for the self-hosted runner group in the organization."],  # noqa: E501
    allow_public_repository_usage: Annotated[bool | None, "Set to `true` to allow the runner group to be used by public repositories."] = False,  # noqa: E501
    allowed_workflow_list: Annotated[list[str] | None, "List of workflows the runner group can run. Ignored unless workflows are restricted."] = None,  # noqa: E501
    restrict_runner_group_to_workflows: Annotated[bool | None, "Set to `true` to restrict the runner group to running only the workflows specified in `selected_workflows`."] = False,  # noqa: E501
    runner_group_visibility: Annotated[str | None, "Specifies the visibility of the runner group. Options: 'selected' for individual repositories, 'all' for all repositories, 'private' for all private repositories."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/update-self-hosted-runner-group-for-org'."]:  # noqa: E501
    """Update name and visibility of a runner group in an organization.

    Call this tool to change the name or visibility of a self-hosted runner group within a GitHub organization. Requires authentication with an access token that has the `admin:org` scope."""  # noqa: E501
    request_data: Any = {"allows_public_repositories": allow_public_repository_usage, "name": runner_group_name, "restricted_to_workflows": restrict_runner_group_to_workflows, "selected_workflows": allowed_workflow_list, "visibility": runner_group_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repos_with_runner_group_access(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the organization."],
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group to fetch repository access details."],  # noqa: E501
    result_page_number: Annotated[int | None, "Page number of the results to fetch."] = 1,
    results_per_page: Annotated[int | None, "The number of results to retrieve per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-repo-access-to-self-hosted-runner-group-in-org'."]:  # noqa: E501
    """Retrieve repositories with access to a runner group in an organization.

    This tool lists the repositories that have access to a specified self-hosted runner group in an organization. It requires authentication with an access token having `admin:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"page": result_page_number, "per_page": results_per_page}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_runner_group_repo_access(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_ids_for_access: Annotated[list[int], "List of repository IDs allowed to access the self-hosted runner group."],  # noqa: E501
    runner_group_id: Annotated[int, "Unique identifier for the self-hosted runner group in the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-repo-access-to-self-hosted-runner-group-in-org'."]:  # noqa: E501
    """Update repository access for a GitHub runner group.

    Replaces the list of repositories that can access a self-hosted runner group in a GitHub organization. Requires authentication with an access token having the `admin:org` scope."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": repository_ids_for_access}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_repo_access_from_runner_group(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This input is not case sensitive."],  # noqa: E501
    repository_unique_id: Annotated[int, "The unique identifier of the repository to remove access from the runner group. Must be an integer."],  # noqa: E501
    runner_group_unique_id: Annotated[int, "Unique identifier of the self-hosted runner group. This is required to specify which runner group's access will be modified."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-repo-access-to-self-hosted-runner-group-in-org'."]:  # noqa: E501
    """Remove repository access from a self-hosted runner group.

    Use this tool to remove a repository's access from a specified self-hosted runner group within an organization. The runner group must have its visibility set to 'selected'. Authentication with an 'admin:org' scope access token is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_unique_id, repository_id=repository_unique_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_repo_access_to_runner_group(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    repository_unique_id: Annotated[int, "The unique identifier of the repository to add access for."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Provide the unique integer identifier for the self-hosted runner group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-repo-access-to-self-hosted-runner-group-in-org'."]:  # noqa: E501
    """Add repository access to a self-hosted runner group.

    This tool is used to add a repository to the list of selected repositories that can access a self-hosted runner group within an organization. It should be called when you need to manage repository access for a self-hosted runner group with `selected` visibility. Requires authentication with an `admin:org` scoped access token."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_identifier, repository_id=repository_unique_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_runner_group_runners(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    runner_group_id: Annotated[int, "Unique identifier of the self-hosted runner group to fetch runners for."],  # noqa: E501
    result_page_number: Annotated[int | None, "Specifies the page number of results to be fetched. Used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-self-hosted-runners-in-group-for-org'."]:  # noqa: E501
    """List self-hosted runners in an organization group.

    Retrieves self-hosted runners in a specified group within an organization. Requires an access token with 'admin:org' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_self_hosted_runners_for_org_group(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    runner_group_id: Annotated[int, "Unique identifier for the self-hosted runner group to update."],  # noqa: E501
    runner_ids_to_add: Annotated[list[int], "Array of integer IDs representing the runners to be added to the organization runner group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-self-hosted-runners-in-group-for-org'."]:  # noqa: E501
    """Update self-hosted runners in an organization's runner group.

    This tool allows you to replace the list of self-hosted runners in a specified organization runner group. It requires an access token with the 'admin:org' scope. Use this when you need to manage or update the runners assigned to a runner group in your organization."""  # noqa: E501
    request_data: Any = {"runners": runner_ids_to_add}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_runner_from_group(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This value is not case sensitive."],  # noqa: E501
    runner_group_identifier: Annotated[int, "Unique identifier of the self-hosted runner group for removal action."],  # noqa: E501
    runner_id: Annotated[int, "Unique identifier of the self-hosted runner to remove from the group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-self-hosted-runner-from-group-for-org'."]:  # noqa: E501
    """Remove a self-hosted runner from an organization's group.

    This tool removes a self-hosted runner from a specific group within an organization on GitHub. The runner is moved back to the default group. Requires authentication with an access token having `admin:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_identifier, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_runner_to_group(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    runner_group_identifier: Annotated[int, "The unique integer identifier of the self-hosted runner group to which the runner will be added."],  # noqa: E501
    runner_identifier: Annotated[int, "Unique identifier of the self-hosted runner to be added to the group."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-self-hosted-runner-to-group-for-org'."]:  # noqa: E501
    """Add a self-hosted runner to an organization's runner group.

    Use this tool to add a self-hosted runner to a specific runner group within an organization on GitHub. Ensure you authenticate with an access token that has the `admin:org` scope to perform this operation."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runner-groups/{runner_group_id}/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_group_id=runner_group_identifier, runner_id=runner_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_self_hosted_runners(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. This is case insensitive."],
    results_page_number: Annotated[int | None, "Specify the page number of the results to fetch. For paginated data retrieval."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-self-hosted-runners-for-org'."]:  # noqa: E501
    """Retrieve self-hosted runners for a GitHub organization.

    This tool fetches all self-hosted runners configured for a specified GitHub organization. Authentication with an admin-level token is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_runner_binaries_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. The organization name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-runner-applications-for-org'."]:  # noqa: E501
    """Retrieve downloadable binaries for GitHub runner application.

    This tool lists the binaries for the GitHub runner application that can be downloaded and run for a specified organization. Authentication with an access token having the `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/downloads".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_org_runner_registration_token(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case-sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-registration-token-for-org'."]:  # noqa: E501
    """Generate a registration token for GitHub organization runners.

    Use this tool to obtain a registration token for configuring a self-hosted runner in a GitHub organization. The token is valid for one hour and requires authentication with an access token having the 'admin:org' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/registration-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_runner_removal_token(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-remove-token-for-org'."]:  # noqa: E501
    """Get a token to remove a self-hosted runner from an organization.

    Use this tool to obtain a token for removing a self-hosted runner from a GitHub organization. The token expires after one hour and requires an access token with the `admin:org` scope for authentication."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/remove-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_self_hosted_runner_from_org(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. This input is not case sensitive."],  # noqa: E501
    runner_id: Annotated[int, "Unique identifier of the self-hosted GitHub runner to be removed."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-self-hosted-runner-from-org'."]:  # noqa: E501
    """Forcefully remove a self-hosted runner from an organization.

    Use this tool to completely remove a self-hosted GitHub Actions runner from an organization when the associated machine no longer exists. Requires an access token with the `admin:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_self_hosted_runner(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name (not case sensitive)."],
    runner_id: Annotated[int, "Unique integer identifier of the self-hosted runner within the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-self-hosted-runner-for-org'."]:  # noqa: E501
    """Get details of a self-hosted runner for an organization.

    Fetches information about a specific self-hosted runner configured in a GitHub organization. Authentication with an access token having the `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_all_custom_labels_runner_org(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name, not case sensitive."],
    runner_id: Annotated[int, "The unique identifier of the self-hosted runner. This is required to target the specific runner for label removal in an organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-all-custom-labels-from-self-hosted-runner-for-org'."]:  # noqa: E501
    """Remove all custom labels from an organization's self-hosted runner.

    Use this tool to remove all custom labels from a specific self-hosted runner in a GitHub organization. Requires authentication with an access token having `admin:org` scope. Returns the runner's remaining read-only labels."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_runner_labels_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This parameter is not case sensitive."],  # noqa: E501
    runner_unique_id: Annotated[int, "The unique identifier for the self-hosted runner. It must be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-labels-for-self-hosted-runner-for-org'."]:  # noqa: E501
    """Retrieve labels for a self-hosted runner in an organization.

    This tool is used to list all labels for a specified self-hosted runner within an organization. Authentication with an access token having the `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_unique_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_labels_to_runner(
    context: ToolContext,
    custom_labels_to_add: Annotated[list[str], "An array of strings representing the custom labels to add to the self-hosted runner. Each label is a string."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    runner_id: Annotated[int, "Unique identifier of the self-hosted runner to which labels are added."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-custom-labels-to-self-hosted-runner-for-org'."]:  # noqa: E501
    """Add custom labels to a self-hosted runner in an organization.

    This tool adds custom labels to a self-hosted GitHub Actions runner configured in an organization. Use this tool when you need to organize or categorize runners with specific labels. Requires authentication with an access token having the `admin:org` scope."""  # noqa: E501
    request_data: Any = {"labels": custom_labels_to_add}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_custom_labels_runner_org(
    context: ToolContext,
    custom_labels: Annotated[list[str], "An array of custom label names to assign to the runner. Pass an empty array to remove all labels."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    runner_identifier: Annotated[int, "Unique identifier of the self-hosted runner to set custom labels."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-custom-labels-for-self-hosted-runner-for-org'."]:  # noqa: E501
    """Set custom labels for a self-hosted runner in an organization.

    Updates custom labels for a specified self-hosted runner in an organization, removing any previous labels. Requires authentication with an access token having `admin:org` scope."""  # noqa: E501
    request_data: Any = {"labels": custom_labels}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_identifier),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_custom_runner_label(
    context: ToolContext,
    custom_label_name: Annotated[str, "The name of the custom label to remove from the self-hosted runner."],  # noqa: E501
    organization_name: Annotated[str, "The case-insensitive name of the organization."],
    runner_identifier: Annotated[int, "Unique identifier of the self-hosted runner. Must be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-custom-label-from-self-hosted-runner-for-org'."]:  # noqa: E501
    """Remove a custom label from a self-hosted runner in an organization.

    Removes a specified custom label from a self-hosted runner within an organization and returns the updated list of labels on the runner. Requires authentication with an access token having the `admin:org` scope. If the label is not found, a `404 Not Found` status is returned."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/runners/{runner_id}/labels/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, runner_id=runner_identifier, name=custom_label_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_secrets(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name to list secrets for. This name is not case-sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "Page number of the results to fetch for pagination purposes."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of secrets to list per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-org-secrets'."]:
    """Retrieve all organization secrets without values.

    This tool fetches a list of all secrets available in a GitHub organization without revealing their encrypted values. It requires authentication with an access token possessing the `admin:org` scope. Ideal for managing and reviewing organization secrets."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_organization_public_key(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This is not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-org-public-key'."]:
    """Retrieve the public key for GitHub organization secrets encryption.

    Use this tool to obtain the public key necessary for encrypting secrets before creation or update within a GitHub organization. This requires authentication with an access token that has the `admin:org` scope. Suitable for use when managing organization secrets securely."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/public-key".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name, not case sensitive."],
    secret_name: Annotated[str, "The name of the secret to be deleted from the GitHub organization. Ensure the name is correct and case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-org-secret'."]:
    """Deletes a secret from a GitHub organization.

    Use this tool to delete a specific secret from a GitHub organization by its name. Requires authentication with an access token with `admin:org` scope or a GitHub App with `secrets` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This name is not case sensitive."],  # noqa: E501
    secret_name: Annotated[str, "The name of the secret within the GitHub organization. It identifies which secret's details to retrieve."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-org-secret'."]:
    """Retrieve details of a GitHub organization secret.

    Use this tool to get the details of a GitHub organization secret without exposing its encrypted value. Authentication with an access token with `admin:org` scope or GitHub App with `secrets` permission is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_or_update_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_visibility: Annotated[str, "Specifies which type of organization repositories have access to the organization secret. Choices are 'all', 'private', or 'selected'."],  # noqa: E501
    secret_name: Annotated[str, "The name of the secret. It's used to identify the secret within the organization."],  # noqa: E501
    encrypted_secret_value: Annotated[str | None, "The secret's encrypted value using LibSodium and a GitHub org public key."] = None,  # noqa: E501
    encryption_key_id: Annotated[str | None, "The ID of the public key used to encrypt the secret. This must match the key used during encryption."] = None,  # noqa: E501
    repository_ids_for_secret_access: Annotated[list[int] | None, "Array of repository ids allowed access to the secret. Provide only when `visibility` is `selected`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-or-update-org-secret'."]:  # noqa: E501
    """Create or update an organization's secret on GitHub.

    Use this tool to create or update a secret within a GitHub organization using an encrypted value. The secret must be encrypted with LibSodium, and authentication requires a token with 'admin:org' scope. GitHub Apps need 'secrets' organization permission."""  # noqa: E501
    request_data: Any = {"encrypted_value": encrypted_secret_value, "key_id": encryption_key_id, "selected_repository_ids": repository_ids_for_secret_access, "visibility": repository_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repos_with_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    secret_name: Annotated[str, "The name of the secret for which to list selected repositories. Case insensitive."],  # noqa: E501
    result_page_number: Annotated[int | None, "Page number of the results to fetch for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of repositories to return per page, with a maximum of 100 allowed."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-selected-repos-for-org-secret'."]:  # noqa: E501
    """Retrieve repositories with access to a specific organization secret.

    This tool lists all repositories that have been given access to a specific organization secret with `selected` visibility. It requires authentication with an access token featuring the `admin:org` scope, or a GitHub App with `secrets` organization permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"page": result_page_number, "per_page": results_per_page}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_org_secret_repos(
    context: ToolContext,
    organization_name: Annotated[str, "The organization's name for the secret. It is not case sensitive."],  # noqa: E501
    repository_ids_for_access: Annotated[list[int], "An array of repository IDs allowed to access the organization secret when visibility is set to 'selected'."],  # noqa: E501
    secret_name: Annotated[str, "The name of the organization secret to update repositories for. This value is case-insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-selected-repos-for-org-secret'."]:  # noqa: E501
    """Update repositories for an organization secret.

    Use this tool to replace all repositories associated with an organization secret when the repository access visibility is set to 'selected'. Authentication with an access token having the 'admin:org' scope or a GitHub App with the 'secrets' organization permission is required."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": repository_ids_for_access}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_repo_from_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name. It is not case sensitive."],
    repository_id: Annotated[int, "The unique ID of the repository to be removed from the organization's secret."],  # noqa: E501
    secret_name: Annotated[str, "The name of the organization secret to remove the repository from."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-selected-repo-from-org-secret'."]:  # noqa: E501
    """Remove a repository from an organization's secret access.

    Use this tool to remove a specified repository from accessing an organization's secret when the visibility is set to 'selected'. Authentication with an access token with 'admin:org' scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name, repository_id=repository_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_repo_to_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The organization's name. It is not case sensitive."],
    repository_id: Annotated[int, "The unique integer ID of the repository to be added to the organization secret. This ID specifies which repository you want to include."],  # noqa: E501
    secret_name: Annotated[str, "The name of the organization secret to which the repository will be added."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-selected-repo-to-org-secret'."]:  # noqa: E501
    """Add a repository to an organization's secret.

    This tool adds a specified repository to an organization secret with 'selected' visibility. It requires authentication with an access token having 'admin:org' scope. GitHub Apps need 'secrets' organization permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/secrets/{secret_name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name, repository_id=repository_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_variables(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    result_page_number: Annotated[int | None, "Specify the page number of the organization variables to retrieve."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, up to a maximum of 30."] = 10,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-org-variables'."]:
    """Retrieve all variables for a GitHub organization.

    Use this tool to list all variables configured in a GitHub organization. Authentication with an access token having the `admin:org` scope or a GitHub App with `organization_actions_variables:read` permission is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_org_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    repository_access_visibility: Annotated[str, "Type of repositories in the organization that can access the variable: 'all', 'private', or 'selected'."],  # noqa: E501
    variable_name: Annotated[str, "The name of the organization variable to be created. This name will be used to reference the variable in workflows."],  # noqa: E501
    variable_value: Annotated[str, "The value assigned to the organization variable in GitHub."],
    repository_ids_with_variable_access: Annotated[list[int] | None, "List of repository IDs allowed to access the organization variable. Required when 'visibility' is 'selected'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-org-variable'."]:
    """Create an organization variable for GitHub Actions workflows.

    Creates an organization variable in GitHub that can be referenced in Actions workflows. Requires an access token with `admin:org` scope or `organization_actions_variables:write` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = {"name": variable_name, "selected_repository_ids": repository_ids_with_variable_access, "value": variable_value, "visibility": repository_access_visibility}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_org_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
    variable_name: Annotated[str, "The name of the organization variable to delete. It should be a string matching the variable's identifier."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-org-variable'."]:
    """Delete an organization's variable on GitHub.

    Deletes a specified variable from an organization on GitHub. Requires authentication with an access token with the `admin:org` scope, or GitHub App with `organization_actions_variables:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_organization_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This value is not case sensitive."],  # noqa: E501
    variable_name: Annotated[str, "The exact name of the specific variable to retrieve from the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-org-variable'."]:
    """Retrieve a specific variable from a GitHub organization.

    Use this tool to access a particular variable within a GitHub organization. Requires authentication with an access token with the 'admin:org' scope or a GitHub App with 'organization_actions_variables:read' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_org_action_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. This name is not case sensitive."],  # noqa: E501
    repository_ids_for_selected_visibility: Annotated[list[int] | None, "An array of repository IDs that can access the organization variable. Only provide when `visibility` is set to `selected`."] = None,  # noqa: E501
    repository_visibility_type: Annotated[str | None, "Specifies which repositories in the organization can access the variable. Options: `all`, `private`, `selected`."] = None,  # noqa: E501
    var_name: Annotated[str | None, "Specify the name of the GitHub organization variable to update."] = None,  # noqa: E501
    variable_name: Annotated[str | None, "The name of the organization variable to update. This is case-insensitive."] = None,  # noqa: E501
    variable_value: Annotated[str | None, "The new value to assign to the organization variable. It must be a string."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/update-org-variable'."]:
    """Update an organization variable in GitHub Actions.

    This tool updates a specified variable for an organization in GitHub Actions workflows. Authentication with an access token having `admin:org` scope or GitHub App with `organization_actions_variables:write` permission is required."""  # noqa: E501
    request_data: Any = {"name": var_name, "selected_repository_ids": repository_ids_for_selected_visibility, "value": variable_value, "visibility": repository_visibility_type}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=var_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repos_with_org_variable_access(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    variable_name: Annotated[str, "The name of the organization variable to check for repository access."],  # noqa: E501
    result_page_number: Annotated[int | None, "The page number of repository results to retrieve. Use this to navigate through results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-selected-repos-for-org-variable'."]:  # noqa: E501
    """Retrieve repos accessing an organization's variable.

    This tool lists all repositories that have access to a specified organization variable. Authentication with an access token having the `admin:org` scope is required. GitHub Apps must have the `organization_actions_variables:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"page": result_page_number, "per_page": results_per_page}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_org_variable_repos(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    repository_ids_for_access: Annotated[list[int], "An array of integers representing the repository IDs that can access the organization variable. Ensure the IDs are valid and accessible."],  # noqa: E501
    variable_name: Annotated[str, "The name of the organization variable to be updated."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-selected-repos-for-org-variable'."]:  # noqa: E501
    """Replace repositories for an organization's variable.

    This tool replaces all repositories associated with an organization variable that is set to selected visibility. Use it to update the repositories list where an org variable is available. Requires appropriate authentication and permissions."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": repository_ids_for_access}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_repo_from_org_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
    repository_id: Annotated[int, "The ID of the repository to be removed from the organization variable. This should be an integer value identifying the repository."],  # noqa: E501
    variable_name: Annotated[str, "The name of the organization variable to remove the repository from."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-selected-repo-from-org-variable'."]:  # noqa: E501
    """Remove a repository from a GitHub organization variable.

    This tool removes a specified repository from an organization's variable that is available to selected repositories. Useful when managing organization-level variable access. Requires authentication with an admin token or necessary permissions for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name, repository_id=repository_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_repo_to_org_variable(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    repository_id: Annotated[int, "The integer ID of the repository to be added to the organization variable. This is a required field."],  # noqa: E501
    variable_name: Annotated[str, "The name of the organization variable to which the repository will be added."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-selected-repo-to-org-variable'."]:  # noqa: E501
    """Add a repository to an organization's selected variables.

    Use this tool to add a repository to a specific organization variable that is available to selected repositories on GitHub. This requires an access token with `admin:org` scope or GitHub App permission `organization_actions_variables:write`. The repository and organization must have the `visibility` field set to `selected`."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/actions/variables/{name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, name=variable_name, repository_id=repository_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_org_announcement_banner(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'announcement-banners/remove-announcement-banner-for-org'."]:  # noqa: E501
    """Remove the announcement banner for a GitHub organization.

    Use this tool to delete the announcement banner that is currently set for a specific GitHub organization. Ideal for managing organizational announcements."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_announcement_banner(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'announcement-banners/get-announcement-banner-for-org'."]:  # noqa: E501
    """Retrieve the announcement banner for a specific organization.

    Fetches the current announcement banner set at the organization level. Does not include enterprise-level banners."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_org_announcement_banner(
    context: ToolContext,
    announcement_text: Annotated[str, "The announcement text formatted in GitHub Flavored Markdown. See GitHub's basic writing syntax for details."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. The name is not case sensitive."],  # noqa: E501
    announcement_expiry_time: Annotated[str | None, "The expiry time for the announcement, in ISO 8601 format. Use `null` or empty for no expiry."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'announcement-banners/set-announcement-banner-for-org'."]:  # noqa: E501
    """Sets the announcement banner for a GitHub organization.

    Use this tool to update the announcement banner displayed for a specified GitHub organization."""  # noqa: E501
    request_data: Any = {"announcement": announcement_text, "expires_at": announcement_expiry_time}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/announcement".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_audit_log(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    event_order: Annotated[str | None, "Specify the order of audit log events. Use 'desc' for newest first or 'asc' for oldest first. Default is 'desc'."] = None,  # noqa: E501
    event_types_to_include: Annotated[str | None, "Specify the event types to include: 'web' for web events, 'git' for Git events, or 'all' for both. Default is 'web'."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number of results to fetch. Useful for navigating through paginated audit log entries."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Defines the number of audit log events returned per page, with a maximum of 100."] = 30,  # noqa: E501
    search_events_after_cursor: Annotated[str | None, "Cursor for searching events after a specific point, as given in the Link header."] = None,  # noqa: E501
    search_events_before_cursor: Annotated[str | None, "A cursor to search for events before this point. Use to limit results to events occurring before a specific reference."] = None,  # noqa: E501
    search_phrase: Annotated[str | None, "A string to filter audit log events based on specific criteria. This can help in retrieving older events. Refer to the GitHub documentation for more details on searching the audit log."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get-audit-log'."]:
    """Retrieve the audit log for a GitHub organization.

    This tool retrieves the audit log for a specified GitHub organization. It is used to monitor and review up to 30 events from the past three months by default. Organization owners or GitHub Apps with appropriate permissions can filter, paginate, and analyze audit events using this tool. Useful for tracking organization activities and security events."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/audit-log".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"phrase": search_phrase, "include": event_types_to_include, "after": search_events_after_cursor, "before": search_events_before_cursor, "order": event_order, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_code_scanning_alerts(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This field is not case sensitive."],  # noqa: E501
    alert_severity: Annotated[str | None, "Specifies the severity of code scanning alerts to be returned. Acceptable values include 'critical', 'high', 'medium', 'low', 'warning', 'note', and 'error'."] = None,  # noqa: E501
    alert_state: Annotated[str | None, "Filter code scanning alerts by their state: 'open', 'closed', 'dismissed', or 'fixed'."] = None,  # noqa: E501
    code_scanning_tool_guid: Annotated[str | None, "The GUID of a specific code scanning tool. Use this to filter alerts by the tool. Must not use with `tool_name`."] = None,  # noqa: E501
    cursor_before: Annotated[str | None, "A cursor used to fetch results occurring before this point in the data timeline."] = None,  # noqa: E501
    filter_by_tool_name: Annotated[str | None, "Specify the name of a code scanning tool to filter results. Only alerts from this tool will be listed. Ensure not to use `tool_guid` if this is specified."] = None,  # noqa: E501
    results_after_cursor: Annotated[str | None, "A cursor indicating the point after which to retrieve results. Used for pagination."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch from the list of code scanning alerts."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page, with a maximum limit of 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "Specifies the sort order of the results. Use 'asc' for ascending or 'desc' for descending."] = "desc",  # noqa: E501
    sort_results_by: Annotated[str | None, "Specifies the property by which to sort the results. Options are 'created' or 'updated'."] = "created",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'code-scanning/list-alerts-for-org'."]:  # noqa: E501
    """Retrieve code scanning alerts for an organization's repositories.

    This tool lists code scanning alerts for the default branch of all eligible repositories within a specified organization. It requires the user to be an owner or security manager of the organization, and access tokens with the necessary scopes (`repo`, `security_events`, or `public_repo` for public repositories)."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/code-scanning/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"tool_name": filter_by_tool_name, "tool_guid": code_scanning_tool_guid, "before": cursor_before, "after": results_after_cursor, "page": results_page_number, "per_page": results_per_page, "direction": sort_direction, "state": alert_state, "sort": sort_results_by, "severity": alert_severity}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_dependabot_alerts_for_organization(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name for which to list Dependabot alerts. This name is not case sensitive."],  # noqa: E501
    alert_severity_filter: Annotated[str | None, "Comma-separated severities to filter alerts. Options: `low`, `medium`, `high`, `critical`."] = None,  # noqa: E501
    alert_states: Annotated[str | None, "Comma-separated list of alert states to filter by. Options: `dismissed`, `fixed`, `open`."] = None,  # noqa: E501
    deprecated_first_results_page_size: Annotated[int | None, "**Deprecated**. Use `per_page` with `after` instead. Specifies the number of results per page, up to 100, starting from the first matching result. Avoid using with `last`."] = 30,  # noqa: E501
    deprecated_last_results_per_page: Annotated[int | None, "**Deprecated**. Use this to specify the number of results per page (max 100) from the last matching result. Cannot be used with 'first'."] = None,  # noqa: E501
    ecosystem_list: Annotated[str | None, "Comma-separated ecosystems to filter alerts. Options: composer, go, maven, npm, nuget, pip, pub, rubygems, rust."] = None,  # noqa: E501
    package_names: Annotated[str | None, "A comma-separated list of package names. Only alerts for these specified packages will be returned."] = None,  # noqa: E501
    paginate_after_cursor: Annotated[str | None, "A cursor indicating the point to continue the listing from, based on the Link header."] = None,  # noqa: E501
    results_before_cursor: Annotated[str | None, "A cursor indicating to search for Dependabot alerts before this position."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of Dependabot alerts to return per page, with a maximum of 100."] = 30,  # noqa: E501
    sort_by_property: Annotated[str | None, "Sort results by `created` (alert creation) or `updated` (state change)."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "The order to sort the results, either ascending ('asc') or descending ('desc')."] = "desc",  # noqa: E501
    vulnerable_dependency_scope: Annotated[str | None, "Specify the scope of the vulnerable dependency as either 'development' or 'runtime'. Only alerts with this scope will be returned."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/list-alerts-for-org'."]:
    """Lists Dependabot alerts for an organization.

    This tool lists Dependabot alerts for a specified organization. It requires you to be an organization owner or security manager with an appropriate access token. GitHub Apps need **Dependabot alerts** read permission. Use for monitoring security alerts related to dependencies."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"state": alert_states, "severity": alert_severity_filter, "ecosystem": ecosystem_list, "package": package_names, "scope": vulnerable_dependency_scope, "sort": sort_by_property, "direction": sort_direction, "before": results_before_cursor, "after": paginate_after_cursor, "first": deprecated_first_results_page_size, "last": deprecated_last_results_per_page, "per_page": results_per_page}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_org_secrets(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. It is not case sensitive."],
    results_page_number: Annotated[int | None, "Page number to fetch results from in the paginated list of organization secrets."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of secrets to list per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/list-org-secrets'."]:
    """Retrieve Dependabot organization secrets from GitHub.

    Use this tool to list all Dependabot secrets of a GitHub organization without revealing their encrypted values. Requires authentication with an `admin:org` scope token or a GitHub App with `dependabot_secrets` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_public_key(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name, not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/get-org-public-key'."]:
    """Retrieve the public key for encrypting GitHub Dependabot secrets.

    Use this tool to get your GitHub organization's public key, which is necessary for encrypting secrets before creating or updating them in Dependabot. Ensure authentication with an access token with the `admin:org` scope or GitHub App permissions for `dependabot_secrets`."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/public-key".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_github_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
    secret_name_to_delete: Annotated[str, "The name of the secret to delete from the GitHub organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/delete-org-secret'."]:
    """Delete a secret from a GitHub organization.

    Use this tool to delete a Dependabot secret from a specific GitHub organization by providing the secret name. Authentication with an access token with the `admin:org` scope is required. This tool is suitable when you need to manage organization secrets programmatically."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name_to_delete),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_secret_info(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    organization_secret_name: Annotated[str, "The name of the secret for the organization. It is required to identify the specific secret without revealing its encrypted value. This name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/get-org-secret'."]:
    """Retrieve details of an organization's secret without revealing the encrypted value.

    This tool fetches a single organization secret's details from GitHub without disclosing the encrypted value. Use it when you need information about a secret within an organization. Authentication with an access token with `admin:org` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=organization_secret_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization, not case sensitive."],
    repository_access_type: Annotated[str, "Determines which organization repositories can access the secret: 'all', 'private', or 'selected' (which requires specifying `selected_repository_ids`)."],  # noqa: E501
    secret_name: Annotated[str, "The name of the secret to create or update. This should be a unique identifier for the secret within the organization."],  # noqa: E501
    encrypted_secret_value: Annotated[str | None, "The secret value encrypted using LibSodium with the organization public key."] = None,  # noqa: E501
    encryption_key_id: Annotated[str | None, "The ID of the key used to encrypt the organization secret, required for security verification."] = None,  # noqa: E501
    selected_repository_ids_to_include: Annotated[list[str] | None, "Array of repository IDs allowed to access the secret. Used when visibility is set to 'selected'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/create-or-update-org-secret'."]:  # noqa: E501
    """Create or update a GitHub organization secret.

    Use this tool to create or update an organization's secret on GitHub, encrypted with LibSodium. Requires an access token with `admin:org` scope or a GitHub App with `dependabot_secrets` organization permission."""  # noqa: E501
    request_data: Any = {"encrypted_value": encrypted_secret_value, "key_id": encryption_key_id, "selected_repository_ids": selected_repository_ids_to_include, "visibility": repository_access_type}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_selected_repositories_for_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive. Required to identify the organization whose secret's repository access is being queried."],  # noqa: E501
    secret_name: Annotated[str, "The name of the secret for which you wish to list selected repositories. It is case insensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/list-selected-repos-for-org-secret'."]:  # noqa: E501
    """Retrieve repositories with selected access for an org secret.

    This tool lists all repositories that have been granted access to a secret in a GitHub organization, where the access visibility is set to 'selected'. Use it when you need to determine which repositories can access a particular secret. Requires an access token with 'admin:org' scope or 'dependabot_secrets' permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"page": results_page_number, "per_page": results_per_page}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_repos_for_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    secret_name: Annotated[str, "The name of the organization secret to update the repository access for. This should match the name of an existing Dependabot secret."],  # noqa: E501
    selected_repository_ids: Annotated[list[int], "List of repository IDs that can access the org secret. Use only when visibility is 'selected'."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/set-selected-repos-for-org-secret'."]:  # noqa: E501
    """Update selected repos for an organization's Dependabot secret.

    This tool updates the list of repositories associated with a Dependabot secret for an organization, when the visibility is set to 'selected'. It requires authentication with an access token with 'admin:org' scope or a GitHub App with 'dependabot_secrets' permission."""  # noqa: E501
    request_data: Any = {"selected_repository_ids": selected_repository_ids}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_repo_from_org_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_id: Annotated[int, "The unique identifier of the repository to be removed from the organization secret."],  # noqa: E501
    secret_name: Annotated[str, "The name of the GitHub organization secret to remove the repository from."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/remove-selected-repo-from-org-secret'."]:  # noqa: E501
    """Remove a repository from a GitHub organization secret.

    Use this tool to remove a specific repository from an organization secret in GitHub. This is applicable when the repository access visibility is set to 'selected'. Ensure authentication with an access token that has the 'admin:org' scope. GitHub Apps require 'dependabot_secrets' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name, repository_id=repository_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_repository_to_secret(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_id_for_org_secret: Annotated[int, "The ID of the repository to be added to the organization secret. This ID should be an integer."],  # noqa: E501
    secret_name: Annotated[str, "The name of the organization secret to which a repository will be added. This is case-insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'dependabot/add-selected-repo-to-org-secret'."]:  # noqa: E501
    """Add a repository to a GitHub organization secret.

    Use this tool to add a repository to an organization's secret in GitHub when the `visibility` is set to `selected`. Requires an access token with `admin:org` scope or a GitHub App with `dependabot_secrets` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/dependabot/secrets/{secret_name}/repositories/{repository_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, secret_name=secret_name, repository_id=repository_id_for_org_secret),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_public_org_events(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch. Determines which set of results to return."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'activity/list-public-org-events'."]:
    """List public events for a GitHub organization.

    This tool retrieves and lists all public events for a specified organization on GitHub. It is useful for monitoring recent activities within an organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/events".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_external_group_info(
    context: ToolContext,
    group_id: Annotated[int, "The unique identifier of the group. Must be an integer."],
    organization_name: Annotated[str, "The name of the GitHub organization. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/external-idp-group-info-for-org'."]:  # noqa: E501
    """Retrieve information about a GitHub external group's usage.

    This tool displays information about a specific external group's usage within a GitHub organization. It provides details on the group's external members and the teams connected to this group. Useful for managing team membership through identity providers using Enterprise Managed Users for GitHub Enterprise Cloud."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/external-group/{group_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, group_id=group_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_external_groups_for_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    group_name_filter: Annotated[str | None, "Filter results to include only groups with names containing this text."] = None,  # noqa: E501
    pagination_token: Annotated[int | None, "Token to specify the starting point for the next set of results."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum value of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-external-idp-groups-for-org'."]:  # noqa: E501
    """Retrieve external groups available in a GitHub organization.

    Fetches a list of external groups for a specified organization. Use the `display_name` parameter to filter groups by name, and `per_page` for pagination."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/external-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": pagination_token, "display_name": group_name_filter}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_webhooks(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This parameter is case insensitive."],  # noqa: E501
    result_page_number: Annotated[int | None, "The specific page number of results to retrieve from the list of organization webhooks."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of webhook results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-webhooks'."]:
    """Retrieve the webhooks for a specific organization on GitHub.

    Call this tool to get a list of configured webhooks for a specified GitHub organization. Useful for checking webhook integrations and statuses."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_org_webhook(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The name of the GitHub organization. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/create-webhook'."]:
    """Create a webhook for a GitHub organization.

    This tool is used to set up a webhook for a GitHub organization to post payloads in JSON format. Call this tool when you need to automate notifications or integrate with other services based on GitHub events.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBORGWEBHOOK"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBORGWEBHOOK"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBORGWEBHOOK"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}/hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBORGWEBHOOK"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_org_webhook(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is case insensitive."],  # noqa: E501
    webhook_identifier: Annotated[int, "The unique identifier of the webhook to be deleted. It should be provided as an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/delete-webhook'."]:
    """Delete a webhook from a GitHub organization.

    Use this tool to delete a specific webhook from a GitHub organization by specifying the organization and webhook ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_webhook_details(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization, not case sensitive."],
    webhook_id: Annotated[int, "The unique identifier for the organization's webhook. This is an integer value used to specify which webhook details to retrieve."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get-webhook'."]:
    """Retrieve details of a specific organization webhook.

    Use this tool to obtain detailed information about a webhook configured in a GitHub organization, including its properties and settings."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_org_webhook(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The name of the GitHub organization; not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    webhook_identifier: Annotated[int | None, "The unique identifier of the webhook to be updated.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/update-webhook'."]:
    """Update a webhook configured in a GitHub organization.

    Use this tool to update a webhook in a GitHub organization. When updating, ensure to provide the current or new secret if one was previously set to avoid its removal.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGWEBHOOK"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))
    if not webhook_identifier:
        missing_params.append(("webhook_identifier", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGWEBHOOK"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGWEBHOOK"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBORGWEBHOOK"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_webhook_configuration(
    context: ToolContext,
    organization_name: Annotated[str, "The non-case-sensitive name of the GitHub organization."],
    webhook_hook_id: Annotated[int, "The unique identifier of the organization's webhook, provided as an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get-webhook-config-for-org'."]:
    """Retrieve webhook configuration for a GitHub organization.

    Call this tool to get the webhook configuration for a specified organization on GitHub. Useful for checking the webhook's setup details. Requires appropriate access tokens or GitHub App permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/config".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_hook_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_org_webhook_config(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The name of the GitHub organization to update. Case insensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    webhook_identifier: Annotated[int | None, "The unique identifier of the webhook to be updated.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/update-webhook-config-for-org'."]:  # noqa: E501
    """Update webhook configuration for a GitHub organization.

    This tool updates the webhook configuration for a specific organization on GitHub. It should be called when you need to modify the webhook's settings, excluding \"active\" state and events. Requires `admin:org_hook` scope for tokens or `organization_hooks:write` permission for GitHub Apps.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEORGWEBHOOKCONFIG"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))
    if not webhook_identifier:
        missing_params.append(("webhook_identifier", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEORGWEBHOOKCONFIG"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEORGWEBHOOKCONFIG"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/config".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEORGWEBHOOKCONFIG"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_webhook_event_deliveries(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    webhook_hook_id: Annotated[int, "The unique identifier of the webhook in the organization."],
    include_redeliveries: Annotated[bool | None, "Indicate whether to include repeated webhook deliveries in the results. Set to true to include."] = None,  # noqa: E501
    pagination_start_cursor: Annotated[str | None, "A cursor to indicate the starting delivery for fetching the page of deliveries. Useful for pagination purposes."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of webhook deliveries to be returned per page (maximum 100)."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-webhook-deliveries'."]:
    """Retrieve webhook deliveries for an organization.

    Call this tool to get a list of webhook deliveries for a specific webhook configured in a GitHub organization. It is useful for monitoring and debugging webhook events."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/deliveries".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_hook_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "cursor": pagination_start_cursor, "redelivery": include_redeliveries}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_webhook_delivery(
    context: ToolContext,
    hook_identifier: Annotated[int, "The unique identifier of the webhook hook. This is an integer value used to specify which webhook's delivery details to retrieve for the organization."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    webhook_delivery_id: Annotated[int, "The unique identifier of the specific delivery to be retrieved. This should be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get-webhook-delivery'."]:
    """Retrieve a webhook delivery for a GitHub organization.

    This tool is used to obtain details about a specific delivery of a webhook configured in a GitHub organization. It provides information about the delivery for a given organization, hook, and delivery identifier."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/deliveries/{delivery_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=hook_identifier, delivery_id=webhook_delivery_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def redeliver_webhook_delivery(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    webhook_delivery_id: Annotated[int, "The unique identifier of the webhook delivery to be redelivered."],  # noqa: E501
    webhook_hook_id: Annotated[int, "The unique identifier of the webhook hook. Provide an integer value."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/redeliver-webhook-delivery'."]:
    """Redeliver an organization's webhook delivery attempt.

    Use this tool to redeliver a delivery for a webhook configured in an organization. It is useful when a webhook delivery initially failed or needs to be tested again."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/deliveries/{delivery_id}/attempts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_hook_id, delivery_id=webhook_delivery_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def send_github_hook_ping(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    webhook_hook_id: Annotated[int, "The unique integer ID of the GitHub organization webhook to ping."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/ping-webhook'."]:
    """Triggers a ping event on a GitHub organization webhook.

    Use this tool to send a ping event to a specified webhook in a GitHub organization. This can be used to verify the connection or test the webhook's configuration."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/hooks/{hook_id}/pings".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, hook_id=webhook_hook_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_org_installation_info(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. It is not case-sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'apps/get-org-installation'."]:
    """Retrieve GitHub organization's installation information.

    Use this tool to find installation details for an organization using an authenticated GitHub App. It requires a JWT for access."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/installation".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def organization_app_installations(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    page_number_to_fetch: Annotated[int | None, "Page number of the results to fetch. Used for pagination in retrieving GitHub App installations."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of GitHub App installations to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-app-installations'."]:
    """Retrieve GitHub App installations for an organization.

    Use this tool to list all GitHub Apps installed within a specified organization. Requires organization owner status with `admin:read` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/installations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number_to_fetch}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_issues_for_user(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    issue_filter: Annotated[str | None, "Specifies the type of issues to return: 'assigned', 'created', 'mentioned', 'subscribed', 'all', or 'repos'."] = "assigned",  # noqa: E501
    issue_labels: Annotated[str | None, "Comma-separated list of label names to filter issues. Example: `bug,ui,@high`."] = None,  # noqa: E501
    issues_state: Annotated[str | None, "Specify the state of issues to return: 'open', 'closed', or 'all'."] = "open",  # noqa: E501
    result_page_number: Annotated[int | None, "Page number of results to fetch, used for pagination. Starts from 1."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page (maximum 100)."] = 30,  # noqa: E501
    sort_criteria: Annotated[str | None, "Defines the attribute to sort the issues by. Options are 'created', 'updated', or 'comments'."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "The direction to sort the results by. Accepted values are 'asc' for ascending and 'desc' for descending."] = "desc",  # noqa: E501
    updated_since: Annotated[str | None, "Filter to show notifications updated after this timestamp in ISO 8601 format: `YYYY-MM-DDTHH:MM:SSZ`."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'issues/list-for-org'."]:
    """Retrieve issues and pull requests for a user in an organization.

    Call this tool to list all issues and pull requests assigned to the authenticated user within a specific organization on GitHub. This tool returns both issues and pull requests, identifiable by the `pull_request` key. It is useful for tracking user-specific tasks within an organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/issues".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"filter": issue_filter, "state": issues_state, "labels": issue_labels, "sort": sort_criteria, "direction": sort_direction, "since": updated_since, "per_page": results_per_page, "page": result_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_members(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization to list members from. It is not case sensitive."],  # noqa: E501
    filter_members: Annotated[str | None, "Filter the list of members. Use '2fa_disabled' to list members without two-factor authentication."] = "all",  # noqa: E501
    member_role_filter: Annotated[str | None, "Filter members by their role in the organization ('all', 'admin', 'member')."] = "all",  # noqa: E501
    page_number: Annotated[int | None, "Page number of the results to fetch. Use this to navigate through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-members'."]:
    """Retrieve members of a GitHub organization.

    Use this tool to list all users who are members of a specified GitHub organization, including both concealed and public members if authenticated."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/members".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"filter": filter_members, "role": member_role_filter, "per_page": results_per_page, "page": page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_organization_member(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user's handle to be removed from the organization."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/remove-member'."]:
    """Remove a user from an organization's access list.

    Use this tool to remove a user from all teams in a GitHub organization, revoking their access to the organization's repositories. Call this when you need to manage access permissions for organization members."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/members/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_user_membership_in_org(
    context: ToolContext,
    github_user_handle: Annotated[str, "The username or handle for the GitHub user account being checked for organization membership."],  # noqa: E501
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization to check membership against."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/check-membership-for-user'."]:
    """Checks if a user is a member of a GitHub organization.

    This tool determines whether a specified user is a member of a specified GitHub organization, either publicly or privately. Use this tool to verify membership status in organizational repositories or workflows."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/members/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_org_member(
    context: ToolContext,
    github_username: Annotated[str, "The GitHub username to remove from the organization; it is not case-sensitive."],  # noqa: E501
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/remove-membership-for-user'."]:
    """Remove a user's membership from a GitHub organization.

    Invoke this tool to remove a user from an organization's membership on GitHub. If the user is a member, they'll be removed; if invited, the invitation will be canceled. The user will be notified via email."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_username),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_user_org_membership_status(
    context: ToolContext,
    github_user_handle: Annotated[str, "The handle for the GitHub user account to check membership status."],  # noqa: E501
    organization_name: Annotated[str, "The GitHub organization name. It is not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/get-membership-for-user'."]:
    """Get a user's membership status in an organization.

    Call this tool to retrieve the membership status of a user within a specified GitHub organization. Requires the authenticated user to be an organization member."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_org_membership(
    context: ToolContext,
    github_user_handle: Annotated[str, "The handle for the GitHub user account to be added or updated in the organization."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    user_role_in_organization: Annotated[str | None, "Specify the user's role in the organization. Options are 'admin' for organization owner, or 'member' for non-owner."] = "member",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/set-membership-for-user'."]:
    """Manage user membership for a GitHub organization.

    This tool allows authenticated organization owners on GitHub to add members to an organization or update their roles. It sends invitations to new members or modifies the roles of existing ones. Use this tool to manage user roles, either by inviting them or updating their status within the organization."""  # noqa: E501
    request_data: Any = {"role": user_role_in_organization}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_recent_github_migrations(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This value is not case sensitive. Use to specify which organization's migrations to list."],  # noqa: E501
    exclude_attributes: Annotated[list[str] | None, "A list of attributes to exclude from the API response to enhance performance."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Page number to fetch specific results from the list of migrations."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum value of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/list-for-org'."]:
    """Retrieve the latest GitHub migrations for an organization.

    Use this tool to list recent migrations for a specified GitHub organization, including export and import activities. The list includes `repositories` only for export migrations."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "exclude": exclude_attributes}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def start_org_migration(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. The name is not case sensitive."],  # noqa: E501
    repositories_to_migrate: Annotated[list[str], "A list of repository names to be included in the migration process."],  # noqa: E501
    exclude_attachments: Annotated[bool | None, "Set to true to exclude attachments from the migration, reducing archive file size."] = False,  # noqa: E501
    exclude_git_data: Annotated[bool | None, "Set to true to exclude repository git data from the migration, reducing archive size. Useful for metadata-only migrations."] = False,  # noqa: E501
    exclude_items: Annotated[list[str] | None, "Specify related items to exclude from the response for performance optimization, e.g., [\"repositories\"]."] = None,  # noqa: E501
    exclude_metadata: Annotated[bool | None, "Set to true to exclude metadata, including only git source. Useful for reducing file complexity."] = False,  # noqa: E501
    exclude_owner_projects: Annotated[bool | None, "Indicate whether projects owned by the organization or users should be excluded from the migration."] = False,  # noqa: E501
    exclude_releases: Annotated[bool | None, "Set to true to exclude releases from the migration archive, reducing file size."] = False,  # noqa: E501
    lock_repositories: Annotated[bool | None, "Set to true to lock repositories during migration, preventing changes."] = False,  # noqa: E501
    only_include_org_metadata: Annotated[bool | None, "Specify whether only organization metadata is included, keeping the repositories array empty and ignoring other flags."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/start-for-org'."]:
    """Initiates a migration archive for a GitHub organization.

    Use this tool to start the process of generating a migration archive for a specified GitHub organization. This is useful for moving or backing up repositories and data from an organization."""  # noqa: E501
    request_data: Any = {"exclude": exclude_items, "exclude_attachments": exclude_attachments, "exclude_git_data": exclude_git_data, "exclude_metadata": exclude_metadata, "exclude_owner_projects": exclude_owner_projects, "exclude_releases": exclude_releases, "lock_repositories": lock_repositories, "org_metadata_only": only_include_org_metadata, "repositories": repositories_to_migrate}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_org_migration_status(
    context: ToolContext,
    migration_id: Annotated[int, "The unique identifier of the migration. Expected as an integer."],
    organization_name: Annotated[str, "The name of the organization, not case sensitive, for which to fetch migration status."],  # noqa: E501
    exclude_attributes: Annotated[list[str] | None, "List of attribute names to exclude from the API response for improved performance."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/get-status-for-org'."]:
    """Fetches the status of an organization's migration.

    This tool retrieves the current status of a specific migration for an organization on GitHub. The migration status can be 'pending', 'exporting', 'exported', or 'failed'. Use this tool to monitor migration progress or verify completion."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations/{migration_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, migration_id=migration_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"exclude": exclude_attributes}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_org_migration_archive(
    context: ToolContext,
    migration_identifier: Annotated[int, "The unique identifier of the migration archive to delete."],  # noqa: E501
    organization_name: Annotated[str, "The GitHub organization name. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/delete-archive-for-org'."]:  # noqa: E501
    """Delete a previous GitHub organization migration archive.

    Use this tool to delete a migration archive for a GitHub organization. Migration archives are automatically deleted after seven days, but you might need to delete them sooner for organizational purposes."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations/{migration_id}/archive".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, migration_id=migration_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fetch_org_migration_archive_url(
    context: ToolContext,
    migration_identifier: Annotated[int, "The unique identifier for the specific migration. This should be an integer value."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization. This value is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/download-archive-for-org'."]:  # noqa: E501
    """Fetches the URL to download an organization's migration archive.

    Use this tool to obtain the URL for downloading an archive of a migration for a specific organization on GitHub."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations/{migration_id}/archive".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, migration_id=migration_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def unlock_github_repo_for_org_migration(
    context: ToolContext,
    migration_unique_identifier: Annotated[int, "The unique identifier for the migration process."],
    organization_name: Annotated[str, "The GitHub organization name, which is not case sensitive."],
    repository_name: Annotated[str, "The name of the repository to be unlocked. This is required and should match the exact repository name used during the migration. Case sensitivity does not matter."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/unlock-repo-for-org'."]:
    """Unlock a locked repository after migration for an organization.

    Use this tool to unlock a repository that was locked for a migration within an organization on GitHub. This is necessary after completing a migration to access the repository again. Once unlocked, consider deleting the repository if you no longer need the source data."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations/{migration_id}/repos/{repo_name}/lock".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, migration_id=migration_unique_identifier, repo_name=repository_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repos_for_org_migration(
    context: ToolContext,
    migration_unique_identifier: Annotated[int, "The unique identifier for the organization migration in GitHub."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization. This name is not case sensitive and identifies the GitHub organization for which the migration repositories will be listed."],  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'migrations/list-repos-for-org'."]:
    """List all repositories for an organization's migration.

    Use this tool to retrieve a list of all repositories associated with a specific organization migration on GitHub. It's useful for tracking migration progress or fetching details about repositories being moved."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/migrations/{migration_id}/repositories".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, migration_id=migration_unique_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_outside_collaborators(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name, not case sensitive."],
    filter_outside_collaborators: Annotated[str | None, "Specify '2fa_disabled' to filter for collaborators without two-factor authentication, or 'all' for all collaborators."] = "all",  # noqa: E501
    result_page_number: Annotated[int | None, "Specify the page number to fetch results from the list of outside collaborators."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-outside-collaborators'."]:
    """Retrieve outside collaborators for a GitHub organization.

    Use this tool to fetch a list of users who are outside collaborators for a specified GitHub organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/outside_collaborators".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"filter": filter_outside_collaborators, "per_page": results_per_page, "page": result_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_org_outside_collaborator(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle to remove from the organization's repositories."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization from which to remove the user. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/remove-outside-collaborator'."]:  # noqa: E501
    """Remove a user from all organization repositories.

    Use this tool to remove a user from all repositories in a specified organization by removing them from the outside collaborators list."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/outside_collaborators/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def convert_member_to_outside_collaborator(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user handle to be converted to an outside collaborator."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    perform_asynchronously: Annotated[bool | None, "Set to true to perform the request asynchronously, queuing the job with a 202 status code."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/convert-member-to-outside-collaborator'."]:  # noqa: E501
    """Convert GitHub org member to outside collaborator.

    Use this tool to convert an organization member into an outside collaborator on GitHub. The member will lose their organization membership and retain access only to repositories allowed by their team memberships. This action might be restricted by enterprise policies."""  # noqa: E501
    request_data: Any = {"async": perform_asynchronously}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/outside_collaborators/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_pre_receive_hooks(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case-sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "Page number of the results to fetch. Use this to navigate through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page. The maximum allowed is 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "The direction to sort the results by. Accepts 'asc' or 'desc'."] = "desc",  # noqa: E501
    sort_order: Annotated[str | None, "Specify the sort order for the response: options are 'created', 'updated', or 'name'."] = "created",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/list-pre-receive-hooks-for-org'."]:  # noqa: E501
    """Retrieve pre-receive hooks for a GitHub organization.

    Use this tool to list all pre-receive hooks that are enabled, testing, or configurable at an organization level in GitHub. It excludes globally disabled hooks with no downstream configuration options."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/pre-receive-hooks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number, "direction": sort_direction, "sort": sort_order}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_org_pre_receive_hook(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization where the pre-receive hook enforcement override will be removed. It is not case sensitive."],  # noqa: E501
    pre_receive_hook_id: Annotated[int, "The unique identifier for the pre-receive hook to be removed."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/remove-pre-receive-hook-enforcement-for-org'."]:  # noqa: E501
    """Removes pre-receive hook enforcement overrides for an organization.

    Use this tool to remove any pre-receive hook enforcement overrides at the organization level. This action is applicable when you need to revert to the default hook settings set at a higher level than the organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, pre_receive_hook_id=pre_receive_hook_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_org_pre_receive_hook(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization; case insensitive."],
    pre_receive_hook_unique_id: Annotated[int, "The unique identifier of the pre-receive hook. Must be an integer."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/get-pre-receive-hook-for-org'."]:  # noqa: E501
    """Retrieve a pre-receive hook for an organization.

    Fetches details of a specific pre-receive hook associated with a GitHub organization using the organization's name and the hook ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, pre_receive_hook_id=pre_receive_hook_unique_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_pre_receive_hook_enforcement(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It's not case sensitive."],  # noqa: E501
    pre_receive_hook_id: Annotated[int, "The unique identifier of the pre-receive hook to be updated."],  # noqa: E501
    allow_downstream_configuration: Annotated[bool | None, "Boolean indicating whether repositories can override the enforcement settings of the pre-receive hook."] = None,  # noqa: E501
    enforcement_state: Annotated[str | None, "Specify the state of enforcement for the hook on this repository. Possible values may include 'enabled', 'disabled', etc."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'enterprise-admin/update-pre-receive-hook-enforcement-for-org'."]:  # noqa: E501
    """Update pre-receive hook enforcement for a GitHub organization.

    This tool updates the enforcement settings and downstream configuration for pre-receive hooks at the organization level on GitHub. Call this tool to modify enforcement and downstream configuration for a specific pre-receive hook within an organization."""  # noqa: E501
    request_data: Any = {"allow_downstream_configuration": allow_downstream_configuration, "enforcement": enforcement_state}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/pre-receive-hooks/{pre_receive_hook_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, pre_receive_hook_id=pre_receive_hook_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_projects(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization for which to list projects."],  # noqa: E501
    project_state: Annotated[str | None, "Specifies the state of projects to return: 'open', 'closed', or 'all'."] = "open",  # noqa: E501
    results_page_number: Annotated[int | None, "Specifies the page number of the results to fetch for organization projects."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of project results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/list-for-org'."]:
    """Retrieve a list of projects for a given organization on GitHub.

    Use this tool to obtain a list of projects within a specified GitHub organization. It returns an error if projects are disabled or if access is unauthorized."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/projects".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"state": project_state, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_org_project_github(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. Not case sensitive."],
    project_name: Annotated[str, "The name of the project board to be created. It will serve as the identifier for the project within the organization. Must be a string."],  # noqa: E501
    project_description: Annotated[str | None, "The description of the project to be created for the GitHub organization. This should be a clear and concise explanation of the project's purpose."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/create-for-org'."]:
    """Create a project board for a GitHub organization.

    This tool creates a project board within a specified GitHub organization. It should be called when you need to set up a new project board for organizational tasks. It returns details about the created project or errors if projects are disabled, insufficient privileges are available, or the organization lacks classic projects."""  # noqa: E501
    request_data: Any = {"body": project_description, "name": project_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/projects".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_public_org_members(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-public-members'."]:
    """Retrieve public members of a GitHub organization.

    Use this tool to obtain a list of members who have publicized their membership in a specified GitHub organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/public_members".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_public_org_membership(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle for which to remove public organization membership."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. This name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/remove-public-membership-for-authenticated-user'."]:  # noqa: E501
    """Remove public organization membership for the user.

    This tool removes the public membership of the authenticated user from a specified GitHub organization. Call this tool when a user wants to revoke their public membership from an organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/public_members/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_github_org_membership(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle to check for public organization membership."],  # noqa: E501
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/check-public-membership-for-user'."]:  # noqa: E501
    """Checks if a user is a public member of a GitHub organization.

    Use this tool to determine if a specific user is publicly a member of a particular GitHub organization. This can be useful for confirming a user's affiliation with an organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/public_members/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_own_github_public_membership(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user account handle to publicize the membership for."],  # noqa: E501
    organization_name: Annotated[str, "The GitHub organization name to make membership public. Case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/set-public-membership-for-authenticated-user'."]:  # noqa: E501
    """Publicize your GitHub organization membership.

    Use this tool to make your GitHub organization membership public. It can only be used for your own membership, not for others."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/public_members/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_repositories(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "Page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    repository_type: Annotated[str | None, "Specify the type of repositories to return, such as 'all', 'public', 'private', etc. Note: 'internal' is unsupported with an installation access token."] = None,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of repository results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
    sort_order: Annotated[str | None, "Specifies the sorting order of the results. Use 'asc' for ascending or 'desc' for descending."] = None,  # noqa: E501
    sort_property: Annotated[str | None, "Specifies the property to sort the repository results by, such as created, updated, pushed, or full_name."] = "created",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/list-for-org'."]:
    """Retrieve repositories for a specific organization on GitHub.

    Fetches a list of repositories belonging to a specified organization on GitHub. Admin permissions or ownership may be required to access certain repository details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/repos".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"type": repository_type, "sort": sort_property, "direction": sort_order, "per_page": results_per_page, "page": page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_organization_repo(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    organization_name: Annotated[str | None, "The name of the GitHub organization where the repository will be created. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/create-in-org'."]:
    """Create a new repository in a GitHub organization.

    Use this tool to create a new repository within a specified GitHub organization. The user must have appropriate OAuth scopes and be a member of the organization.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBORGANIZATIONREPO"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not organization_name:
        missing_params.append(("organization_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBORGANIZATIONREPO"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBORGANIZATIONREPO"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/orgs/{org}/repos".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBORGANIZATIONREPO"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_org_secret_scanning_alerts(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization for which secret scanning alerts are to be listed. This value is not case-sensitive."],  # noqa: E501
    alert_resolution_filter: Annotated[str | None, "A comma-separated list of resolutions to filter secret scanning alerts. Valid options are `false_positive`, `wont_fix`, `revoked`, `pattern_edited`, `pattern_deleted`, and `used_in_tests`."] = None,  # noqa: E501
    alert_state: Annotated[str | None, "Specify 'open' or 'resolved' to filter secret scanning alerts by their state."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of results to retrieve, starting from 1. Determines which subset of results will be returned."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
    search_after_cursor: Annotated[str | None, "A cursor for paginating results, provided in the Link header. Use an empty string for the initial request to receive a starting cursor."] = None,  # noqa: E501
    search_before_cursor: Annotated[str | None, "A cursor indicating that the query should only look for events before this point. Use an empty string to get an initial cursor."] = None,  # noqa: E501
    secret_types: Annotated[str | None, "Comma-separated list of secret types to return. Defaults to all secret types. Refer to the GitHub documentation for details on secret types."] = None,  # noqa: E501
    sort_by_property: Annotated[str | None, "Choose 'created' to sort by alert creation date or 'updated' to sort by last update or resolution."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "Specifies the order to sort the results: ascending ('asc') or descending ('desc')."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'secret-scanning/list-alerts-for-org'."]:  # noqa: E501
    """Retrieve secret scanning alerts for an organization's repositories.

    This tool provides a list of secret scanning alerts from eligible repositories within an organization, sorted from newest to oldest. It should be called by users who are administrators or security managers of the organization, using an appropriate access token. GitHub Apps with `secret_scanning_alerts` read permission can also utilize this endpoint."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/secret-scanning/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"state": alert_state, "secret_type": secret_types, "resolution": alert_resolution_filter, "sort": sort_by_property, "direction": sort_direction, "page": results_page_number, "per_page": results_per_page, "before": search_before_cursor, "after": search_after_cursor}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_security_manager_teams(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization's name. It is not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/list-security-manager-teams'."]:  # noqa: E501
    """Retrieve teams that are security managers in an organization.

    This tool retrieves a list of teams designated as security managers for a specified organization on GitHub. It should be used when you need to identify teams responsible for security within an organization. Requires appropriate access rights, like being an administrator or security manager with a token having the `read:org` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/security-managers".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_security_manager_role(
    context: ToolContext,
    organization_name: Annotated[str, "The organization's name. It is not case sensitive."],
    team_identifier: Annotated[str, "The unique slug identifying the team by name. This is required to specify the team whose security manager role is to be removed."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/remove-security-manager-team'."]:  # noqa: E501
    """Remove security manager role from a team in an organization.

    Use this tool to remove the security manager role from a specific team within an organization on GitHub. This requires the user to be an organization administrator with an access token having the 'admin:org' scope, or for GitHub Apps, the 'administration' organization read-write permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/security-managers/teams/{team_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_security_manager_team(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive. Required for adding a team as a security manager."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name to be added as a security manager. This is case-insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/add-security-manager-team'."]:
    """Add a team as a security manager for an organization.

    Use this tool to designate a specific team as the security manager for a GitHub organization. Requires organization admin rights and a token with `write:org` scope. Useful for managing security roles within an organization."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/security-managers/teams/{team_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_advanced_security_committers(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name for which to retrieve security committers. It is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch. Useful for paginating through large sets of results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum limit of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'billing/get-github-advanced-security-billing-org'."]:  # noqa: E501
    """Retrieve GitHub Advanced Security committers for an organization.

    Use this tool to obtain details about active committers using GitHub Advanced Security across repositories within a specified organization. This includes the total count of unique committers and repository-level details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/settings/billing/advanced-security".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_organization_teams(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. It is not case sensitive."],
    results_page_number: Annotated[int | None, "Page number of the results to fetch from the list of teams."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list'."]:
    """Retrieve teams visible to the user in a GitHub organization.

    Use this tool to list all teams within a specified GitHub organization that the authenticated user can see."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_team(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization where the team will be created."],  # noqa: E501
    team_name: Annotated[str, "The name of the team to be created. It should be a string."],
    deprecated_repository_permission: Annotated[str | None, "Specifies the permission for new repositories, though it is deprecated. Options are `pull` or `push`."] = "pull",  # noqa: E501
    ldap_distinguished_name: Annotated[str | None, "The distinguished name (DN) of the LDAP entry to map to a team. Ensure LDAP synchronization is enabled."] = None,  # noqa: E501
    parent_team_id: Annotated[int | None, "The numerical ID of the team to assign as the parent for the new team."] = None,  # noqa: E501
    repository_names_to_add_to_team: Annotated[list[str] | None, "Array of full repository names (e.g., \"organization-name/repository-name\") to associate with the team."] = None,  # noqa: E501
    team_description: Annotated[str | None, "A brief description of the team being created. This helps specify the team's purpose or role within the organization."] = None,  # noqa: E501
    team_maintainers_github_ids: Annotated[list[str] | None, "List of GitHub IDs for organization members who will become team maintainers."] = None,  # noqa: E501
    team_privacy_level: Annotated[str | None, "Specifies if the team is 'secret' or 'closed'. Defaults: 'secret' for non-nested teams, 'closed' for parent/child teams."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/create'."]:
    """Create a new team in a GitHub organization.

    Use this tool to create a new team within a specified GitHub organization. The authenticated user must be a member or owner of the organization. Upon creation, the user becomes a team maintainer by default."""  # noqa: E501
    request_data: Any = {"description": team_description, "ldap_dn": ldap_distinguished_name, "maintainers": team_maintainers_github_ids, "name": team_name, "parent_team_id": parent_team_id, "permission": deprecated_repository_permission, "privacy": team_privacy_level, "repo_names": repository_names_to_add_to_team}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_team_in_org(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization containing the team to be deleted."],  # noqa: E501
    team_slug: Annotated[str, "The unique slug identifier of the team within the organization to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/delete-in-org'."]:
    """Delete a team in a GitHub organization.

    Use this tool to delete a team within a GitHub organization. The action requires the user to be an organization owner or team maintainer. Deleting a parent team will also delete its child teams."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_team_by_slug(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name. This is a URL-friendly version, typically all lowercase with special characters and spaces replaced by hyphens."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/get-by-name'."]:
    """Retrieve team details using organization and team slug.

    Use this tool to get detailed information about a specific team in a GitHub organization by providing the organization's name and the team's slug. The slug is a modified version of the team's name, formatted by GitHub."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_team(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    team_slug: Annotated[str, "The slug of the team name, used to uniquely identify the team within the organization."],  # noqa: E501
    parent_team_id: Annotated[int | None, "The ID of the team to set as the parent team for nesting purposes."] = None,  # noqa: E501
    team_description: Annotated[str | None, "The description of the team. Provide a concise summary or details for team identification."] = None,  # noqa: E501
    team_name: Annotated[str | None, "The new name for the GitHub team within the organization."] = None,  # noqa: E501
    team_privacy_level: Annotated[str | None, "Specifies the team's privacy level. Options: 'secret' (visible only to owners and team members) or 'closed' (visible to all organization members). Parent teams cannot be 'secret'."] = None,  # noqa: E501
    team_repository_permission: Annotated[str | None, "**Deprecated**. Specifies the default permission for newly added repositories. Options: 'pull', 'push', 'admin'."] = "pull",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/update-in-org'."]:
    """Update a team's details within a GitHub organization.

    This tool allows modifying the details of a specific team within a GitHub organization. The authenticated user must be an organization owner or team maintainer to perform this action. It's intended for updating team information such as name or permissions."""  # noqa: E501
    request_data: Any = {"description": team_description, "name": team_name, "parent_team_id": parent_team_id, "permission": team_repository_permission, "privacy": team_privacy_level}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_discussions(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization."],
    team_slug: Annotated[str, "The slug of the team's name to identify which team's discussions to retrieve in the organization."],  # noqa: E501
    pinned_discussions_only: Annotated[str | None, "Filter to retrieve only pinned discussions. Use 'true' for pinned only, 'false' for all."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The specific page number of discussion results to retrieve."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page, with a maximum of 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "Specify the sorting direction for the results. Use 'asc' for ascending or 'desc' for descending order."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-discussions-in-org'."]:
    """Retrieve all discussions from a team's page in an organization.

    Use this tool to get a list of discussions on a specified team's page within a GitHub organization. Requires OAuth access tokens with the `read:discussion` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"direction": sort_direction, "per_page": results_per_page, "page": results_page_number, "pinned": pinned_discussions_only}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_team_discussion_github(
    context: ToolContext,
    discussion_body_text: Annotated[str, "The content of the discussion post. Provide detailed text for the discussion body."],  # noqa: E501
    discussion_post_title: Annotated[str, "The title for the discussion post on the team's page."],
    organization_name: Annotated[str, "The organization name, not case-sensitive, for which the team discussion will be created."],  # noqa: E501
    team_slug: Annotated[str, "The unique slug of the team name. This is required to specify which team's page the discussion will be posted on."],  # noqa: E501
    create_private_post: Annotated[bool | None, "Set to `true` to create a private post visible only to team members and maintainers, or `false` for a public post visible to all organization members."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/create-discussion-in-org'."]:
    """Create a discussion post on a GitHub team's page.

    Use this tool to create a new discussion post on a specified team's page within an organization on GitHub. This requires appropriate OAuth access tokens with the `write:discussion` scope. Be aware that creating content too quickly can lead to rate limiting. This action will trigger notifications associated with the created discussion."""  # noqa: E501
    request_data: Any = {"body": discussion_body_text, "private": create_private_post, "title": discussion_post_title}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_team_discussion(
    context: ToolContext,
    discussion_number: Annotated[int, "The unique number identifying the discussion to be deleted."],  # noqa: E501
    organization_name: Annotated[str, "The organization name on GitHub. It is not case sensitive."],
    team_slug: Annotated[str, "The slug identifier of the team name on GitHub. This is required to specify which team's discussion is to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/delete-discussion-in-org'."]:
    """Delete a discussion from a team's page on GitHub.

    This tool deletes a specified discussion from a team's page within an organization on GitHub. It requires an OAuth access token with `write:discussion` scope. Use this tool when you need to remove a discussion from a team's page."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_number),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_team_discussion(
    context: ToolContext,
    discussion_identifier_number: Annotated[int, "The unique number identifying the discussion to retrieve."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the GitHub team name, used to specify the team."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/get-discussion-in-org'."]:
    """Retrieve a specific team discussion from GitHub.

    Call this tool to get details of a specific discussion on a GitHub team's page. This is useful when you need information about a particular topic discussed within a team. Ensure OAuth access tokens with `read:discussion` scope are used."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_identifier_number),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_team_discussion(
    context: ToolContext,
    discussion_id: Annotated[int, "The unique number identifying the discussion to be updated."],
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive and uniquely identifies the organization on GitHub."],  # noqa: E501
    team_slug: Annotated[str, "Provide the slug (URL-friendly version) of the team's name. Case sensitivity is ignored."],  # noqa: E501
    discussion_body_text: Annotated[str | None, "The updated body text of the discussion post. Provide the new content you want for the discussion."] = None,  # noqa: E501
    discussion_title: Annotated[str | None, "The new title for the discussion post. Only the provided title will be updated."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/update-discussion-in-org'."]:
    """Edits the title and body of a team discussion post.

    Use this tool to update the title and content of a discussion in a GitHub team within an organization. Only the specified fields will be changed. Requires OAuth tokens with `write:discussion` scope."""  # noqa: E501
    request_data: Any = {"body": discussion_body_text, "title": discussion_title}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_id),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_discussion_comments(
    context: ToolContext,
    discussion_id: Annotated[int, "The unique number identifying the discussion to retrieve comments from."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    team_slug: Annotated[str, "The identifier for the team, typically a URL-friendly version of the team name."],  # noqa: E501
    results_page_number: Annotated[int | None, "The specific page of discussion comments to retrieve, starting with 1 for the first page."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of discussion comments to return per page, maximum of 100."] = 30,  # noqa: E501
    sort_direction: Annotated[str | None, "Specify the sort order for results: 'asc' for ascending or 'desc' for descending."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-discussion-comments-in-org'."]:  # noqa: E501
    """Retrieve comments from a team discussion in an organization.

    This tool lists all comments on a specified team discussion within an organization on GitHub. It requires an OAuth token with the `read:discussion` scope and can be used when you need to see all comments made in a particular team discussion."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"direction": sort_direction, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_team_discussion_comment(
    context: ToolContext,
    comment_body_text: Annotated[str, "The text content of the comment to be added to the team discussion."],  # noqa: E501
    discussion_number: Annotated[int, "The number that identifies the specific discussion within the team to which you want to add a comment."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization where the team discussion is located. This is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug identifier for the team name within the organization, used to specify which team's discussion to comment on."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/create-discussion-comment-in-org'."]:  # noqa: E501
    """Create a new comment on a team discussion in an organization.

    Use this tool to add a comment to an existing team discussion in a GitHub organization. This action requires appropriate permissions and will generate notifications. Be mindful of rate limits when using this tool."""  # noqa: E501
    request_data: Any = {"body": comment_body_text}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_number),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_team_discussion_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "The unique number identifying the comment to be deleted."],
    discussion_identifier: Annotated[int, "The unique number identifying the discussion for the comment to be deleted."],  # noqa: E501
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    team_slug: Annotated[str, "The slug of the team name in the organization. Case insensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/delete-discussion-comment-in-org'."]:  # noqa: E501
    """Deletes a comment on a team discussion in an organization.

    Use this tool to delete a specific comment from a team discussion within a GitHub organization. Ideal for managing discussions and removing unwanted comments. Requires `write:discussion` OAuth scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_identifier, comment_number=comment_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_team_discussion_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "The specific number identifying the comment in the discussion."],  # noqa: E501
    discussion_id: Annotated[int, "The unique number identifying the specific discussion on GitHub."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. This is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug (URL-friendly version) of the GitHub team's name. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/get-discussion-comment-in-org'."]:  # noqa: E501
    """Retrieve a specific comment from a GitHub team discussion.

    Use this tool to get details of a specific comment within a team discussion on GitHub. This is useful for accessing detailed information about comments in organizational discussions. Ensure OAuth tokens have the `read:discussion` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_id, comment_number=comment_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_discussion_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "A unique integer identifying the comment to be updated in the discussion."],  # noqa: E501
    discussion_comment_body: Annotated[str, "The new text for the discussion comment to be updated."],  # noqa: E501
    discussion_id: Annotated[int, "The unique number identifying the GitHub discussion to update the comment in."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case-sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name. This is used to specify the team in the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/update-discussion-comment-in-org'."]:  # noqa: E501
    """Updates a GitHub discussion comment's text.

    This tool edits the body text of a discussion comment in a GitHub organization team. Use it when you need to update the content of a comment in a team discussion. Requires OAuth access tokens with `write:discussion` scope."""  # noqa: E501
    request_data: Any = {"body": discussion_comment_body}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_id, comment_number=comment_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_discussion_comment_reactions(
    context: ToolContext,
    comment_identifier: Annotated[int, "The unique number identifying the discussion comment."],
    discussion_number: Annotated[int, "The number identifying the specific discussion in the team."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name, case-insensitive, used to identify the team in the organization."],  # noqa: E501
    filter_by_reaction_type: Annotated[str | None, "Specify a single reaction type to filter results. Options: '+1', '-1', 'laugh', 'confused', 'heart', 'hooray', 'rocket', 'eyes'. Omit to list all reactions."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the comments reactions to fetch from the results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page to return (maximum 100)."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/list-for-team-discussion-comment-in-org'."]:  # noqa: E501
    """Retrieve reactions for a team discussion comment in an organization.

    This tool retrieves the list of reactions for a specific comment in a team discussion within an organization on GitHub. It requires OAuth access tokens with the `read:discussion` scope. Use this when you need to analyze or display reactions to a particular discussion comment."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}/reactions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_number, comment_number=comment_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"content": filter_by_reaction_type, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_reaction_to_team_discussion_comment(
    context: ToolContext,
    comment_identifier: Annotated[int, "The unique number identifying the team discussion comment to react to."],  # noqa: E501
    discussion_identifier: Annotated[int, "The number identifying the discussion within the team."],
    organization_name: Annotated[str, "The name of the organization. This value is not case sensitive."],  # noqa: E501
    reaction_type: Annotated[str, "The type of reaction emoji to add to the team discussion comment. Accepted values are: '+1', '-1', 'laugh', 'confused', 'heart', 'hooray', 'rocket', 'eyes'."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name within the organization, used to identify the team."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/create-for-team-discussion-comment-in-org'."]:  # noqa: E501
    """Add a reaction to a GitHub team discussion comment.

    Use this tool to add a reaction emoji to a specific GitHub team discussion comment within an organization. Useful for engaging with team discussions or acknowledging comments. Requires appropriate OAuth scope."""  # noqa: E501
    request_data: Any = {"content": reaction_type}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}/reactions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_identifier, comment_number=comment_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_team_discussion_comment_reaction(
    context: ToolContext,
    comment_identifier: Annotated[int, "The number that identifies the comment in the team discussion."],  # noqa: E501
    discussion_identifier: Annotated[int, "The number identifying the specific discussion in the team."],  # noqa: E501
    organization_name: Annotated[str, "The organization name on GitHub. It is not case sensitive."],
    reaction_id: Annotated[int, "The unique identifier of the reaction to be deleted. This should be an integer value."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name, used to identify the team in the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/delete-for-team-discussion-comment'."]:  # noqa: E501
    """Delete a reaction from a team discussion comment on GitHub.

    Use this tool to delete a reaction from a specific comment in a team discussion on GitHub. You need appropriate OAuth access tokens with `write:discussion` scope. Specify team, organization, discussion, comment, and reaction details to execute."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/comments/{comment_number}/reactions/{reaction_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_identifier, comment_number=comment_identifier, reaction_id=reaction_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_team_discussion_reactions(
    context: ToolContext,
    discussion_identifier: Annotated[int, "The number identifying the team discussion to retrieve reactions for."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team's name in the specified GitHub organization. It identifies the team for which reactions are being retrieved."],  # noqa: E501
    reaction_type: Annotated[str | None, "Specify the type of reaction to retrieve (e.g., '+1', '-1', 'laugh'). Omit to retrieve all reactions."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch. Use to navigate through paginated responses."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/list-for-team-discussion-in-org'."]:  # noqa: E501
    """Retrieve reactions to a specific team discussion in a GitHub organization.

    Use this tool to obtain the reactions to a team discussion within a GitHub organization. It requires an OAuth token with the `read:discussion` scope. Useful for analyzing feedback or participation in team discussions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/reactions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"content": reaction_type, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_reaction_to_github_team_discussion(
    context: ToolContext,
    discussion_id: Annotated[int, "The unique identifier number for the team discussion."],
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    reaction_type: Annotated[str, "The reaction type to add to the team discussion. Valid options include: '+1', '-1', 'laugh', 'confused', 'heart', 'hooray', 'rocket', 'eyes'."],  # noqa: E501
    team_slug: Annotated[str, "The URL-friendly version of the team's name."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/create-for-team-discussion-in-org'."]:  # noqa: E501
    """Add a reaction to a GitHub team discussion.

    Use this tool to add a specific reaction emoji to a team discussion in a GitHub organization. It requires OAuth access tokens with the `write:discussion` scope. This is useful for engaging with team discussions directly through the API."""  # noqa: E501
    request_data: Any = {"content": reaction_type}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/reactions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_team_discussion_reaction(
    context: ToolContext,
    discussion_number: Annotated[int, "The number that identifies the GitHub team discussion to delete a reaction from. Must be an integer."],  # noqa: E501
    organization_name: Annotated[str, "The organization name. This value is not case sensitive and identifies the GitHub organization."],  # noqa: E501
    reaction_unique_identifier: Annotated[int, "The unique identifier for the specific reaction to be deleted from the discussion."],  # noqa: E501
    team_slug: Annotated[str, "The slug (URL-friendly version) of the team name in GitHub, used to identify a team within an organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'reactions/delete-for-team-discussion'."]:  # noqa: E501
    """Delete a reaction from a GitHub team discussion.

    Use this tool to delete a reaction from a specific team discussion on GitHub. Requires appropriate OAuth scope: `write:discussion`."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/discussions/{discussion_number}/reactions/{reaction_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, discussion_number=discussion_number, reaction_id=reaction_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def unlink_external_idp_group_from_team(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This is not case sensitive."],
    team_slug: Annotated[str, "The slug of the team's name. It identifies the team within the organization."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/unlink-external-idp-group-from-team-for-org'."]:  # noqa: E501
    """Unlink an external IdP group from a GitHub team.

    This tool is used to remove a connection between a GitHub team and an external identity provider group. It is useful when managing team membership through an IdP in GitHub Enterprise Cloud."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/external-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_linked_external_groups(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "Slug of the team name to identify the specific GitHub team."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-linked-external-idp-groups-to-team-for-org'."]:  # noqa: E501
    """Retrieve connections between a GitHub team and external groups.

    Use this tool to list connections between a specific GitHub team and external identity provider groups within an organization. Ideal for managing team membership using Enterprise Managed Users for GitHub Enterprise Cloud."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/external-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def link_external_group_to_team(
    context: ToolContext,
    external_group_id: Annotated[int, "The ID of the external group to be linked with the GitHub team."],  # noqa: E501
    organization_name: Annotated[str, "The GitHub organization name. This value is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name to connect with an external group. It is case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/link-external-idp-group-to-team-for-org'."]:  # noqa: E501
    """Link an external IDP group to a GitHub team.

    Creates a connection between a GitHub team and an external identity provider group. Use this tool when you need to manage team membership through your identity provider using GitHub Enterprise Managed Users."""  # noqa: E501
    request_data: Any = {"group_id": external_group_id}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/external-groups".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_members_in_org(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name within the organization. Used to identify the specific team."],  # noqa: E501
    filter_by_role: Annotated[str | None, "Filters team members by their role: 'member', 'maintainer', or 'all'."] = "all",  # noqa: E501
    result_page_number: Annotated[int | None, "The page number of results to fetch. Use this for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of team members to return per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-members-in-org'."]:
    """Retrieve team members in a specified organization.

    Call this tool to get a list of members in a specific team within an organization on GitHub. It includes members from child teams, provided that the authenticated user has visibility of the team."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/members".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"role": filter_by_role, "per_page": results_per_page, "page": result_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_team_membership(
    context: ToolContext,
    github_user_handle: Annotated[str, "The handle for the GitHub user account to be removed from the team."],  # noqa: E501
    organization_name: Annotated[str, "The GitHub organization name. This input is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug identifier for the GitHub team's name. This is used to specify the team you want to modify."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/remove-membership-for-user-in-org'."]:  # noqa: E501
    """Remove a user's membership from a GitHub team.

    Use this tool to remove a user's membership from a specific GitHub team within an organization. Ensure the authenticated user has 'admin' permissions or is an owner of the organization. Note that team synchronization settings may affect API changes."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_user_team_membership_in_org(
    context: ToolContext,
    github_username: Annotated[str, "The GitHub username of the account whose team membership status is being retrieved. This username is not case sensitive."],  # noqa: E501
    organization_name: Annotated[str, "The case-insensitive name of the GitHub organization."],
    team_slug: Annotated[str, "The slug of the team name. It uniquely identifies the team within the organization. Case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/get-membership-for-user-in-org'."]:  # noqa: E501
    """Retrieve a user's team membership status in an organization.

    Fetches the membership state and role of a specified user within a specific team in an organization on GitHub. Useful for determining if a user is a member of a team, including the role they hold."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, username=github_username),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_update_github_team_membership(
    context: ToolContext,
    github_team_slug: Annotated[str, "The slug identifier of the team's name within the organization. This is not case-sensitive."],  # noqa: E501
    github_user_handle: Annotated[str, "The GitHub user account handle to add or update in the organization team."],  # noqa: E501
    organization_name: Annotated[str, "The name of the GitHub organization (case insensitive) to which the team belongs."],  # noqa: E501
    user_team_role: Annotated[str | None, "Specifies the role for the user in the team, either 'member' or 'maintainer'."] = "member",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/add-or-update-membership-for-user-in-org'."]:  # noqa: E501
    """Add or update a user's membership in a GitHub team.

    This tool adds a member to a GitHub organization's team or updates their role if they are already a member. It is used by authenticated organization owners or team maintainers. If the person isn't part of the organization, an invitation is sent, and their membership will be pending until accepted."""  # noqa: E501
    request_data: Any = {"role": user_team_role}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/memberships/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=github_team_slug, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_projects_in_org(
    context: ToolContext,
    organization_name: Annotated[str, "The case-insensitive name of the organization for which to list team projects."],  # noqa: E501
    team_slug: Annotated[str, "The team's unique slug identifier. This is used to specify which team's projects to list."],  # noqa: E501
    results_page_number: Annotated[int | None, "The page number for the results you wish to retrieve. Used for paginating through results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-projects-in-org'."]:
    """Retrieve a list of projects for a team in an organization.

    Use this tool to get a list of organization projects associated with a specific team based on the organization name and team slug. Ideal for fetching project details for managerial or planning purposes."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/projects".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_project_from_team(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This value is not case sensitive."],  # noqa: E501
    project_unique_identifier: Annotated[int, "The unique identifier of the project to be removed from the team."],  # noqa: E501
    team_slug: Annotated[str, "The slug identifier for the team name in the organization. Case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/remove-project-in-org'."]:
    """Remove a project from a team in a GitHub organization.

    Use this tool to remove an organization project from a specific team within GitHub. It requires appropriate permissions: `read` access to both the team and project, or `admin` access to either. The operation only removes the project from the team and does not delete the project itself."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, project_id=project_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_team_project_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    project_unique_identifier: Annotated[int, "The unique integer ID of the project to check permissions for."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name used to identify the team within the organization. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/check-permissions-for-project-in-org'."]:  # noqa: E501
    """Check team's permissions for an organization's project.

    Use this tool to determine if a team has `read`, `write`, or `admin` permissions for a specific project within an organization. This includes projects inherited from a parent team."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, project_id=project_unique_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_or_update_github_team_project_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The GitHub organization name. This is not case sensitive."],
    project_id: Annotated[int, "The unique identifier of the project to update or add to the team."],  # noqa: E501
    team_slug: Annotated[str, "The slug representation of the GitHub team's name within the organization. This is used to identify the team in the request."],  # noqa: E501
    project_permission_level: Annotated[str | None, "Permission level to grant the team for the project. Options: 'read', 'write', 'admin'. Default uses the team's current level."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/add-or-update-project-permissions-in-org'."]:  # noqa: E501
    """Add or update a GitHub team's permissions on an organization project.

    Use this tool to add a project to a GitHub team or update a team's permissions on a project within the same organization. The user must have admin permissions for the project."""  # noqa: E501
    request_data: Any = {"permission": project_permission_level}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, project_id=project_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_team_repositories(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name. Case insensitivity applies."],
    team_slug: Annotated[str, "The slug of the team name (case-insensitive)."],
    results_page_number: Annotated[int | None, "The page number of the results to fetch for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-repos-in-org'."]:
    """Retrieve a list of repositories for a specified team.

    This tool lists repositories associated with a specified team within an organization, visible to the authenticated user. Ideal for obtaining an overview of team-managed projects."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/repos".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_repo_from_team(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case-sensitive."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository to remove from the team. It is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug of the team name to identify which team's repository link should be removed. This is required and case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/remove-repo-in-org'."]:
    """Remove a repository from a GitHub team within an organization.

    Call this tool to remove a repository from a specific team in a GitHub organization. The user needs to be an organization owner, team maintainer, or have admin access to the repository. This action will only unlink the repository from the team, not delete it."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_team_repo_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The organization name on GitHub. It is not case sensitive. Required to check team permissions."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository for which you want to check permissions. The name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository, not case sensitive."],
    team_slug: Annotated[str, "The slug of the team name. Used to identify the team whose permissions you are checking."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/check-permissions-for-repo-in-org'."]:  # noqa: E501
    """Check a team's permissions for a specific repository within an organization.

    This tool verifies whether a team has specific permissions (admin, push, maintain, triage, or pull) for a given repository within an organization on GitHub. It can be used to ensure access rights are correctly configured. If the team lacks permissions, a 404 response indicates no access."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_team_repo_permissions(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. This parameter is not case sensitive."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository to be managed. The name is not case sensitive."],  # noqa: E501
    repository_owner_account: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    team_slug: Annotated[str, "The slug identifier of the team within the organization. It is not case-sensitive."],  # noqa: E501
    team_repo_permission: Annotated[str | None, "Permission to grant the team on this repository. Options: `pull`, `triage`, `push`, `maintain`, `admin`, or a custom role name defined by the organization. Defaults to team's current permission if unspecified."] = "push",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/add-or-update-repo-permissions-in-org'."]:  # noqa: E501
    """Manage team repository access and permissions.

    Use this tool to add a repository to a team or update a team's permission on a repository within an organization. The caller must have admin access and visibility of the team. The repository should be owned by the organization. Returns a confirmation of the action taken."""  # noqa: E501
    request_data: Any = {"permission": team_repo_permission}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_slug, owner=repository_owner_account, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_child_teams(
    context: ToolContext,
    organization_name: Annotated[str, "The organization's name. It's case insensitive and used to specify which organization's team structure to query."],  # noqa: E501
    team_identifier_slug: Annotated[str, "The slug of the team name for which to list child teams. This is used to uniquely identify the team within the organization."],  # noqa: E501
    page_number: Annotated[int | None, "The page number to retrieve in the list of child teams. Use to paginate the results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'teams/list-child-in-org'."]:
    """Retrieves child teams of a specified team in an organization.

    Use this tool to get a list of child teams under a specific team within an organization on GitHub. It requires specifying the team by its slug and the organization name."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/teams/{team_slug}/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, team_slug=team_identifier_slug),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def manage_org_security_features(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. This value is not case sensitive."],  # noqa: E501
    security_feature: Annotated[str, "Specifies the security feature to enable or disable. Options include: dependency_graph, dependabot_alerts, dependabot_security_updates, advanced_security, secret_scanning, secret_scanning_push_protection."],  # noqa: E501
    security_feature_action: Annotated[str, "Specifies whether to enable or disable the security feature for all organization repositories. Use 'enable_all' to activate and 'disable_all' to deactivate."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'orgs/enable-or-disable-security-product-on-all-org-repos'."]:  # noqa: E501
    """Toggle security features for all repositories in an organization.

    This tool is used to enable or disable a specified security feature across all repositories in a GitHub organization. It requires organization owner access or a security manager role, and a token with 'write:org' scope. GitHub Apps must have the 'organization_administration:write' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/orgs/{org}/{security_product}/{enablement}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, security_product=security_feature, enablement=security_feature_action),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_project_card(
    context: ToolContext,
    card_id: Annotated[int, "The unique identifier of the project card to delete."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/delete-card'."]:
    """Delete a project card from GitHub projects.

    Call this tool when you need to delete a specific project card from a GitHub project board by providing the card ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/cards/{card_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), card_id=card_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_project_card(
    context: ToolContext,
    project_card_id: Annotated[int, "The unique ID of the GitHub project card to retrieve details for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/get-card'."]:
    """Retrieve details of a specific project card in GitHub.

    Use this tool to get information about a project card in a GitHub project by specifying the card ID. Useful for retrieving details such as card content, column association, and other metadata."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/cards/{card_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), card_id=project_card_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_project_card(
    context: ToolContext,
    card_identifier: Annotated[int, "The unique identifier of the GitHub project card to be updated."],  # noqa: E501
    card_note: Annotated[str | None, "The text note associated with the project card. It can include details or remarks."] = None,  # noqa: E501
    set_card_archived_status: Annotated[bool | None, "Specify true to archive the card or false to unarchive it."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/update-card'."]:
    """Update an existing project card on GitHub.

    Use this tool to modify the details of a project card in a GitHub project. Ideal for updating card information such as changing the note or moving it between columns."""  # noqa: E501
    request_data: Any = {"archived": set_card_archived_status, "note": card_note}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/cards/{card_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), card_id=card_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def move_project_card(
    context: ToolContext,
    card_identifier: Annotated[int, "The unique identifier for the card to be moved within the project."],  # noqa: E501
    card_position: Annotated[str, "Specify where to place the card within the column: 'top', 'bottom', or 'after:<card_id>'."],  # noqa: E501
    destination_column_id: Annotated[int | None, "The unique identifier of the column to which the card should be moved."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/move-card'."]:
    """Move a project card to a different position.

    Use this tool to move a card in a GitHub project to a different position within a column. This is helpful when reorganizing tasks or priorities in project management."""  # noqa: E501
    request_data: Any = {"column_id": destination_column_id, "position": card_position}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/cards/{card_id}/moves".format(git_server_url=context.get_secret("GIT_SERVER_URL"), card_id=card_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_project_column(
    context: ToolContext,
    project_column_id: Annotated[int, "The unique integer identifier of the GitHub project column to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/delete-column'."]:
    """Deletes a specific project column on GitHub.

    This tool deletes a specified column from a project on GitHub. It should be called when a user wants to remove a column from their project setup."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/{column_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=project_column_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_project_column(
    context: ToolContext,
    project_column_id: Annotated[int, "The unique identifier for the project column to retrieve."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/get-column'."]:
    """Retrieve details of a GitHub project column using its ID.

    Use this tool to obtain information about a specific column in a GitHub project. It should be called when you need to access the details of a project column using its column ID."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/{column_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=project_column_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_project_column(
    context: ToolContext,
    column_identifier: Annotated[int, "The unique identifier of the project column to update."],
    project_column_name: Annotated[str, "The new name for the project column on GitHub."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/update-column'."]:
    """Update an existing project column on GitHub.

    Use this tool to update details of an existing project column on GitHub. It's applicable when modifications to the project's column are needed, such as changing its name or settings."""  # noqa: E501
    request_data: Any = {"name": project_column_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/{column_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=column_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_project_cards(
    context: ToolContext,
    column_identifier: Annotated[int, "The unique identifier for the specified project column, used to list its cards."],  # noqa: E501
    filter_by_archived_state: Annotated[str | None, "Filters project cards by their archived state. Options are 'all', 'archived', or 'not_archived'."] = "not_archived",  # noqa: E501
    results_page_number: Annotated[int | None, "Specifies the page number of the project cards results to fetch. Useful for navigating through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of project cards returned per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/list-cards'."]:
    """Retrieve project cards for a specific column on GitHub projects.

    Use this tool to obtain a list of project cards within a specified column in GitHub projects. It is useful for managing tasks and tracking project progress by accessing detailed card information in a particular column."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/{column_id}/cards".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=column_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"archived_state": filter_by_archived_state, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_project_card(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    column_identifier: Annotated[int | None, "The unique identifier of the GitHub project column where the card will be added.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/create-card'."]:
    """Create a project card in a specified GitHub column.

    Use this tool to create a new project card in a specified column within a GitHub project. It helps organize tasks or notes in a project on GitHub.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBPROJECTCARD"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not column_identifier:
        missing_params.append(("column_identifier", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBPROJECTCARD"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBPROJECTCARD"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/projects/columns/{column_id}/cards".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=column_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBPROJECTCARD"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def move_github_project_column(
    context: ToolContext,
    column_position: Annotated[str, "Specifies where to move the project column. Use `first`, `last`, or `after:<column_id>` to position after a specific column."],  # noqa: E501
    project_column_id: Annotated[int, "The unique identifier of the column to be moved in the GitHub project."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/move-column'."]:
    """Move a column within a GitHub project board.

    Use this tool to change the position of a project column in a GitHub project board to better organize tasks or workflow."""  # noqa: E501
    request_data: Any = {"position": column_position}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/columns/{column_id}/moves".format(git_server_url=context.get_secret("GIT_SERVER_URL"), column_id=project_column_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_project_board(
    context: ToolContext,
    project_identifier: Annotated[int, "The unique identifier of the GitHub project board to be deleted."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/delete'."]:
    """Deletes a specified project board on GitHub.

    Use this tool to delete a GitHub project board with a given project ID. It is useful for managing and cleaning up project boards. If projects are disabled, a `404 Not Found` status will be returned."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_project_by_id(
    context: ToolContext,
    project_id: Annotated[int, "The unique identifier for the GitHub project to retrieve."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/get'."]:
    """Retrieve details of a GitHub project by its ID.

    Call this tool to obtain information about a specific GitHub project using its ID. It provides project details if accessible or returns an appropriate error status if there are issues like insufficient privileges or if projects are disabled."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_project_board(
    context: ToolContext,
    project_unique_identifier: Annotated[int, "The unique identifier of the GitHub project board to update."],  # noqa: E501
    is_private: Annotated[bool | None, "A boolean indicating if the project is private. Set to true for private (not visible to everyone) and false for public."] = None,  # noqa: E501
    organization_permission_level: Annotated[str | None, "Sets the baseline permission for all organization members on this project. Options are 'read', 'write', 'admin', or 'none'."] = None,  # noqa: E501
    project_description: Annotated[str | None, "A detailed description or content of the project board."] = None,  # noqa: E501
    project_name: Annotated[str | None, "The new name for the project board. Must be a string."] = None,  # noqa: E501
    project_state: Annotated[str | None, "Specify the state of the project; use 'open' or 'closed'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/update'."]:
    """Update a project board's information on GitHub.

    This tool updates the details of a specified project board on GitHub. It should be called when there is a need to modify the information of a project board. Note that a `404` error indicates projects are disabled, while a `401` or `410` error indicates insufficient privileges."""  # noqa: E501
    request_data: Any = {"body": project_description, "name": project_name, "organization_permission": organization_permission_level, "private": is_private, "state": project_state}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_unique_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_project_collaborators(
    context: ToolContext,
    project_id: Annotated[int, "The unique identifier for the GitHub organization project to retrieve collaborators for."],  # noqa: E501
    collaborator_affiliation_filter: Annotated[str | None, "Specifies how to filter collaborators: `outside`, `direct`, or `all`."] = "all",  # noqa: E501
    results_page_number: Annotated[int | None, "The page number to retrieve from the list of collaborators."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/list-collaborators'."]:
    """Retrieve collaborators for a GitHub organization project.

    Use this tool to obtain a list of all collaborators for a specified GitHub organization project, including outside collaborators and organization members with various access levels. Requires organization owner or project admin permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/collaborators".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"affiliation": collaborator_affiliation_filter, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_project_collaborator(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub user's handle to be removed as a collaborator."],
    project_unique_identifier: Annotated[int, "The unique numeric identifier of the GitHub organization project."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/remove-collaborator'."]:
    """Remove a collaborator from a GitHub organization project.

    Call this tool to remove a collaborator from a specified GitHub organization project. The user must be an organization owner or a project admin to perform this operation."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/collaborators/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_unique_identifier, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_project_collaborator(
    context: ToolContext,
    collaborator_username: Annotated[str, "The GitHub username of the collaborator to be added to the project."],  # noqa: E501
    project_id: Annotated[int, "The unique identifier of the project to which a collaborator is being added. This ID is required to specify the exact project for collaboration."],  # noqa: E501
    collaborator_permission_level: Annotated[str | None, "The permission level to assign to the collaborator. Options are: 'read', 'write', or 'admin'."] = "write",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/add-collaborator'."]:
    """Add a collaborator to an organization project.

    This tool adds a collaborator to a specified organization project and assigns a permission level. It requires the user to be an organization owner or a project admin."""  # noqa: E501
    request_data: Any = {"permission": collaborator_permission_level}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/collaborators/{username}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_id, username=collaborator_username),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_user_project_permission(
    context: ToolContext,
    github_user_handle: Annotated[str, "The GitHub username of the user whose project permission level is being queried."],  # noqa: E501
    project_id: Annotated[int, "The unique identifier of the GitHub project for which to fetch the user's permission."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/get-permission-for-user'."]:  # noqa: E501
    """Retrieve a user's permission level for an organization project.

    Use this tool to find out what level of access a specific user has to an organization project on GitHub. This is useful for organization owners or project admins to verify permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/collaborators/{username}/permission".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_id, username=github_user_handle),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_project_columns(
    context: ToolContext,
    project_identifier: Annotated[int, "The unique identifier of the GitHub project to list columns for."],  # noqa: E501
    page_number: Annotated[int | None, "The specific page number of the results to fetch from the GitHub project columns list."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/list-columns'."]:
    """Retrieve columns of a specific GitHub project.

    Use this tool to get a list of columns for a specified GitHub project. It's useful when you need to know the structure or organization of a GitHub project board."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/columns".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_project_column(
    context: ToolContext,
    column_name: Annotated[str, "The name of the column to be created in the GitHub project."],
    project_id: Annotated[int, "The unique integer identifier for the GitHub project where the column will be created."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'projects/create-column'."]:
    """Create a new column in a GitHub project.

    This tool creates a new column in a specified GitHub project board. Useful for organizing tasks or issues within project management workflows."""  # noqa: E501
    request_data: Any = {"name": column_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/projects/{project_id}/columns".format(git_server_url=context.get_secret("GIT_SERVER_URL"), project_id=project_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_rate_limit(
    context: ToolContext,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'rate-limit/get'."]:
    """Retrieve current GitHub API rate limit status.

    Use this tool to check the current API rate limit status on GitHub. It provides information on the remaining requests allowed in the current rate limit window without affecting your API usage."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/rate_limit".format(git_server_url=context.get_secret("GIT_SERVER_URL")),  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repo_required_workflows(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the organization. It is not case sensitive."],
    repository_name: Annotated[str, "The name of the GitHub repository you want to query. This is not case sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "The specific page of workflow results to retrieve. Use this for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-repo-required-workflows'."]:  # noqa: E501
    """Retrieve required workflows in a GitHub repository.

    Use this tool to list required workflows in a GitHub repository. This is accessible to anyone with read access. For private repositories, an access token with the `repo` scope is necessary. GitHub Apps need `actions:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{org}/{repo}/actions/required_workflows".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_repo_required_workflow(
    context: ToolContext,
    organization_name: Annotated[str, "The name of the GitHub organization. It is not case sensitive."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    required_workflow_id: Annotated[int, "The unique ID of the required workflow that has executed at least once in the repository."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-repo-required-workflow'."]:  # noqa: E501
    """Retrieve a specific required workflow from a GitHub repository.

    This tool fetches details of a required workflow from a specified GitHub repository. It can be used by anyone with read access to the repository. For private repositories, an access token with the 'repo' scope is needed. Useful for managing and auditing workflows in GitHub repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{org}/{repo}/actions/required_workflows/{required_workflow_id_for_repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), org=organization_name, repo=repository_name, required_workflow_id_for_repo=required_workflow_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_repository(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository you want to delete. This is not case sensitive."],  # noqa: E501
    repository_owner_name: Annotated[str, "The account owner of the repository; not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete'."]:
    """Deletes a specified GitHub repository.

    Use this tool to delete a GitHub repository when you have admin access and the necessary permissions. Ensure that OAuth is used with the 'delete_repo' scope if required. Be aware of organization-level restrictions that might prohibit deletion."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_name, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_repository_details(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "Specify the account owner of the repository. This name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get'."]:
    """Retrieve detailed information about a GitHub repository.

    This tool retrieves detailed information about a specified GitHub repository. It provides details such as the parent and source repositories if the repository is a fork. Admin or owner permissions might be required to access certain security details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_repository(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository to update, not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/update'."]:
    """Update repository details on GitHub.

    This tool updates the details of a specified GitHub repository. Note: To modify a repository's topics, a different endpoint should be used.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEGITHUBREPOSITORY"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEGITHUBREPOSITORY"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEGITHUBREPOSITORY"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEGITHUBREPOSITORY"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_repo_artifacts(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository to retrieve artifacts from. This input is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The username or organization name of the repository owner. This is not case sensitive."],  # noqa: E501
    filter_artifacts_by_name: Annotated[str | None, "Filters artifacts by providing an exact match for the artifact name. Use a string to specify the artifact name."] = None,  # noqa: E501
    result_page_number: Annotated[int | None, "Specify the page number of results to fetch for paginated artifact listings."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of artifacts to return per page, with a maximum limit of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-artifacts-for-repo'."]:
    """Retrieve all artifacts for a GitHub repository.

    Use this tool to get a list of all artifacts for a specific GitHub repository. It requires read access to the repository, and if private, an access token with the `repo` scope is needed. GitHub Apps require `actions:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/artifacts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": result_page_number, "name": filter_artifacts_by_name}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_artifact(
    context: ToolContext,
    artifact_unique_identifier: Annotated[int, "The unique identifier of the artifact to be deleted. Must be an integer value."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository where the artifact resides. The name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository, case-insensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-artifact'."]:
    """Deletes a specified GitHub artifact.

    Use this tool to delete an artifact from a GitHub workflow run. Authentication with an access token having the `repo` scope is required, or a GitHub App must have `actions:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/artifacts/{artifact_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, artifact_id=artifact_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_workflow_artifact(
    context: ToolContext,
    artifact_identifier: Annotated[int, "The unique identifier for the artifact to retrieve from a workflow run."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-artifact'."]:
    """Retrieve a specific artifact from a GitHub workflow run.

    Use this tool to access a specific artifact from a GitHub repository's workflow run. Requires appropriate permissions and access token for private repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/artifacts/{artifact_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, artifact_id=artifact_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_artifact_download_url(
    context: ToolContext,
    archive_format_zip: Annotated[str, "Specify the archive format as 'zip'. This is required for the download link."],  # noqa: E501
    artifact_id: Annotated[int, "The unique identifier of the artifact to be downloaded."],
    repository_name: Annotated[str, "The name of the GitHub repository (not case sensitive)."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/download-artifact'."]:
    """Retrieve a URL to download a GitHub artifact zip file.

    Use this tool to get a redirect URL for downloading an artifact zip file from a GitHub repository. The URL is valid for 1 minute and requires read access to the repository. For private repositories, an access token with the 'repo' scope is needed. This tool is useful for automating the retrieval of artifact files from GitHub Actions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/artifacts/{artifact_id}/{archive_format}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, artifact_id=artifact_id, archive_format=archive_format_zip),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_cache_usage(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Case insensitive name."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage'."]:
    """Fetch GitHub Actions cache usage for a repository.

    Use this tool to get the current cache usage for GitHub Actions in a specific repository. Cache data is updated approximately every 5 minutes. Requires read access to the repository and appropriate permissions for private repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/cache/usage".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fetch_github_actions_cache_policy(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-usage-policy'."]:  # noqa: E501
    """Retrieve the cache usage policy for GitHub Actions in a repository.

    This tool retrieves the GitHub Actions cache usage policy for a specified repository. It requires authentication with an access token that has the `repo` scope. GitHub Apps need the `actions:read` permission to access this endpoint."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/cache/usage-policy".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def configure_github_actions_cache(
    context: ToolContext,
    repository_cache_size_limit_gb: Annotated[int, "Specify the size limit for all GitHub Actions caches in the repository, in gigabytes."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. This field is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. This is the GitHub username or organization name and is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-actions-cache-usage-policy'."]:  # noqa: E501
    """Set GitHub Actions cache usage policy for a repository.

    Use this tool to configure the cache usage policy for GitHub Actions in a repository. Authentication with an access token with 'repo' scope is required, or GitHub Apps need 'actions:write' permission."""  # noqa: E501
    request_data: Any = {"repo_cache_size_limit_in_gb": repository_cache_size_limit_gb}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/cache/usage-policy".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_actions_cache_key(
    context: ToolContext,
    cache_key: Annotated[str, "The key used to identify and delete a specific GitHub Actions cache."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
    git_reference_for_cache_deletion: Annotated[str | None, "Specify the Git reference to restrict cache deletion. Use `refs/heads/<branch>` for branches or `refs/pull/<number>/merge` for pull requests."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-actions-cache-by-key'."]:  # noqa: E501
    """Delete GitHub Actions caches by key for a repository.

    Use this tool to delete one or more GitHub Actions caches for a repository by specifying a complete cache key. Optionally, provide a Git ref to restrict deletions. Requires an access token with `repo` scope or `actions:write` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/caches".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({"key": cache_key, "ref": git_reference_for_cache_deletion}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_actions_caches(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This value is not case sensitive."],  # noqa: E501
    cache_key_or_prefix: Annotated[str | None, "Explicit key or prefix to identify the cache. Use this to filter caches by specific keys or prefixes."] = None,  # noqa: E501
    git_reference: Annotated[str | None, "Specify the Git reference for the results to list. Use `refs/heads/<branch>` for branches or `refs/pull/<number>/merge` for pull requests."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
    sort_by_property: Annotated[str | None, "The property to sort results by. Options are 'created_at', 'last_accessed_at', or 'size_in_bytes'."] = "last_accessed_at",  # noqa: E501
    sort_direction: Annotated[str | None, "Specify 'asc' for ascending or 'desc' for descending order of results."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-actions-cache-list'."]:
    """Retrieve the list of GitHub Actions caches for a repository.

    This tool fetches the list of GitHub Actions caches for a specified repository. It requires authentication with an access token having the 'repo' scope, or GitHub Apps must have the 'actions:read' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/caches".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number, "ref": git_reference, "key": cache_key_or_prefix, "sort": sort_by_property, "direction": sort_direction}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_actions_cache(
    context: ToolContext,
    github_actions_cache_id: Annotated[int, "The unique identifier for the GitHub Actions cache to be deleted."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. Not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-actions-cache-by-id'."]:  # noqa: E501
    """Delete a GitHub Actions cache by ID for a repository.

    Use this tool to delete a specific GitHub Actions cache from a repository using its cache ID. Requires authentication with a token having `repo` scope. Useful for managing and cleaning up repository caches."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/caches/{cache_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, cache_id=github_actions_cache_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_workflow_job(
    context: ToolContext,
    job_identifier: Annotated[int, "Unique integer identifier of the workflow job to retrieve."],
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Case insensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-job-for-workflow-run'."]:  # noqa: E501
    """Retrieve a specific job from a GitHub workflow run.

    Use this tool to get details of a specific job from a GitHub workflow run. Requires read access to the repository. For private repositories, an access token with 'repo' scope is needed. GitHub Apps need 'actions:read' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/jobs/{job_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, job_id=job_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def download_github_workflow_job_logs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository (case insensitive)."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. Provide the name in a non-case sensitive format."],  # noqa: E501
    workflow_job_id: Annotated[int, "The unique identifier of the GitHub workflow job to download logs for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/download-job-logs-for-workflow-run'."]:  # noqa: E501
    """Retrieve a URL to download GitHub workflow job logs.

    Use this tool to get a link for downloading logs of a specific GitHub workflow job. The link is a plain text file URL and expires in 1 minute. Users must have read access to the repository. For private repositories, an access token with 'repo' scope is required, or 'actions:read' permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/jobs/{job_id}/logs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, job_id=workflow_job_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def github_rerun_workflow_job(
    context: ToolContext,
    job_identifier: Annotated[int, "The unique integer identifier of the job to be re-run. This is required to specify which job in the workflow needs to be restarted."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    enable_debug_logging: Annotated[bool | None, "Set to true to enable debug logging for the re-run."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/re-run-job-for-workflow-run'."]:  # noqa: E501
    """Re-run a job in a GitHub workflow.

    This tool re-runs a specific job and its dependent jobs within a GitHub workflow run. Use it when you need to restart a job that has failed or requires re-execution. Requires authentication with an access token having `repo` scope or `actions:write` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = {"enable_debug_logging": enable_debug_logging}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/jobs/{job_id}/rerun".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, job_id=job_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_oidc_subject_claim_template(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-custom-oidc-sub-claim-for-repo'."]:  # noqa: E501
    """Retrieve the OIDC subject claim customization template for a repository.

    Use this tool to get the customization template for an OpenID Connect (OIDC) subject claim for a specific GitHub repository. Requires authentication with a token having `repo` scope, or `organization_administration:read` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/oidc/customization/sub".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_oidc_subject_claim(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This value is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
    use_default_template: Annotated[bool, "Set to true to use the default template, which ignores `include_claim_keys`."],  # noqa: E501
    claim_keys_to_include: Annotated[list[str] | None, "Array of unique strings for claim keys, containing only alphanumeric characters and underscores."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-custom-oidc-sub-claim-for-repo'."]:  # noqa: E501
    """Customize OIDC subject claim for a GitHub repository.

    This tool is used to set the customization template and opt-in or opt-out flag for an OpenID Connect (OIDC) subject claim in a GitHub repository. It requires authentication with a token having `repo` scope or GitHub App permissions with `actions:write`."""  # noqa: E501
    request_data: Any = {"include_claim_keys": claim_keys_to_include, "use_default": use_default_template}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/oidc/customization/sub".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def fetch_github_actions_perms(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-github-actions-permissions-repository'."]:  # noqa: E501
    """Retrieve GitHub Actions permissions for a repository.

    Use this tool to get information on whether GitHub Actions is enabled for a repository and what actions are permitted. Requires authentication with a token with `repo` scope or a GitHub App with `administration` repository permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_repo_actions_permissions(
    context: ToolContext,
    enable_github_actions: Annotated[bool, "Boolean flag indicating if GitHub Actions should be enabled on the repository. `True` enables, `False` disables."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    actions_permission_policy: Annotated[str | None, "Defines the policy for actions allowed to run: 'all', 'local_only', or 'selected'."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-github-actions-permissions-repository'."]:  # noqa: E501
    """Sets GitHub Actions permissions for a repository.

    Use this tool to set the GitHub Actions permissions policy for a specific repository. Ideal when you need to enable or restrict actions in a repository under certain conditions. Ensure to authenticate with a token having the `repo` scope or a GitHub App with `administration` permissions."""  # noqa: E501
    request_data: Any = {"allowed_actions": actions_permission_policy, "enabled": enable_github_actions}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_workflow_access_level(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository (case insensitive)."],
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-workflow-access-to-repository'."]:  # noqa: E501
    """Determine external workflow access level for a repository.

    Utilize this tool to find out what level of access workflows outside of a specified internal or private repository have to its actions and reusable workflows. This tool is applicable only to internal and private repositories and requires appropriate authentication."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/access".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_workflow_access(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub username or organization name that owns the repository. This is not case sensitive."],  # noqa: E501
    workflow_access_level: Annotated[str, "Specifies access level for workflows outside the repository: 'none', 'user', or 'organization'."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-workflow-access-to-repository'."]:  # noqa: E501
    """Set the access level for workflows in a repository.

    Adjusts the access level for workflows and reusable workflows in internal or private GitHub repositories. Requires authentication with a token that has the `repo` scope or a GitHub App with `administration` permissions."""  # noqa: E501
    request_data: Any = {"access_level": workflow_access_level}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/access".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_allowed_actions_for_repo(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The username or organization name of the repository owner. Case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-allowed-actions-repository'."]:  # noqa: E501
    """Retrieve allowed GitHub Actions settings for a repository.

    Use this tool to fetch the settings for actions that are allowed in a GitHub repository. This is applicable when the repository's policy is set to `selected` for `allowed_actions`. Requires authentication with a token having `repo` scope or GitHub Apps with `administration` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_actions_allowed_in_repo(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-allowed-actions-repository'."]:  # noqa: E501
    """Set allowed GitHub Actions in a repository.

    Configure which GitHub Actions are permitted in a specific repository. Requires repository permission policy to be set to 'selected' and appropriate authentication. Cannot override organization-level settings.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETGITHUBACTIONSALLOWEDINREPO"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETGITHUBACTIONSALLOWEDINREPO"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETGITHUBACTIONSALLOWEDINREPO"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/selected-actions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETGITHUBACTIONSALLOWEDINREPO"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_default_github_actions_permissions(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository, which is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-github-actions-default-workflow-permissions-repository'."]:  # noqa: E501
    """Retrieve default GitHub Actions workflow permissions for a repository.

    This tool retrieves the default workflow permissions associated with the `GITHUB_TOKEN` in a specific GitHub repository. It also checks if GitHub Actions can submit approving pull request reviews for that repository. Authentication with an access token having the `repo` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def configure_github_token_permissions(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
    default_workflow_permissions: Annotated[str | None, "Specify the default permissions ('read' or 'write') for the GITHUB_TOKEN when running workflows."] = None,  # noqa: E501
    enable_pull_request_approval: Annotated[bool | None, "Set to true to allow GitHub Actions to approve pull requests. Enabling this may pose a security risk."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-github-actions-default-workflow-permissions-repository'."]:  # noqa: E501
    """Set default workflow permissions for a repository's GitHub Actions.

    This tool updates the default permissions granted to the `GITHUB_TOKEN` when workflows run in a GitHub repository. It also configures GitHub Actions' ability to submit approving pull request reviews. Authentication with a token having `repo` scope is required, or GitHub Apps must have `administration` permission."""  # noqa: E501
    request_data: Any = {"can_approve_pull_request_reviews": enable_pull_request_approval, "default_workflow_permissions": default_workflow_permissions}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/permissions/workflow".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_required_workflow_runs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. Case sensitivity is ignored."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    required_workflow_id: Annotated[int, "The ID of the required workflow that has run at least once in a repository."],  # noqa: E501
    branch_name: Annotated[str | None, "Specify the branch name to filter workflow runs associated with it. Use the name of the branch from the `push`."] = None,  # noqa: E501
    check_suite_identifier: Annotated[int | None, "Specify the Check Suite ID to filter workflow runs associated with this specific ID."] = None,  # noqa: E501
    exclude_pull_requests: Annotated[bool | None, "If true, pull requests are omitted from the response."] = False,  # noqa: E501
    results_page_number: Annotated[int | None, "The page number of the results to fetch, used for pagination of the workflow runs."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of workflow run results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
    sha_for_head_commit: Annotated[str | None, "Returns workflow runs associated with the specified head SHA (commit identifier)."] = None,  # noqa: E501
    trigger_event: Annotated[str | None, "Specify the event type that triggers the workflow run, such as `push`, `pull_request`, or `issue`."] = None,  # noqa: E501
    workflow_actor_username: Annotated[str | None, "Specify the username of the actor whose workflow runs you want to retrieve. Use the GitHub login for the user who initiated the push."] = None,  # noqa: E501
    workflow_run_status: Annotated[str | None, "Specify the workflow run status or conclusion to filter results. Options include 'completed', 'in_progress', 'success', etc."] = None,  # noqa: E501
    workflow_runs_created_date_range: Annotated[str | None, "Specify the date-time range to filter workflow runs based on their creation date. Use GitHub's date search syntax for formatting."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-required-workflow-runs'."]:  # noqa: E501
    """Retrieve all workflow runs for a required workflow.

    Use this tool to list all workflow runs associated with a specific required workflow in a GitHub repository. It is useful for monitoring and analyzing the execution of required workflows. Requires repository read access, and for private repositories, an access token with the `repo` scope is needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/required_workflows/{required_workflow_id_for_repo}/runs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, required_workflow_id_for_repo=required_workflow_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"actor": workflow_actor_username, "branch": branch_name, "event": trigger_event, "status": workflow_run_status, "per_page": results_per_page, "page": results_page_number, "created": workflow_runs_created_date_range, "exclude_pull_requests": exclude_pull_requests, "check_suite_id": check_suite_identifier, "head_sha": sha_for_head_commit}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_self_hosted_runners(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository (case-insensitive) for which to list self-hosted runners."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository, not case sensitive."],
    results_page_number: Annotated[int | None, "Page number of the results to fetch for listing self-hosted runners."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-self-hosted-runners-for-repo'."]:  # noqa: E501
    """Retrieve self-hosted runners for a GitHub repository.

    Use this tool to list all self-hosted runners configured in a specific GitHub repository. Authentication with an access token having the `repo` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_runner_apps_for_repo(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. Case insensitive."],
    repository_owner: Annotated[str, "The account owner of the repository on GitHub. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-runner-applications-for-repo'."]:  # noqa: E501
    """Retrieve runner application binaries for a GitHub repository.

    Use this tool to list binaries for GitHub runner applications that can be downloaded and run. Requires authentication with a token that has 'repo' scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/downloads".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_repo_registration_token(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Provide the GitHub username or organization name. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-registration-token-for-repo'."]:  # noqa: E501
    """Obtain a registration token for GitHub repository actions.

    Use this tool to get a token needed to configure a self-hosted runner for a GitHub repository. The token expires after one hour and requires authentication with a `repo`-scoped token."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/registration-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def generate_github_runner_remove_token(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This should match exactly as it appears on GitHub but is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-remove-token-for-repo'."]:  # noqa: E501
    """Generate a token to remove a GitHub self-hosted runner.

    This tool generates a token needed to remove a self-hosted runner from a GitHub repository. The token is valid for one hour. Ensure authentication using an access token with the 'repo' scope before calling this tool."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/remove-token".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_self_hosted_runner(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    runner_unique_identifier: Annotated[int, "Unique identifier of the self-hosted runner to be removed."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-self-hosted-runner-from-repo'."]:  # noqa: E501
    """Removes a self-hosted runner from a GitHub repository.

    Use this tool to forcibly remove a self-hosted runner from a GitHub repository when the machine no longer exists. Requires authentication with a token that has the `repo` scope."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def retrieve_runner_details(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This corresponds to the GitHub repository name and is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. This name is not case sensitive."],  # noqa: E501
    runner_identifier: Annotated[int, "Unique identifier of the self-hosted runner. Required to fetch specific runner details."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Retrieve information about a self-hosted runner in a GitHub repo.

"""
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_custom_labels_runner_repo(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case-sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Case insensitive."],
    runner_unique_identifier: Annotated[int, "Unique identifier of the self-hosted runner to remove custom labels from."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-all-custom-labels-from-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Remove all custom labels from a self-hosted runner in a repository.

    This tool removes all custom labels from a specified self-hosted runner configured in a repository, returning the remaining read-only labels. Authentication using an access token with the `repo` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_unique_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_runner_labels(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub account owner of the repository. Case insensitive."],  # noqa: E501
    self_hosted_runner_id: Annotated[int, "The unique integer identifier of the self-hosted runner in the repository."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-labels-for-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Retrieve all labels for a self-hosted runner in a GitHub repo.

    Use this tool to get all labels for a specific self-hosted runner configured in a GitHub repository. Authentication with a token that has the `repo` scope is required."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=self_hosted_runner_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def label_runner_for_repo(
    context: ToolContext,
    custom_labels_to_add: Annotated[list[str], "The names of the custom labels to add to the self-hosted runner. Provide as an array of strings."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Case insensitive."],
    runner_id: Annotated[int, "Unique identifier for the self-hosted runner in the repository."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/add-custom-labels-to-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Add custom labels to a repository's self-hosted runner.

    Use this tool to add custom labels to a self-hosted runner in a specified GitHub repository. Authentication with an access token with the `repo` scope is required."""  # noqa: E501
    request_data: Any = {"labels": custom_labels_to_add}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_runner_labels(
    context: ToolContext,
    custom_labels_for_runner: Annotated[list[str], "An array of custom labels to set for the self-hosted runner. Pass an empty array to clear all labels."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case-sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the GitHub repository. This name is not case sensitive."],  # noqa: E501
    runner_id: Annotated[int, "The unique integer identifier of the self-hosted runner to update labels for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/set-custom-labels-for-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Update custom labels for a self-hosted runner in a GitHub repo.

    This tool removes existing custom labels and sets new ones for a specific self-hosted runner in a GitHub repository. It requires authentication with an access token that has the 'repo' scope."""  # noqa: E501
    request_data: Any = {"labels": custom_labels_for_runner}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}/labels".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_id),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_runner_label(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This name is not case sensitive."],  # noqa: E501
    runner_custom_label_name: Annotated[str, "The name of the custom label on the self-hosted runner to be removed."],  # noqa: E501
    runner_unique_identifier: Annotated[int, "The unique ID number of the self-hosted runner to identify which runner to remove the label from."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/remove-custom-label-from-self-hosted-runner-for-repo'."]:  # noqa: E501
    """Remove a custom label from a self-hosted runner in a repository.

    Use this tool to remove a custom label from a self-hosted runner configured in a GitHub repository. It returns the remaining labels for the runner. Ensure to authenticate with a token with `repo` scope. A `404 Not Found` status is returned if the label is not found."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runners/{runner_id}/labels/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, runner_id=runner_unique_identifier, name=runner_custom_label_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_workflow_runs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository (case insensitive)."],
    repository_owner: Annotated[str, "The GitHub username or organization name that owns the repository. It is case-insensitive."],  # noqa: E501
    actor_username: Annotated[str | None, "Specify the username of the user whose workflow runs you want to retrieve. Use the login of the user who initiated the run."] = None,  # noqa: E501
    branch_name: Annotated[str | None, "Specify the branch name to filter workflow runs associated with that branch."] = None,  # noqa: E501
    check_suite_id: Annotated[int | None, "Returns workflow runs with the specified check suite ID."] = None,  # noqa: E501
    filter_by_head_sha: Annotated[str | None, "Only returns workflow runs associated with the specified commit SHA (head_sha)."] = None,  # noqa: E501
    omit_pull_requests: Annotated[bool | None, "If true, pull requests are excluded from the workflow runs response."] = False,  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch, used for paginating results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to return per page, with a maximum limit of 100."] = 30,  # noqa: E501
    triggering_event: Annotated[str | None, "Specify the event that triggers the workflow run, such as 'push', 'pull_request', or 'issue'."] = None,  # noqa: E501
    workflow_created_date_range: Annotated[str | None, "Specify a date-time range to filter workflow runs by creation date. Use GitHub's specific date syntax for format."] = None,  # noqa: E501
    workflow_run_status: Annotated[str | None, "Specifies the desired status or conclusion of the workflow runs to retrieve, such as 'success', 'in_progress', etc."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-workflow-runs-for-repo'."]:  # noqa: E501
    """Retrieve all workflow runs for a GitHub repository.

    This tool retrieves a list of all workflow runs for a specified GitHub repository. It can be used to track the status and history of workflows in the repository. Works with repositories you have read access to. For private repositories, an access token with the 'repo' scope is needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"actor": actor_username, "branch": branch_name, "event": triggering_event, "status": workflow_run_status, "per_page": results_per_page, "page": results_page_number, "created": workflow_created_date_range, "exclude_pull_requests": omit_pull_requests, "check_suite_id": check_suite_id, "head_sha": filter_by_head_sha}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_workflow_run(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run to be deleted. This should be an integer value."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-workflow-run'."]:
    """Delete a specific GitHub workflow run.

    Use this tool to delete a specific workflow run in a GitHub repository. Requires write access to the repository. For private repositories, an access token with `repo` scope is necessary. GitHub Apps need `actions:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_workflow_run(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive and is used to identify the repo for the workflow run."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository, not case sensitive."],
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run. Must be an integer."],  # noqa: E501
    omit_pull_requests: Annotated[bool | None, "If true, omits pull requests from the response, resulting in an empty array."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-workflow-run'."]:
    """Retrieve details of a specific GitHub workflow run.

    Use this tool to get information about a specific workflow run on GitHub. It requires read access to the repository; for private repositories, an access token with `repo` scope is needed. Ideal for tracking and reviewing the status and details of a workflow run."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"exclude_pull_requests": omit_pull_requests}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_actions_run_reviews(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. The input is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub Actions workflow run. This integer is required to fetch the reviews."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-reviews-for-run'."]:
    """Retrieve reviews for a GitHub Actions run.

    Call this tool to get reviews and approval details for a specific GitHub Actions run in a repository. It requires an access token with `repo` scope for private repositories or `actions:read` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/approvals".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_workflow_artifacts(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the specific workflow run to retrieve artifacts from."],  # noqa: E501
    page_number_to_fetch: Annotated[int | None, "The page number of the artifacts results to retrieve."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Number of results to return per page, up to a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-workflow-run-artifacts'."]:  # noqa: E501
    """Retrieve artifacts from a GitHub workflow run.

    Use this tool to list artifacts associated with a specific GitHub workflow run. It requires read access to the repository and appropriate permissions if the repository is private."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/artifacts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number_to_fetch}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_workflow_run_attempt(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository, not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This value is not case sensitive."],  # noqa: E501
    workflow_attempt_number: Annotated[int, "The numeric identifier for the specific attempt of the workflow run to be retrieved."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run. This integer specifies the exact run to retrieve information for."],  # noqa: E501
    omit_pull_requests: Annotated[bool | None, "Set to true to omit pull requests from the response."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-workflow-run-attempt'."]:  # noqa: E501
    """Retrieve details of a specific GitHub workflow run attempt.

    Use this tool to access detailed information about a specific workflow run attempt in a GitHub repository. Accessible with repository read permissions. Requires an access token for private repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/attempts/{attempt_number}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id, attempt_number=workflow_attempt_number),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"exclude_pull_requests": omit_pull_requests}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_workflow_run_jobs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository to fetch jobs from. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository, not case sensitive."],
    workflow_run_attempt_number: Annotated[int, "The attempt number of the specific workflow run to retrieve jobs for. This is typically used to distinguish between multiple attempts of the same run."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to list jobs for."],
    results_page_number: Annotated[int | None, "Specify the page number of the results to fetch, used for pagination."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specifies the number of job results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-jobs-for-workflow-run-attempt'."]:  # noqa: E501
    """Retrieve jobs from a specific GitHub workflow run attempt.

    This tool retrieves jobs for a specific workflow run attempt on GitHub. It is useful for users who need to monitor or analyze job details of a workflow run. Access to the repository is required, and permissions depend on whether the repository is public or private."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/attempts/{attempt_number}/jobs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id, attempt_number=workflow_run_attempt_number),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_workflow_run_attempt_logs_url(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive and identifies the repository for which to retrieve logs."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_attempt_number: Annotated[int, "The specific attempt number of the workflow run to fetch logs for."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run. Used to specify which workflow's logs to download."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/download-workflow-run-attempt-logs'."]:  # noqa: E501
    """Retrieve a URL to download workflow run attempt logs.

    Returns a redirect URL to download log files for a specific workflow run attempt. This URL expires in 1 minute and requires appropriate repository access permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/attempts/{attempt_number}/logs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id, attempt_number=workflow_attempt_number),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def cancel_github_workflow_run(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive and should match the repository where the workflow run is located."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run to be canceled. This should be an integer value."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/cancel-workflow-run'."]:
    """Cancels a GitHub workflow run using its ID.

    Use this tool to cancel an ongoing GitHub Actions workflow run. Authentication with an access token having `repo` scope is required. Useful for stopping workflows that may be executing erroneously or need interruption."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/cancel".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_workflow_jobs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. It is not case sensitive and identifies whose account owns the repository."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run to fetch jobs for."],  # noqa: E501
    job_filter_by_completion_time: Annotated[str | None, "Filter jobs by their `completed_at` timestamp. Use 'latest' for the most recent execution or 'all' for all executions."] = "latest",  # noqa: E501
    results_page_number: Annotated[int | None, "Page number for paginated results to fetch from the workflow jobs list."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of job results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-jobs-for-workflow-run'."]:  # noqa: E501
    """Fetches jobs for a specific GitHub workflow run.

    Use this tool to retrieve jobs associated with a GitHub workflow run. Requires read access to the repository. For private repositories, an access token with the `repo` scope is needed. GitHub Apps must have the `actions:read` permission. Parameters can be used to filter results."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/jobs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"filter": job_filter_by_completion_time, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_workflow_run_logs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case-sensitive."],
    repository_owner_name: Annotated[str, "The account owner of the repository. This name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to delete logs for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-workflow-run-logs'."]:  # noqa: E501
    """Deletes all logs for a specified workflow run on GitHub.

    Use this tool to delete all logs associated with a specific workflow run in a GitHub repository. Authentication with a token having `repo` scope is necessary, and GitHub Apps need `actions:write` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/logs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_name, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def download_workflow_run_logs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This field is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. This is case-insensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to download logs for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/download-workflow-run-logs'."]:  # noqa: E501
    """Get a redirect URL to download workflow run log files.

    This tool retrieves a URL for downloading an archive of log files from a specific GitHub repository workflow run. The link expires after one minute. Use this tool when you need access to the logs for analysis or debugging, ensuring you have the necessary read access or appropriate access token if the repository is private."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/logs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_pending_deployments_for_run(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to fetch pending deployments for."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-pending-deployments-for-run'."]:  # noqa: E501
    """Retrieve pending deployments for a GitHub workflow run.

    Use this tool to obtain all deployment environments for a specific GitHub workflow run that are awaiting protection rules to pass. Accessible to users with read access to the repository; requires `repo` scope for private repos or `actions:read` permission for GitHub Apps."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/pending_deployments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def approve_or_reject_pending_deployments(
    context: ToolContext,
    deployment_review_comment: Annotated[str, "A comment to accompany the approval or rejection of the deployment review."],  # noqa: E501
    deployment_review_state: Annotated[str, "Specify 'approved' to approve or 'rejected' to reject deployments to the environments."],  # noqa: E501
    environment_ids: Annotated[list[int], "List of environment IDs to approve or reject. Each ID must be an integer."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. This value is not case-sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to be approved or rejected."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/review-pending-deployments-for-run'."]:  # noqa: E501
    """Approve or reject pending deployments for a workflow run.

    This tool approves or rejects pending deployments that are awaiting a required reviewer's decision. It requires access to repository contents and deployments, and authentication using a token with `repo` scope."""  # noqa: E501
    request_data: Any = {"comment": deployment_review_comment, "environment_ids": environment_ids, "state": deployment_review_state}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/pending_deployments".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def rerun_github_workflow(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub account owner of the repository. The name is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the workflow run to be re-run."],
    enable_debug_logging: Annotated[bool | None, "Enable debug logging for the workflow re-run by setting to true."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/re-run-workflow'."]:
    """Initiates the rerun of a specific GitHub workflow.

    Use this tool to re-run a GitHub workflow run by providing its ID. Requires authentication with a token having 'repo' scope, or 'actions:write' permission for GitHub Apps."""  # noqa: E501
    request_data: Any = {"enable_debug_logging": enable_debug_logging}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/rerun".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def rerun_failed_github_workflow_jobs(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. This is not case sensitive."],  # noqa: E501
    workflow_run_id: Annotated[int, "The unique identifier of the GitHub workflow run to re-run failed jobs for."],  # noqa: E501
    enable_debug_logging: Annotated[bool | None, "Enable debug logging for the re-run of failed workflow jobs."] = False,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/re-run-workflow-failed-jobs'."]:  # noqa: E501
    """Re-run failed jobs in a GitHub workflow run.

    This tool re-runs all failed jobs and their dependents in a specified GitHub workflow run. Use this when a workflow has failed jobs you want to attempt again. Authentication with a token having `repo` scope is required."""  # noqa: E501
    request_data: Any = {"enable_debug_logging": enable_debug_logging}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/runs/{run_id}/rerun-failed-jobs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, run_id=workflow_run_id),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_repo_secrets(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. It is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specifies the page number of the secrets list to fetch."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-repo-secrets'."]:
    """Retrieve all repository secrets without values.

    This tool retrieves all the secrets available in a specified GitHub repository without revealing their encrypted values. Authentication is required using an access token with `repo` scope. GitHub Apps need `secrets` repository permission to access this endpoint."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/secrets".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_repo_public_key(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This value is not case sensitive. Provide the repository name for which you want to retrieve the public key."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. Enter a case-insensitive string specifying the owner's account name."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-repo-public-key'."]:
    """Retrieve the public key for encrypting repository secrets.

    Use this tool to get the public key necessary for encrypting secrets in a GitHub repository. This is a prerequisite step before creating or updating secrets. Ensure you have the necessary access rights: read access for public repositories, or an access token with the `repo` scope for private ones. GitHub Apps require `secrets` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/secrets/public-key".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_repo_secret(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    secret_name: Annotated[str, "The specific name of the secret to delete from the repository."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-repo-secret'."]:
    """Deletes a secret from a GitHub repository.

    Use this tool to remove a specific secret from a GitHub repository by providing the secret's name. Authentication with a `repo`-scoped access token is required. GitHub Apps need the `secrets` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_repository_secret_info(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
    secret_name: Annotated[str, "The name of the secret to retrieve metadata for."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-repo-secret'."]:
    """Retrieve metadata for a specific GitHub repository secret.

    This tool retrieves the metadata of a specified secret from a GitHub repository without exposing its encrypted value. It requires authentication with an access token that has the `repo` scope, or GitHub Apps with `secrets` repository permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_or_update_github_repo_secret(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    secret_name: Annotated[str, "The name of the secret to create or update in the repository."],
    encrypted_secret_value: Annotated[str | None, "The secret's value encrypted with LibSodium using a repository's public key."] = None,  # noqa: E501
    encryption_key_id: Annotated[str | None, "Provide the ID of the key used to encrypt the secret."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-or-update-repo-secret'."]:  # noqa: E501
    """Create or update a GitHub repository secret with an encrypted value.

    Use this tool to create or update a secret in a GitHub repository by providing an encrypted value. Requires authentication with a token having the `repo` scope or GitHub App permissions. Useful for managing sensitive information in repositories."""  # noqa: E501
    request_data: Any = {"encrypted_value": encrypted_secret_value, "key_id": encryption_key_id}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/secrets/{secret_name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, secret_name=secret_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_repo_variables(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository, case insensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    page_number: Annotated[int | None, "Page number to fetch in the list of repository variables. Use for paginating results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of results to return per page, with a maximum of 30."] = 10,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-repo-variables'."]:
    """Retrieve all variables for a specified GitHub repository.

    Use this tool to list all variables associated with a specific GitHub repository. Requires authentication with an access token having the 'repo' scope or a GitHub App with 'actions_variables:read' permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/variables".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_repo_variable(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. This value is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The GitHub account owner of the repository. Case insensitive."],  # noqa: E501
    variable_name: Annotated[str, "The name of the repository variable to create."],
    variable_value: Annotated[str, "The content or data for the repository variable."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/create-repo-variable'."]:
    """Create a variable for a GitHub repository to use in Actions workflows.

    Use this tool to create a repository variable that can be referenced in a GitHub Actions workflow. Ensure that you authenticate with a token having the 'repo' scope, or ensure GitHub Apps have 'actions_variables:write' permission."""  # noqa: E501
    request_data: Any = {"name": variable_name, "value": variable_value}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/variables".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_repo_variable(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository to delete the variable from. It is not case-sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    variable_name: Annotated[str, "The name of the variable to delete from the repository. It should match exactly as stored."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/delete-repo-variable'."]:
    """Delete a repository variable on GitHub using its name.

    This tool deletes a specified variable from a GitHub repository. It requires authentication with an access token that has the 'repo' scope. Suitable for use in managing repository configurations or cleanup tasks."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_repo_variable(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. The name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    variable_name: Annotated[str, "The name of the variable to retrieve from the repository. This is case-sensitive and must match the variable's exact name."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/get-repo-variable'."]:
    """Retrieve a specific variable from a GitHub repository.

    This tool retrieves information about a specific variable within a GitHub repository. It's useful when you need to access configuration or environment variables stored in a repository. Authentication with an access token having the `repo` scope is required, or GitHub Apps must have the `actions_variables:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, name=variable_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_github_repo_variable(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This name is not case sensitive."],  # noqa: E501
    env_variable_name: Annotated[str | None, "The name of the variable to update in the GitHub repository."] = None,  # noqa: E501
    repository_variable_value: Annotated[str | None, "The new value for the specified repository variable."] = None,  # noqa: E501
    variable_name: Annotated[str | None, "The name of the variable in the repository."] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/update-repo-variable'."]:
    """Update a variable in a GitHub repository for actions workflows.

    Use this tool to update a specific variable in a GitHub repository that is used in GitHub Actions workflows. Authentication is required with an access token having `repo` scope or for GitHub Apps with `actions_variables:write` permission."""  # noqa: E501
    request_data: Any = {"name": env_variable_name, "value": repository_variable_value}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/variables/{name}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, name=env_variable_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_repo_workflows(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. This field is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the GitHub repository. Case insensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "The page number for pagination, used to fetch specific sets of results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of workflow results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'actions/list-repo-workflows'."]:
    """Retrieve GitHub workflows in a repository.

    This tool retrieves the list of workflows for a specified GitHub repository. It requires read access to the repository. For private repositories, an access token with the `repo` scope is needed, and GitHub Apps must have `actions:read` permission."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/actions/workflows".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_issue_assignees(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository, case insensitive."],
    repository_owner: Annotated[str, "The owner of the repository. Input is not case sensitive."],
    results_page_number: Annotated[int | None, "Page number of the results to fetch. Use to paginate through results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results to display per page, with a maximum limit of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'issues/list-assignees'."]:
    """Retrieve available assignees for GitHub issues.

    This tool is used to list the available assignees for issues in a specified GitHub repository. Call this tool when you need to know who can be assigned to issues in a repository."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/assignees".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_user_assignment_permission(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. Input is not case sensitive."],  # noqa: E501
    user_assignee: Annotated[str, "The username of the GitHub user to check for issue assignment permissions."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'issues/check-user-can-be-assigned'."]:  # noqa: E501
    """Check if a user can be assigned to a GitHub issue.

    This tool checks if a specified user has permission to be assigned to an issue within a given GitHub repository. It should be called when you need to verify a user's eligibility to be an assignee for issues in a repository."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/assignees/{assignee}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, assignee=user_assignee),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_repo_autolinks(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the repository. It is case insensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This name is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number to retrieve results from when fetching autolinks for a repository."] = 1,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/list-autolinks'."]:
    """Retrieve autolinks for a specific GitHub repository.

    Use this tool to obtain information about all autolinks configured in a specified GitHub repository. It is only accessible to repository administrators."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/autolinks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_repository_autolink(
    context: ToolContext,
    autolink_key_prefix: Annotated[str, "The prefix that triggers link creation when found in issues, pull requests, or commits."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner_account: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    url_template_for_autolink: Annotated[str, "URL containing `<num>` for reference. It should match the characters based on `is_alphanumeric` value."],  # noqa: E501
    match_alphanumeric_characters: Annotated[bool | None, "Determines if the autolink reference matches alphanumeric characters. True includes A-Z, 0-9, '-', false matches only numeric characters."] = True,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/create-autolink'."]:
    """Create an autolink reference in a GitHub repository.

    Use this tool to create an autolink reference in a GitHub repository, provided you have admin access. It's helpful for linking external resources or references directly from issues and pull requests."""  # noqa: E501
    request_data: Any = {"is_alphanumeric": match_alphanumeric_characters, "key_prefix": autolink_key_prefix, "url_template": url_template_for_autolink}  # noqa: E501
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/autolinks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_account, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_github_repo_autolink(
    context: ToolContext,
    autolink_identifier: Annotated[int, "The unique integer identifier of the autolink to be deleted from the repository."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-autolink'."]:
    """Delete an autolink reference from a GitHub repository.

    Use this tool to delete a specific autolink reference from a GitHub repository by providing the repository owner, repository name, and autolink ID. This action is restricted to repository administrators."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/autolinks/{autolink_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, autolink_id=autolink_identifier),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_repository_autolink(
    context: ToolContext,
    autolink_id: Annotated[int, "The unique identifier of the autolink configured in the GitHub repository."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-autolink'."]:
    """Retrieve a specific GitHub repository autolink by ID.

    Use this tool to obtain details of a specific autolink reference configured in a GitHub repository. This is accessible only to repository administrators."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/autolinks/{autolink_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, autolink_id=autolink_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_repo_branches(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. This is not case-sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Page number of the results to fetch from the list of branches."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Specify the number of branch results per page (maximum 100)."] = 30,  # noqa: E501
    return_only_protected_branches: Annotated[bool | None, "Set to `true` to return only protected branches, `false` for only unprotected, or omit to return all branches."] = None,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/list-branches'."]:
    """Retrieve branches from a specific GitHub repository.

    Use this tool to get a list of all branches in a specified GitHub repository. It requires specifying the repository owner and name."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"protected": return_only_protected_branches, "per_page": results_per_page, "page": results_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_repo_branch(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the GitHub branch. Avoid using wildcard characters. For wildcard support, refer to the GitHub GraphQL API."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository (case-insensitive)."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-branch'."]:
    """Retrieve details of a specific branch from a GitHub repository.

    Use this tool to obtain information about a particular branch within a specified GitHub repository. Provide the owner, repository name, and branch name to get the branch details."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_branch_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch to remove protection from. Cannot contain wildcard characters."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-branch-protection'."]:
    """Remove protection from a specified GitHub branch.

    Use this tool to remove protection from a specific branch in a GitHub repository. Applicable for public repositories under GitHub Free and in both public and private repositories under GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server. Ideal for scenarios where branch updates are needed."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_branch_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The specific name of the GitHub branch to retrieve protection settings for. Must not contain wildcard characters."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-branch-protection'."]:
    """Retrieve protection settings for a GitHub branch.

    Use this tool to obtain protection settings for a specific branch in a GitHub repository. This is useful for understanding the rules and restrictions applied to the branch, including who can push changes or enforce specific merge rules."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_branch_protection(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository on GitHub. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository, which is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to update protection settings for. It cannot contain wildcard characters. For wildcard support, use the GraphQL API.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/update-branch-protection'."]:
    """Update GitHub repository branch protection settings.

    Use this tool to update the protection settings of a branch in a GitHub repository. It requires admin or owner permissions. Note that specifying new arrays for users and teams will replace their previous values, and there is a limit of 100 items for users, apps, and teams combined.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEBRANCHPROTECTION"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEBRANCHPROTECTION"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEBRANCHPROTECTION"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEBRANCHPROTECTION"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def delete_admin_branch_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The exact name of the branch for which admin enforcement will be removed. Wildcards are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive and should not include any special characters."],  # noqa: E501
    repository_owner: Annotated[str, "The GitHub account owner of the repository. This is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-admin-branch-protection'."]:  # noqa: E501
    """Remove admin enforcement on a protected branch.

    This tool removes admin enforcement from a protected branch on GitHub. It requires admin or owner permissions for the repository and that branch protection is enabled. Use when you need to disable admin restrictions on a branch."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/enforce_admins".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_admin_branch_protection_status(
    context: ToolContext,
    branch_name: Annotated[str, "The exact name of the branch. Wildcards are not allowed; use GraphQL API for wildcards."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. This name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-admin-branch-protection'."]:  # noqa: E501
    """Get admin branch protection status on GitHub.

    Use this tool to fetch the admin branch protection status for a specific branch within a GitHub repository. This is applicable to both public and private repositories across various GitHub plans."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/enforce_admins".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_admin_branch_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The exact name of the branch to set admin protection. Wildcards not supported."],  # noqa: E501
    repository_name: Annotated[str, "The case-insensitive name of the repository to set admin branch protection for."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/set-admin-branch-protection'."]:  # noqa: E501
    """Set admin branch protection in a GitHub repository.

    Use this tool to enforce admin branch protection on a specified branch in a GitHub repository. Requires admin or owner permissions, and branch protection must be enabled."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/enforce_admins".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_pull_request_review_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The specific name of the branch to remove pull request review protection from. Wildcard characters are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The case-insensitive name of the repository from which to remove pull request review protection."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. It's not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-pull-request-review-protection'."]:  # noqa: E501
    """Remove pull request review protection from a branch.

    Use this tool to remove the requirement for pull request reviews on a protected branch in a GitHub repository. This is useful for adjusting branch protection rules in repositories. Applies to various GitHub plans as outlined in GitHub documentation."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_pull_request_reviews".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_pull_request_review_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch. Cannot contain wildcard characters. For wildcard usage, refer to the GraphQL API."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository to retrieve pull request review protection details for. This name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. It is not case sensitive and identifies who owns the repository."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-pull-request-review-protection'."]:  # noqa: E501
    """Get pull request review protection details for a branch.

    Fetches details about the required pull request review protection for a specific branch in a GitHub repository. Useful for understanding branch protection rules in repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_pull_request_reviews".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_pull_request_review_protection(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner_name: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to update. It must not contain wildcard characters. For wildcard support, use the GraphQL API.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/update-pull-request-review-protection'."]:  # noqa: E501
    """Update pull request review protection settings for a branch.

    This tool allows updating the pull request review enforcement settings on a protected branch, requiring admin or owner permissions. It applies to branches in both public and private repositories with enabled branch protection settings.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEPULLREQUESTREVIEWPROTECTION"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner_name:
        missing_params.append(("repository_owner_name", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEPULLREQUESTREVIEWPROTECTION"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEPULLREQUESTREVIEWPROTECTION"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_pull_request_reviews".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_name, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEPULLREQUESTREVIEWPROTECTION"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def disable_commit_signature_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch where you want to disable commit signature protection. Wildcards are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository, case insensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. Not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-commit-signature-protection'."]:  # noqa: E501
    """Disable required signed commits on a branch.

    This tool disables required signed commits on a specified branch in a GitHub repository. It requires admin or owner permissions and is applicable when branch protection is enabled."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_signatures".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def check_branch_commit_signature_status(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch to check. It cannot contain wildcard characters."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. This is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The username of the repository owner. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-commit-signature-protection'."]:  # noqa: E501
    """Check if a branch requires signed commits for protection.

    Use this tool to determine if a branch in a GitHub repository requires signed commits. This is relevant for repositories with branch protection enabled. Appropriate permissions are needed to access this information."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_signatures".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def require_signed_commits_on_branch(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch for which to require signed commits. It cannot contain wildcard characters. For using wildcards, refer to the GitHub GraphQL API."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. Name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/create-commit-signature-protection'."]:  # noqa: E501
    """Enable signed commit requirement on a GitHub branch.

    This tool requires signed commits on a specified branch of a GitHub repository. It is used when you have admin or owner permissions and have enabled branch protection on the target branch."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_signatures".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_status_check_protection(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch from which you want to remove status check protection. Wildcard characters are not allowed. Use the GraphQL API for wildcard support."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub username or organization name that owns the repository, case insensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/remove-status-check-protection'."]:  # noqa: E501
    """Remove status check protection from a GitHub branch.

    This tool removes the required status check protection from a specified branch in a GitHub repository. It should be called when you want to lift status check requirements from a protected branch."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_branch_protection_status_checks(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch. It must not contain wildcard characters."],
    repository_name: Annotated[str, "The name of the GitHub repository to check. The name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-status-checks-protection'."]:  # noqa: E501
    """Retrieve status check protections for a GitHub branch.

    Use this tool to get information on the status check protections for a specific branch in a GitHub repository. It is useful for ensuring that a branch has the required checks before merging."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_branch_status_check_protection(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch for which to update status check protection. Wildcard characters are not allowed.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/update-status-check-protection'."]:  # noqa: E501
    """Update status check protection for a GitHub branch.

    Use this tool to update the required status checks for a protected branch in a GitHub repository. Requires admin or owner permissions.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATEBRANCHSTATUSCHECKPROTECTION"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATEBRANCHSTATUSCHECKPROTECTION"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATEBRANCHSTATUSCHECKPROTECTION"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATEBRANCHSTATUSCHECKPROTECTION"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_branch_status_check_contexts(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. Enter a GitHub username, which is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository, case insensitive, to target for status check context removal.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch from which to remove status check contexts. Cannot contain wildcard characters.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/remove-status-check-contexts'."]:  # noqa: E501
    """Remove status check contexts from a protected branch.

    Use this tool to delete status check contexts from a protected branch in a GitHub repository. This applies to various GitHub plans, allowing for flexible management of branch protections.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["REMOVEBRANCHSTATUSCHECKCONTEXTS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["REMOVEBRANCHSTATUSCHECKCONTEXTS"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["REMOVEBRANCHSTATUSCHECKCONTEXTS"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks/contexts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["REMOVEBRANCHSTATUSCHECKCONTEXTS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_status_check_contexts(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch. Cannot include wildcard characters."],
    repository_name: Annotated[str, "The name of the GitHub repository. This value is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the GitHub repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-all-status-check-contexts'."]:  # noqa: E501
    """Retrieve status check contexts for a protected GitHub branch.

    This tool fetches all status check contexts for a specific protected branch in a GitHub repository. Useful for understanding the checks required for branch protections. Suitable for repositories with branch protection enabled."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks/contexts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_status_check_contexts_to_branch(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository to add status check contexts to. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to add status check contexts to. Cannot contain wildcard characters. Use GraphQL API for wildcards.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/add-status-check-contexts'."]:
    """Add status check contexts to a protected branch.

    This tool allows users to add status check contexts to a specified protected branch in a GitHub repository. It should be called when managing branch protection rules and ensuring specific contexts are required for status checks.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["ADDSTATUSCHECKCONTEXTSTOBRANCH"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["ADDSTATUSCHECKCONTEXTSTOBRANCH"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["ADDSTATUSCHECKCONTEXTSTOBRANCH"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks/contexts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["ADDSTATUSCHECKCONTEXTSTOBRANCH"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_branch_status_check_contexts(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch. It cannot contain wildcard characters. Use the GraphQL API for wildcard support.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/set-status-check-contexts'."]:
    """Set status check contexts for a protected branch.

    Utilize this tool to configure the required status check contexts for a protected branch in a GitHub repository. This tool is relevant for managing branch protections in public repositories with GitHub Free, and both public and private repositories with GitHub Pro, Team, Enterprise Cloud, and Server.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETBRANCHSTATUSCHECKCONTEXTS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETBRANCHSTATUSCHECKCONTEXTS"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETBRANCHSTATUSCHECKCONTEXTS"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/required_status_checks/contexts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETBRANCHSTATUSCHECKCONTEXTS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_branch_access_restriction(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch to remove access restrictions from. Cannot contain wildcard characters."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub account owner of the repository. The value is not case-sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/delete-access-restrictions'."]:  # noqa: E501
    """Remove access restrictions from a GitHub branch.

    This tool is used to remove restrictions on who can push to a protected branch in a GitHub repository. It's applicable for both public and private repositories under various GitHub plans. Call this tool when you need to allow all collaborators to push to a specific branch."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_branch_access_restrictions(
    context: ToolContext,
    branch_name: Annotated[str, "The exact name of the branch to retrieve access information for. Wildcard characters are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-access-restrictions'."]:
    """Retrieve access information for a protected branch.

    Use this tool to list users, apps, and teams with access to a specified protected branch within an organization-owned repository on GitHub. It provides details about who has permissions to interact with the branch."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_github_app_branch_access(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner_account: Annotated[str | None, "The account owner of the repository, case insensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository, not case-sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to remove app access from. Wildcards are not allowed.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/remove-app-access-restrictions'."]:  # noqa: E501
    """Remove an app's access to a protected GitHub branch.

    Use this tool to remove the ability of a GitHub app to push to a protected branch. Applicable for branches in public and private repositories under various GitHub plans. Only apps with 'write' access to 'contents' permission can be removed.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["REMOVEGITHUBAPPBRANCHACCESS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner_account:
        missing_params.append(("repository_owner_account", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["REMOVEGITHUBAPPBRANCHACCESS"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["REMOVEGITHUBAPPBRANCHACCESS"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/apps".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_account, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["REMOVEGITHUBAPPBRANCHACCESS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_apps_with_branch_access(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch. Wildcard characters are not allowed; use exact names only."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository, not case sensitive."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-apps-with-access-to-protected-branch'."]:  # noqa: E501
    """Retrieve GitHub Apps with access to a protected branch.

    This tool lists the GitHub Apps that have push access to a specific protected branch. It should be called when you need to identify which installed GitHub Apps are authorized to interact with a branch under protection."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/apps".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_app_access_restrictions(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The repository account owner. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to grant push access. Cannot contain wildcard characters.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/add-app-access-restrictions'."]:  # noqa: E501
    """Grant specified apps push access to a protected branch.

    Use this tool to grant specific GitHub Apps push access to protected branches in repositories. Apps must have 'write' access to 'contents' to be authorized.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["ADDAPPACCESSRESTRICTIONS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["ADDAPPACCESSRESTRICTIONS"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["ADDAPPACCESSRESTRICTIONS"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/apps".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["ADDAPPACCESSRESTRICTIONS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_branch_app_access_restrictions(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The exact name of the branch to update app access restrictions for. Wildcard characters are not allowed.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/set-app-access-restrictions'."]:  # noqa: E501
    """Replace apps with push access on a protected branch.

    Use this tool to specify which GitHub Apps have push access to a protected branch, replacing any previous app list. It is applicable to public repositories with various GitHub plans and private repositories with higher-tier plans. Only GitHub Apps with write access to the contents permission can be authorized.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETBRANCHAPPACCESSRESTRICTIONS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETBRANCHAPPACCESSRESTRICTIONS"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETBRANCHAPPACCESSRESTRICTIONS"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/apps".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETBRANCHAPPACCESSRESTRICTIONS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_team_access_from_branch(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The exact name of the branch from which to remove team access. Wildcard characters are not allowed. For wildcard usage, employ the GraphQL API.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/remove-team-access-restrictions'."]:  # noqa: E501
    """Remove a team's push access to a protected GitHub branch.

    This tool removes the ability of a specified team to push to a protected branch in a GitHub repository. It should be called when you need to revoke push access for a team, including child teams, on a specific branch.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["REMOVETEAMACCESSFROMBRANCH"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["REMOVETEAMACCESSFROMBRANCH"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["REMOVETEAMACCESSFROMBRANCH"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["REMOVETEAMACCESSFROMBRANCH"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_teams_with_push_access_to_branch(
    context: ToolContext,
    branch_name: Annotated[str, "The name of the branch to retrieve teams with push access. Wildcard characters are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The GitHub username or organization name that owns the repository. It is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-teams-with-access-to-protected-branch'."]:  # noqa: E501
    """Retrieve teams with push access to a protected branch.

    Call this tool to get a list of teams that have push access to a specified protected branch in a GitHub repository. Useful for managing and reviewing team permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def add_team_access_to_branch(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "Specify the branch name to grant team access. Wildcard characters are not allowed.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/add-team-access-restrictions'."]:  # noqa: E501
    """Grant push access to teams for a specific branch.

    Use this tool to grant specified teams push access to a protected branch within a repository. Applicable to repositories on GitHub Free, Pro, Team, and Enterprise plans.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["ADDTEAMACCESSTOBRANCH"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["ADDTEAMACCESSTOBRANCH"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["ADDTEAMACCESSTOBRANCH"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["ADDTEAMACCESSTOBRANCH"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_github_branch_team_access(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch. It cannot contain wildcard characters.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/set-team-access-restrictions'."]:  # noqa: E501
    """Update the team access restrictions on a GitHub branch.

    This tool replaces the current list of teams with push access to a specified branch in a GitHub repository. It removes all previous teams and sets the new list, including any child teams, for team access control.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETGITHUBBRANCHTEAMACCESS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETGITHUBBRANCHTEAMACCESS"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETGITHUBBRANCHTEAMACCESS"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/teams".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETGITHUBBRANCHTEAMACCESS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def remove_user_access_from_branch(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to remove user access from. It must not contain wildcard characters. For using wildcards, refer to the GitHub GraphQL API.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/remove-user-access-restrictions'."]:  # noqa: E501
    """Remove users' push access from a GitHub branch.

    This tool is used to remove the ability of specified users to push to a protected branch in a GitHub repository. It supports repositories with GitHub Free, Pro, Team, Enterprise Cloud, and Server plans. Use this tool to manage access restrictions on your protected branches.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["REMOVEUSERACCESSFROMBRANCH"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["REMOVEUSERACCESSFROMBRANCH"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["REMOVEUSERACCESSFROMBRANCH"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/users".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="DELETE",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["REMOVEUSERACCESSFROMBRANCH"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_users_with_branch_access(
    context: ToolContext,
    branch_name: Annotated[str, "The exact name of the branch to check for push access. Wildcard characters are not allowed."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/get-users-with-access-to-protected-branch'."]:  # noqa: E501
    """Retrieve users with push access to a protected branch on GitHub.

    This tool should be called to obtain a list of users who have push access to a specific protected branch in a GitHub repository. It is applicable to both public and private repositories under various GitHub plans, including Free, Pro, Team, and Enterprise. Use this when you need to manage or review user permissions for branch access."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/users".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def grant_push_access_github_branch(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch to grant push access. Cannot contain wildcard characters.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/add-user-access-restrictions'."]:  # noqa: E501
    """Grant push access to specified users for a GitHub branch.

    This tool is used to grant push access to specific users for a protected branch in a GitHub repository. It is applicable in scenarios involving public repositories with GitHub Free and GitHub Free for organizations, as well as public and private repositories with GitHub Pro, GitHub Team, GitHub Enterprise Cloud, and GitHub Enterprise Server.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["GRANTPUSHACCESSGITHUBBRANCH"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["GRANTPUSHACCESSGITHUBBRANCH"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["GRANTPUSHACCESSGITHUBBRANCH"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/users".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["GRANTPUSHACCESSGITHUBBRANCH"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_branch_user_access_restrictions(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. Note that it is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    branch_name: Annotated[str | None, "The name of the branch for which to set user access restrictions. Cannot contain wildcard characters.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/set-user-access-restrictions'."]:  # noqa: E501
    """Set user access restrictions for a GitHub branch.

    This tool replaces the list of users with push access to a specified GitHub branch, removing previous users and granting access to the new list. Useful for managing who can perform actions on protected branches.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETBRANCHUSERACCESSRESTRICTIONS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not branch_name:
        missing_params.append(("branch_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETBRANCHUSERACCESSRESTRICTIONS"] + "\n```"  # noqa: E501
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETBRANCHUSERACCESSRESTRICTIONS"] + "\n```"  # noqa: E501
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/protection/restrictions/users".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=branch_name),  # noqa: E501  # noqa: UP032
        method="PUT",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETBRANCHUSERACCESSRESTRICTIONS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def rename_github_branch(
    context: ToolContext,
    current_branch_name: Annotated[str, "The current name of the branch to be renamed. Cannot include wildcard characters."],  # noqa: E501
    new_branch_name: Annotated[str, "The new name for the branch. Ensure it doesn’t contain wildcard characters."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository. It is case insensitive."],
    repository_owner: Annotated[str, "The GitHub account owner of the repository, not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'repos/rename-branch'."]:
    """Rename a branch in a GitHub repository.

    Use this tool to rename a branch in a specific GitHub repository. Requires appropriate permissions based on whether the branch is default or not. Note that pushing to the old branch name is prohibited during the rename process."""  # noqa: E501
    request_data: Any = {"new_name": new_branch_name}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/branches/{branch}/rename".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, branch=current_branch_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_check_run(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. The name is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/create'."]:
    """Create a new check run for a GitHub repository commit.

    This tool creates a new check run associated with a specific commit in a GitHub repository. It requires the 'checks:write' permission in the GitHub App. Note that this tool is sensitive to the branch it's applied to, as it only processes pushes in the repository where it's created, not in forked repositories.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["CREATEGITHUBCHECKRUN"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["CREATEGITHUBCHECKRUN"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["CREATEGITHUBCHECKRUN"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/check-runs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["CREATEGITHUBCHECKRUN"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_check_run(
    context: ToolContext,
    check_run_identifier: Annotated[int, "The unique identifier of the GitHub check run to retrieve."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository, not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/get'."]:
    """Retrieve a specific GitHub check run by its ID.

    Use this tool to get details of a single check run from a GitHub repository using its ID. This is applicable for both private and public repositories, with necessary permissions."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-runs/{check_run_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_run_id=check_run_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def update_check_run_status(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. This is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the repository to update the check run. It is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    check_run_identifier: Annotated[int | None, "The unique identifier of the check run to update. This should be an integer value.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/update'."]:
    """Update a check run for a specific commit in a repository.

    This tool updates a check run for a specific commit within a repository. It is useful when you need to modify the status or details of check runs created by a GitHub App. Ensure the app has the 'checks:write' permission to perform this action.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["UPDATECHECKRUNSTATUS"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))
    if not check_run_identifier:
        missing_params.append(("check_run_identifier", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["UPDATECHECKRUNSTATUS"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["UPDATECHECKRUNSTATUS"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/check-runs/{check_run_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_run_id=check_run_identifier),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["UPDATECHECKRUNSTATUS"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_check_run_annotations(
    context: ToolContext,
    check_run_identifier: Annotated[int, "The ID of the check run to retrieve annotations for."],
    repository_name: Annotated[str, "The name of the GitHub repository (case insensitive)."],
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    results_page_number: Annotated[int | None, "Specify the page number to fetch annotations from. Used for paginating results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page to return, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/list-annotations'."]:
    """Retrieve annotations for a GitHub check run.

    Use this tool to fetch a list of annotations associated with a specific check run in a GitHub repository. Requires appropriate permissions on private repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-runs/{check_run_id}/annotations".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_run_id=check_run_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"per_page": results_per_page, "page": results_page_number}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def trigger_github_check_rerequest(
    context: ToolContext,
    check_run_identifier: Annotated[int, "The unique identifier for the GitHub check run to be rerequested."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. The name is not case sensitive."],
    repository_owner_account: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/rerequest-run'."]:
    """Triggers a rerequest for an existing GitHub check run.

    This tool triggers GitHub to rerequest an existing check run without needing to push new code. Ideal for resetting the status of check runs to 'queued' and clearing the conclusion. It requires the `checks:read` permission for private repositories or pull access for public ones."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-runs/{check_run_id}/rerequest".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_account, repo=repository_name, check_run_id=check_run_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def create_github_check_suite(
    context: ToolContext,
    head_commit_sha: Annotated[str, "The SHA of the head commit for the check suite. Ensure it's a valid commit SHA."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository where the check suite will be created. It is not case sensitive."],  # noqa: E501
    repository_owner_account: Annotated[str, "Specify the account owner of the repository. This name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/create-suite'."]:
    """Manually create a check suite on GitHub.

    Use this tool to manually create a check suite in a GitHub repository when automatic creation is disabled. Ensure your GitHub App has the `checks:write` permission before using this endpoint."""  # noqa: E501
    request_data: Any = {"head_sha": head_commit_sha}
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-suites".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner_account, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def set_check_suite_preferences(
    context: ToolContext,
    mode: Annotated[
        ToolMode,
        "Operation mode: 'get_request_schema' returns the OpenAPI spec "
        "for the request body, 'execute' performs the actual operation",
    ],
    repository_owner: Annotated[str | None, "The account owner of the repository. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    repository_name: Annotated[str | None, "The name of the GitHub repository. This value is not case sensitive.  Required when mode is 'execute', ignored when mode is 'get_request_schema'."] = None,  # noqa: E501

    request_body: Annotated[
        str | None,
        "Stringified JSON representing the request body. Required when "
        "mode is 'execute', ignored when mode is 'get_request_schema'",
    ] = None,
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/set-suites-preferences'."]:
    """Set preferences for check suite creation in a repository.

    Changes the default automatic flow for creating check suites in a GitHub repository. Use this tool to enable or disable the automatic creation of check suites when code is pushed. Requires admin permissions in the repository.

    Note: Understanding the request schema is necessary to properly create
    the stringified JSON input object for execution.\n\nThis operation also requires path parameters.

    Modes:
    - GET_REQUEST_SCHEMA: Returns the schema. Only call if you don't
      already have it. Do NOT call repeatedly if you already received
      the schema.
    - EXECUTE: Performs the operation with the provided request body
      JSON.\n      Note: You must also provide the required path parameters when executing.

    If you need the schema, call with mode='get_request_schema' ONCE, then execute.
    """  # noqa: E501
    if mode == ToolMode.GET_REQUEST_SCHEMA:
        return {
            "request_body_schema": REQUEST_BODY_SCHEMAS["SETCHECKSUITEPREFERENCES"],
            "instructions": (
                "Use the request_body_schema to construct a valid JSON object. "
                "Once you have populated the object following the schema "
                "structure and requirements, call this tool again with "
                "mode='execute' and the stringified JSON as the "
                "request_body parameter along with the required path parameters. "
                "Do NOT call the schema mode again - you already have "
                "the schema now."
            ),
        }

    # Mode is EXECUTE - validate parameters
    # Validate required parameters
    missing_params = []
    if not repository_owner:
        missing_params.append(("repository_owner", "path"))
    if not repository_name:
        missing_params.append(("repository_name", "path"))

    if missing_params:
        param_names = [p[0] for p in missing_params]
        param_details = ", ".join(
            [f"{p[0]} ({p[1]})" for p in missing_params]
        )
        raise RetryableToolError(
            message=f"Missing required parameters: {param_names}",
            developer_message=(
                f"Required parameters validation failed: {param_details}"
            ),
            additional_prompt_content=(
                f"The following required parameters are missing: "
                f"{param_details}. Please call this tool again with all "
                "required parameters."
            ),
        )


    # Validate request body is provided (not None or empty string)
    # Note: Empty objects like {} are allowed - schema validation will check if valid
    if request_body is None or request_body.strip() == "":
        raise RetryableToolError(
            message="Request body is required when mode is 'execute'",
            developer_message="The request_body parameter was null or empty string",
            additional_prompt_content=(
                "The request body is required to perform this operation. "
                "Use the schema below to construct a valid JSON object, "
                "then call this tool again in execute mode with the "
                "stringified JSON as the request_body parameter.\n\n"
                "Schema:\n\n```json\n" + REQUEST_BODY_SCHEMAS["SETCHECKSUITEPREFERENCES"] + "\n```"
            ),
        )

    # Parse JSON
    try:
        request_data = json.loads(request_body)
    except json.JSONDecodeError as e:
        raise RetryableToolError(
            message=f"Invalid JSON in request body: {e!s}",
            developer_message=f"JSON parsing failed: {e!s}",
            additional_prompt_content=(
                f"The request body contains invalid JSON. Error: {e!s}\n\n"
                "Please provide a valid JSON string that matches the schema "
                "below, then call this tool again in execute mode.\n\n"
                "Schema:\n\n```json\n"+ REQUEST_BODY_SCHEMAS["SETCHECKSUITEPREFERENCES"] + "\n```"
            ),
        ) from e

    response = await make_request_with_schema_validation(
        url="{git_server_url}/repos/{owner}/{repo}/check-suites/preferences".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="PATCH",
        request_data=request_data,
        schema=REQUEST_BODY_SCHEMAS["SETCHECKSUITEPREFERENCES"],
        params=remove_none_values({}),
        headers=remove_none_values({"Content-Type": "application/json", "Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def get_github_check_suite(
    context: ToolContext,
    check_suite_id: Annotated[int, "The unique identifier for the GitHub check suite to retrieve."],
    repository_name: Annotated[str, "The name of the repository. This is not case sensitive."],
    repository_owner: Annotated[str, "The account owner of the GitHub repository. The name is not case sensitive."],  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/get-suite'."]:
    """Retrieve a GitHub check suite by ID.

    Use this tool to get information about a specific check suite in a GitHub repository using its ID. Suitable for users with appropriate permissions on private or public repositories."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-suites/{check_suite_id}".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_suite_id=check_suite_id),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_github_check_runs_for_suite(
    context: ToolContext,
    check_suite_identifier: Annotated[int, "The unique identifier of the check suite to list its check runs."],  # noqa: E501
    repository_name: Annotated[str, "The name of the GitHub repository to query. It is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner's username of the repository. Case insensitive."],  # noqa: E501
    check_run_name: Annotated[str | None, "Returns check runs with the specified name."] = None,
    check_run_status: Annotated[str | None, "Specify the status of the check runs to be returned. Options: 'queued', 'in_progress', 'completed'."] = None,  # noqa: E501
    filter_by_completion_time: Annotated[str | None, "Filters check runs by their `completed_at` timestamp. Use 'latest' for the most recent runs or 'all' for all runs."] = "latest",  # noqa: E501
    result_page_number: Annotated[int | None, "Specifies which page of the results to fetch, for paginated data."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "Sets the number of results to return per page, with a maximum of 100."] = 30,  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/list-for-suite'."]:
    """List check runs for a GitHub check suite using its ID.

    This tool retrieves a list of check runs for a specified check suite in a GitHub repository. It requires appropriate permissions or scopes depending on the repository type."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-suites/{check_suite_id}/check-runs".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_suite_id=check_suite_identifier),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"check_name": check_run_name, "status": check_run_status, "filter": filter_by_completion_time, "per_page": results_per_page, "page": result_page_number}),  # noqa: E501
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def rerequest_github_check_suite(
    context: ToolContext,
    check_suite_identifier: Annotated[int, "The unique identifier of the GitHub check suite to be rerequested."],  # noqa: E501
    repository_name: Annotated[str, "The name of the repository. It is not case sensitive. Specify the repository whose check suite you want to rerequest."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository (case insensitive)."],
) -> Annotated[dict[str, Any], "Response from the API endpoint 'checks/rerequest-suite'."]:
    """Rerequest a check suite on GitHub without code changes.

    Use this tool to trigger GitHub to rerequest an existing check suite, resetting its status to 'queued' and clearing its conclusion. This is useful for re-running checks without pushing new code. Ensure your GitHub App has the necessary permissions to perform this action."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/check-suites/{check_suite_id}/rerequest".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name, check_suite_id=check_suite_identifier),  # noqa: E501  # noqa: UP032
        method="POST",
        params=remove_none_values({}),
        headers=remove_none_values({"Authorization": "Bearer {authorization}".format(authorization=context.get_auth_token_or_empty())}),  # noqa: E501  # noqa: UP032
        content=content,
    )
    try:
        return {"response_json": response.json()}
    except Exception:
        return {"response_text": response.text}


@tool(requires_auth=GitHub(), requires_secrets=["GIT_SERVER_URL"])
async def list_code_scanning_alerts(
    context: ToolContext,
    repository_name: Annotated[str, "The name of the GitHub repository for which to list code scanning alerts. This name is not case sensitive."],  # noqa: E501
    repository_owner: Annotated[str, "The account owner of the repository. The name is not case sensitive."],  # noqa: E501
    alert_state_filter: Annotated[str | None, "Filter code scanning alerts by state. Valid options are 'open', 'closed', 'dismissed', and 'fixed'."] = None,  # noqa: E501
    code_scanning_tool_guid: Annotated[str | None, "The GUID of a code scanning tool to filter alerts by this tool only. This can't be used with 'tool_name'."] = None,  # noqa: E501
    code_scanning_tool_name: Annotated[str | None, "Specify the name of the code scanning tool to filter alerts by this tool only. Use either `tool_name` or `tool_guid`, but not both."] = None,  # noqa: E501
    filter_by_severity: Annotated[str | None, "Specify the severity of code scanning alerts to filter, using values like 'critical', 'high', 'medium', 'low', 'warning', 'note', or 'error'."] = None,  # noqa: E501
    git_reference_for_scan_results: Annotated[str | None, "The Git reference for listing results. Use `refs/heads/<branch>` or `<branch>` for branches, `refs/pull/<number>/merge` for pull requests."] = None,  # noqa: E501
    results_page_number: Annotated[int | None, "Page number to fetch results from. Use this to navigate through paginated results."] = 1,  # noqa: E501
    results_per_page: Annotated[int | None, "The number of results per page, up to a maximum of 100."] = 30,  # noqa: E501
    sort_by_property: Annotated[str | None, "Property to sort the results by, either 'created' or 'updated'."] = "created",  # noqa: E501
    sort_direction: Annotated[str | None, "The direction to sort the results. Choose 'asc' for ascending or 'desc' for descending order."] = "desc",  # noqa: E501
) -> Annotated[dict[str, Any], "Response from the API endpoint 'code-scanning/list-alerts-for-repo'."]:  # noqa: E501
    """Retrieve code scanning alerts for a repository.

    Retrieve a list of code scanning alerts for a specific GitHub repository, using an access token with the necessary permissions. Useful for monitoring code security issues in a repository."""  # noqa: E501
    request_data: Any = None
    if isinstance(request_data, dict):
        request_data = remove_none_values(request_data)
    content = json.dumps(request_data) if request_data else None
    response = await make_request(
        url="{git_server_url}/repos/{owner}/{repo}/code-scanning/alerts".format(git_server_url=context.get_secret("GIT_SERVER_URL"), owner=repository_owner, repo=repository_name),  # noqa: E501  # noqa: UP032
        method="GET",
        params=remove_none_values({"tool_name": code_scanning_tool_name, "tool_guid": code_scanning_