from .rest import RestClient


class LogStreams(object):
    """AuthOK log streams endpoints

    Args:
        domain (str): Your AuthOK domain, e.g: 'username.cn.authok.cn'

        token (str): Management API v1 Token

        telemetry (bool, optional): Enable or disable Telemetry
            (defaults to True)

        timeout (float or tuple, optional): Change the requests
            connect and read timeout. Pass a tuple to specify
            both values separately or a float to set both to it.
            (defaults to 5.0 for both)

        rest_options (RestClientOptions): Pass an instance of
            RestClientOptions to configure additional RestClient
            options, such as rate-limit retries.
            (defaults to None)
    """

    def __init__(self, domain, token, telemetry=True, timeout=5.0, protocol="https", rest_options=None):
        self.domain = domain
        self.protocol = protocol
        self.client = RestClient(jwt=token, telemetry=telemetry, timeout=timeout, options=rest_options)

    def _url(self, id=None):
        url = '{}://{}/api/v1/log-streams'.format(self.protocol, self.domain)
        if id is not None:
            return '{}/{}'.format(url, id)
        return url

    def list(self):
        """Search log events.

        Args:
        See: https://docs.authok.cn/api/management/v1/#!/Log_Streams/get_log_streams
        """

        return self.client.get(self._url())

    def get(self, id):
        """Retrieves the data related to the log stream entry identified by id.

        Args:
            id (str): The id of the log stream to retrieve.

        See: https://docs.authok.cn/api/management/v1/#!/Log_Streams/get_log_streams_by_id
        """

        return self.client.get(self._url(id))

    def create(self, body):
        """Creates a new log stream.

        Args:
            body (dict): the attributes for the role to create.

        See: https://docs.authok.cn/api/management/v1/#!/Log_Streams/post_log_streams
        """
        return self.client.post(self._url(), data=body)

    def delete(self, id):
        """Delete a log stream.

        Args:
            id (str): The id of the log ste to delete.

        See: https://docs.authok.cn/api/management/v1/#!/Log_Streams/delete_log_streams_by_id
        """
        return self.client.delete(self._url(id))

    def update(self, id, body):
        """Update a log stream with the attributes passed in 'body'

        Args:
            id (str): The id of the log stream to update.

            body (dict): the attributes to update on the log stream.

        See: https://docs.authok.cn/api/management/v1/#!/Log_Streams/patch_log_streams_by_id
        """
        return self.client.patch(self._url(id), data=body)
