from .rest import RestClient


class Tickets(object):
    """AuthOK tickets endpoints

    Args:
        domain (str): Your AuthOK domain, e.g: 'username.cn.authok.cn'

        token (str): Management API v1 Token

        telemetry (bool, optional): Enable or disable Telemetry
            (defaults to True)

        timeout (float or tuple, optional): Change the requests
            connect and read timeout. Pass a tuple to specify
            both values separately or a float to set both to it.
            (defaults to 5.0 for both)

        rest_options (RestClientOptions): Pass an instance of
            RestClientOptions to configure additional RestClient
            options, such as rate-limit retries.
            (defaults to None)
    """

    def __init__(self, domain, token, telemetry=True, timeout=5.0, protocol="https", rest_options=None):
        self.domain = domain
        self.protocol = protocol
        self.client = RestClient(jwt=token, telemetry=telemetry, timeout=timeout, options=rest_options)

    def _url(self, action):
        return '{}://{}/api/v1/tickets/{}'.format(self.protocol, self.domain, action)

    def create_email_verification(self, body):
        """Create an email verification ticket.

        Args:
            body (dict): attributes to set on the email verification request.

        See: https://docs.authok.cn/api/v1#!/Tickets/post_email_verification
        """
        return self.client.post(self._url('email-verification'), data=body)

    def create_pswd_change(self, body):
        """Create password change ticket.

        Args:
            body (dict): attributes to set on the password change request.

        See: https://docs.authok.cn/api/v1#!/Tickets/post_password_change
        """
        return self.client.post(self._url('password-change'), data=body)
