"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    /**
     * Canary deployment type
     */
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    /**
     * Linear deployment type
     */
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codedeploy_CustomLambdaDeploymentConfigProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomLambdaDeploymentConfig);
            }
            throw error;
        }
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = props.deploymentConfigName
            ?? `${core_1.Names.uniqueId(this)}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
                ? 'Every'
                : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    validate() {
        return utils_1.validateName('Deployment config', this.deploymentConfigName);
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
_a = JSII_RTTI_SYMBOL_1;
CustomLambdaDeploymentConfig[_a] = { fqn: "@aws-cdk/aws-codedeploy.CustomLambdaDeploymentConfig", version: "1.167.0" };
//# sourceMappingURL=data:application/json;base64,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