"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const esbuild_installation_1 = require("./esbuild-installation");
const package_manager_1 = require("./package-manager");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : esbuild_installation_1.EsbuildInstallation.detect();
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_b = props.externalModules) !== null && _b !== void 0 ? _b : ['aws-sdk'],
            ...(_c = props.nodeModules) !== null && _c !== void 0 ? _c : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage
            ? (_d = props.dockerImage) !== null && _d !== void 0 ? _d : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_e = props.buildArgs) !== null && _e !== void 0 ? _e : {},
                    IMAGE: props.runtime.bundlingImage.image,
                    ESBUILD_VERSION: (_f = props.esbuildVersion) !== null && _f !== void 0 ? _f : ESBUILD_MAJOR_VERSION,
                },
            }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const pathJoin = osPathJoin(options.osPlatform);
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${pathJoin(options.inputDir, this.relativeEntryPath)}"`,
            `--target=${(_c = this.props.target) !== null && _c !== void 0 ? _c : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(options.outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_d = this.relativeDepsLockFilePath) !== null && _d !== void 0 ? _d : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_f = (_e = this.props.commandHooks) === null || _e === void 0 ? void 0 : _e.beforeBundling(options.inputDir, options.outputDir)) !== null && _f !== void 0 ? _f : [],
            esbuildCommand.join(' '),
            ...(_h = (this.props.nodeModules && ((_g = this.props.commandHooks) === null || _g === void 0 ? void 0 : _g.beforeInstall(options.inputDir, options.outputDir)))) !== null && _h !== void 0 ? _h : [],
            depsCommand,
            ...(_k = (_j = this.props.commandHooks) === null || _j === void 0 ? void 0 : _j.afterBundling(options.inputDir, options.outputDir)) !== null && _k !== void 0 ? _k : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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