"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('tsc');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    static clearTscCompilationCache() {
        this.tscCompiled = false;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        const pathJoin = osPathJoin(options.osPlatform);
        let tscCommand = '';
        let relativeEntryPath = this.relativeEntryPath;
        if (this.props.preCompilation) {
            let tsconfig = this.relativeTsconfigPath;
            if (!tsconfig) {
                const findConfig = util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
                if (!findConfig) {
                    throw new Error('Cannot find a tsconfig.json, please specify the prop: tsconfig');
                }
                tsconfig = path.relative(this.projectRoot, findConfig);
            }
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
            if (!Bundling.tscCompiled) {
                // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next ts file.
                tscCommand = `${options.tscRunner} --project ${pathJoin(options.inputDir, tsconfig)} --rootDir ./ --outDir ./`;
                Bundling.tscCompiled = true;
            }
        }
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_c = this.props.sourceMapMode) !== null && _c !== void 0 ? _c : this.props.sourceMap;
        const sourceMapMode = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_e = this.props.sourcesContent) !== null && _e !== void 0 ? _e : true;
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${pathJoin(options.inputDir, relativeEntryPath)}"`,
            `--target=${(_f = this.props.target) !== null && _f !== void 0 ? _f : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(options.outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_g = this.relativeDepsLockFilePath) !== null && _g !== void 0 ? _g : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.beforeBundling(options.inputDir, options.outputDir)) !== null && _j !== void 0 ? _j : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_l = (this.props.nodeModules && ((_k = this.props.commandHooks) === null || _k === void 0 ? void 0 : _k.beforeInstall(options.inputDir, options.outputDir)))) !== null && _l !== void 0 ? _l : [],
            depsCommand,
            ...(_o = (_m = this.props.commandHooks) === null || _m === void 0 ? void 0 : _m.afterBundling(options.inputDir, options.outputDir)) !== null && _o !== void 0 ? _o : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
Bundling.tscCompiled = false;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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