"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('tsc');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    static clearTscCompilationCache() {
        this.tscCompiled = false;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o;
        const pathJoin = osPathJoin(options.osPlatform);
        let tscCommand = '';
        let relativeEntryPath = this.relativeEntryPath;
        if (this.props.preCompilation) {
            let tsconfig = this.relativeTsconfigPath;
            if (!tsconfig) {
                const findConfig = util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
                if (!findConfig) {
                    throw new Error('Cannot find a tsconfig.json, please specify the prop: tsconfig');
                }
                tsconfig = path.relative(this.projectRoot, findConfig);
            }
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
            if (!Bundling.tscCompiled) {
                // Intentionally Setting rootDir and outDir, so that the compiled js file always end up next ts file.
                tscCommand = `${options.tscRunner} --project ${pathJoin(options.inputDir, tsconfig)} --rootDir ./ --outDir ./`;
                Bundling.tscCompiled = true;
            }
        }
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_c = this.props.sourceMapMode) !== null && _c !== void 0 ? _c : this.props.sourceMap;
        const sourceMapMode = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_e = this.props.sourcesContent) !== null && _e !== void 0 ? _e : true;
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${pathJoin(options.inputDir, relativeEntryPath)}"`,
            `--target=${(_f = this.props.target) !== null && _f !== void 0 ? _f : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile="${pathJoin(options.outputDir, 'index.js')}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_g = this.relativeDepsLockFilePath) !== null && _g !== void 0 ? _g : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.beforeBundling(options.inputDir, options.outputDir)) !== null && _j !== void 0 ? _j : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_l = (this.props.nodeModules && ((_k = this.props.commandHooks) === null || _k === void 0 ? void 0 : _k.beforeInstall(options.inputDir, options.outputDir)))) !== null && _l !== void 0 ? _l : [],
            depsCommand,
            ...(_o = (_m = this.props.commandHooks) === null || _m === void 0 ? void 0 : _m.afterBundling(options.inputDir, options.outputDir)) !== null && _o !== void 0 ? _o : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
Bundling.tscCompiled = false;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUE2RTtBQUM3RSxxQ0FBcUM7QUFDckMsaUVBQTZEO0FBQzdELHVEQUFtRDtBQUNuRCxtQ0FBeUQ7QUFDekQsaUNBQTJEO0FBRTNELE1BQU0scUJBQXFCLEdBQUcsR0FBRyxDQUFDO0FBc0NsQzs7R0FFRztBQUNILE1BQWEsUUFBUTtJQTBDbkIsWUFBNkIsS0FBb0I7O1FBQXBCLFVBQUssR0FBTCxLQUFLLENBQWU7UUFDL0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxnQ0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUUxRSxRQUFRLENBQUMsbUJBQW1CLFNBQUcsUUFBUSxDQUFDLG1CQUFtQixtQ0FBSSwwQ0FBbUIsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDckcsUUFBUSxDQUFDLGVBQWUsU0FBRyxRQUFRLENBQUMsZUFBZSxtQ0FBSSwwQ0FBbUIsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFekYsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNwRixJQUFJLENBQUMsd0JBQXdCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztRQUV0RyxJQUFJLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsS0FBSyxDQUFDLGdCQUFnQiw2QkFBNkIsSUFBSSxDQUFDLFdBQVcsS0FBSyxJQUFJLENBQUMsd0JBQXdCLEdBQUcsQ0FBQyxDQUFDO1NBQ3pKO1FBRUQsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztTQUMzRjtRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUMxRTtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUc7WUFDZixTQUFHLEtBQUssQ0FBQyxlQUFlLG1DQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRTtTQUMzQixDQUFDO1FBRUYsa0JBQWtCO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDO1FBQ3BGLElBQUksQ0FBQyxLQUFLLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxPQUFDLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxFQUMzRztZQUNFLFNBQVMsRUFBRTtnQkFDVCxTQUFHLEtBQUssQ0FBQyxTQUFTLG1DQUFJLEVBQUU7Z0JBQ3hCLEtBQUssRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQyxLQUFLO2dCQUN4QyxlQUFlLFFBQUUsS0FBSyxDQUFDLGNBQWMsbUNBQUkscUJBQXFCO2FBQy9EO1lBQ0QsUUFBUSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsY0FBYztTQUM1QyxDQUFDLENBQ0YsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsbUNBQW1DO1FBRTlFLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQztZQUNqRCxRQUFRLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxrQkFBa0I7WUFDN0MsU0FBUyxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsbUJBQW1CO1lBQy9DLGFBQWEsRUFBRSxTQUFTO1lBQ3hCLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLFVBQVUsRUFBRSxPQUFPO1NBQ3BCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyw2RUFBNkU7UUFDN0Usa0VBQWtFO1FBQ2xFLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxHQUFHLENBQUM7UUFFNUIsaUJBQWlCO1FBQ2pCLElBQUksQ0FBQyxLQUFLLENBQUMsbUJBQW1CLEVBQUUsRUFBRSwrQkFBK0I7WUFDL0QsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztTQUM5QztLQUNGO0lBbEdEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFzQjtRQUN6QyxPQUFPLGlCQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUU7WUFDekMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxTQUFTO1lBQzVCLGFBQWEsRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNO1lBQ3RGLFFBQVEsRUFBRSxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUM7U0FDaEMsQ0FBQyxDQUFDO0tBQ0o7SUFFTSxNQUFNLENBQUMsNkJBQTZCO1FBQ3pDLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxTQUFTLENBQUM7S0FDdEM7SUFFTSxNQUFNLENBQUMseUJBQXlCO1FBQ3JDLElBQUksQ0FBQyxlQUFlLEdBQUcsU0FBUyxDQUFDO0tBQ2xDO0lBRU0sTUFBTSxDQUFDLHdCQUF3QjtRQUNwQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQztLQUMxQjtJQStFTyxxQkFBcUIsQ0FBQyxPQUErQjs7UUFDM0QsTUFBTSxRQUFRLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNoRCxJQUFJLFVBQVUsR0FBVyxFQUFFLENBQUM7UUFDNUIsSUFBSSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUM7UUFFL0MsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUU3QixJQUFJLFFBQVEsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUM7WUFDekMsSUFBSSxDQUFDLFFBQVEsRUFBRTtnQkFDYixNQUFNLFVBQVUsR0FBRyxhQUFNLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO2dCQUMzRSxJQUFJLENBQUMsVUFBVSxFQUFFO29CQUNmLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztpQkFDbkY7Z0JBQ0QsUUFBUSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxVQUFVLENBQUMsQ0FBQzthQUN4RDtZQUVELGlCQUFpQixHQUFHLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDcEUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUU7Z0JBQ3pCLHFHQUFxRztnQkFDckcsVUFBVSxHQUFHLEdBQUcsT0FBTyxDQUFDLFNBQVMsY0FBYyxRQUFRLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxRQUFRLENBQUMsMkJBQTJCLENBQUM7Z0JBQy9HLFFBQVEsQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDO2FBQzdCO1NBQ0Y7UUFFRCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUN4RCxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxPQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxFQUFFLENBQUMsQ0FBQztRQUV4RCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxLQUFLLEtBQUssSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxDQUFDLENBQUM7U0FDekU7UUFFRCxNQUFNLGdCQUFnQixTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxtQ0FBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUMxRSxNQUFNLGFBQWEsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUkscUJBQWEsQ0FBQyxPQUFPLENBQUM7UUFDeEUsTUFBTSxjQUFjLEdBQUcsYUFBYSxLQUFLLHFCQUFhLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUNyRyxNQUFNLGNBQWMsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsbUNBQUksSUFBSSxDQUFDO1FBRXpELE1BQU0sY0FBYyxHQUFhO1lBQy9CLE9BQU8sQ0FBQyxhQUFhO1lBQ3JCLFVBQVUsRUFBRSxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLGlCQUFpQixDQUFDLEdBQUc7WUFDaEUsWUFBWSxNQUFBLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvRCxpQkFBaUI7WUFDakIsY0FBYyxRQUFRLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxVQUFVLENBQUMsR0FBRztZQUN4RCxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3hDLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxjQUFjLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzNELEdBQUcsY0FBYyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMscUJBQXFCLGNBQWMsRUFBRSxDQUFDO1lBQ2hFLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxjQUFjLFFBQVEsRUFBRSxDQUFDO1lBQzNELEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxFQUFFLEVBQUUsQ0FBQyxZQUFZLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztZQUMxRCxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsRUFBRSxFQUFFLENBQUMsWUFBWSxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzVFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDcEUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUMvQyxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUMzRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDOUYsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDaEYsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDaEYsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtTQUNqRSxDQUFDO1FBRUYsSUFBSSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3JCLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDMUIsMkVBQTJFO1lBQzNFLDRCQUE0QjtZQUM1QixNQUFNLE9BQU8sR0FBRyxhQUFNLENBQUMsY0FBYyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ3ZFLElBQUksQ0FBQyxPQUFPLEVBQUU7Z0JBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyw4RkFBOEYsQ0FBQyxDQUFDO2FBQ2pIO1lBRUQsMkRBQTJEO1lBQzNELE1BQU0sWUFBWSxHQUFHLDBCQUFtQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQzFFLE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztZQUVwRCxNQUFNLFlBQVksR0FBRyxRQUFRLENBQUMsT0FBTyxDQUFDLFFBQVEsUUFBRSxJQUFJLENBQUMsd0JBQXdCLG1DQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7WUFFL0csb0VBQW9FO1lBQ3BFLFdBQVcsR0FBRyxLQUFLLENBQUM7Z0JBQ2xCLFNBQVMsQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsY0FBYyxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQztnQkFDbEYsU0FBUyxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUUsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDdkYsU0FBUyxDQUFDLGVBQWUsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDO2dCQUM1QyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO2FBQzdDLENBQUMsQ0FBQztTQUNKO1FBRUQsT0FBTyxLQUFLLENBQUM7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxjQUFjLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsU0FBUyxvQ0FBSyxFQUFFO1lBQ3JGLFVBQVU7WUFDVixjQUFjLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQztZQUN4QixTQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLFdBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLDBDQUFFLGFBQWEsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxTQUFTLEVBQUMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ2hILFdBQVc7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxhQUFhLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsU0FBUyxvQ0FBSyxFQUFFO1NBQ3JGLENBQUMsQ0FBQztLQUNKO0lBRU8sd0JBQXdCOztRQUM5QixNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDakMsTUFBTSxrQkFBa0IsR0FBRyxDQUFDLFNBQWlCLEVBQUUsT0FBNEIsRUFBRSxHQUF5QixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUM7WUFDcEksUUFBUSxFQUFFLElBQUksQ0FBQyxXQUFXO1lBQzFCLFNBQVM7WUFDVCxhQUFhLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDekYsU0FBUyxFQUFFLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7WUFDbEYsVUFBVTtTQUNYLENBQUMsQ0FBQztRQUNILE1BQU0sV0FBVyxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxtQ0FBSSxFQUFFLENBQUM7UUFDakQsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQztRQUU3QixPQUFPO1lBQ0wsU0FBUyxDQUFDLFNBQWlCO2dCQUN6QixJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixFQUFFO29CQUNqQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO29CQUNwRixPQUFPLEtBQUssQ0FBQztpQkFDZDtnQkFFRCxJQUFJLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsR0FBRyxxQkFBcUIsR0FBRyxDQUFDLEVBQUU7b0JBQ2pGLE1BQU0sSUFBSSxLQUFLLENBQUMsNEJBQTRCLHFCQUFxQixjQUFjLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO2lCQUN4SDtnQkFFRCxNQUFNLFlBQVksR0FBRyxrQkFBa0IsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLG1CQUFtQixFQUFFLFFBQVEsQ0FBQyxlQUFlLENBQUMsQ0FBQztnQkFFM0csV0FBSSxDQUNGLFVBQVUsS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUN2QztvQkFDRSxVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUk7b0JBQ3BDLFlBQVk7aUJBQ2IsRUFDRDtvQkFDRSxHQUFHLEVBQUUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsR0FBRyxXQUFXLEVBQUU7b0JBQ3ZDLEtBQUssRUFBRTt3QkFDTCxRQUFRO3dCQUNSLE9BQU8sQ0FBQyxNQUFNO3dCQUNkLFNBQVM7cUJBQ1Y7b0JBQ0QsR0FBRztvQkFDSCx3QkFBd0IsRUFBRSxVQUFVLEtBQUssT0FBTztpQkFDakQsQ0FBQyxDQUFDO2dCQUVMLE9BQU8sSUFBSSxDQUFDO1lBQ2QsQ0FBQztTQUNGLENBQUM7S0FDSDs7QUE3T0gsNEJBOE9DO0FBcE5nQixvQkFBVyxHQUFHLEtBQUssQ0FBQTtBQThOcEM7O0dBRUc7QUFDSCxNQUFNLFNBQVM7SUFDYixZQUE2QixVQUEyQjtRQUEzQixlQUFVLEdBQVYsVUFBVSxDQUFpQjtLQUFJO0lBRXJELFNBQVMsQ0FBQyxRQUFnQixFQUFFLElBQVM7UUFDMUMsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssT0FBTyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxlQUFlLE9BQU8sUUFBUSxFQUFFLENBQUM7U0FDbEQ7UUFFRCxPQUFPLFNBQVMsZUFBZSxPQUFPLFFBQVEsRUFBRSxDQUFDO0tBQ2xEO0lBRU0sSUFBSSxDQUFDLEdBQVcsRUFBRSxJQUFZO1FBQ25DLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxPQUFPLEVBQUU7WUFDL0IsT0FBTyxRQUFRLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztTQUM5QjtRQUVELE9BQU8sTUFBTSxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7S0FDNUI7SUFFTSxlQUFlLENBQUMsR0FBVztRQUNoQyxPQUFPLE1BQU0sR0FBRyxFQUFFLENBQUM7S0FDcEI7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsU0FBUyxLQUFLLENBQUMsUUFBa0I7SUFDL0IsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNoRCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFVBQVUsQ0FBQyxRQUF5QjtJQUMzQyxPQUFPLFVBQVMsR0FBRyxLQUFlO1FBQ2hDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztRQUNuQyxnREFBZ0Q7UUFDaEQsSUFBSSxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssT0FBTyxJQUFJLFFBQVEsS0FBSyxPQUFPLEVBQUU7WUFDckQsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztTQUNuQztRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsUUFBUSxDQUFDLE9BQWdCO0lBQ2hDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRWhELElBQUksQ0FBQyxLQUFLLEVBQUU7UUFDVixNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7S0FDekQ7SUFFRCxPQUFPLE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7QUFDM0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBcmNoaXRlY3R1cmUsIEFzc2V0Q29kZSwgQ29kZSwgUnVudGltZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgUGFja2FnZUluc3RhbGxhdGlvbiB9IGZyb20gJy4vcGFja2FnZS1pbnN0YWxsYXRpb24nO1xuaW1wb3J0IHsgUGFja2FnZU1hbmFnZXIgfSBmcm9tICcuL3BhY2thZ2UtbWFuYWdlcic7XG5pbXBvcnQgeyBCdW5kbGluZ09wdGlvbnMsIFNvdXJjZU1hcE1vZGUgfSBmcm9tICcuL3R5cGVzJztcbmltcG9ydCB7IGV4ZWMsIGV4dHJhY3REZXBlbmRlbmNpZXMsIGZpbmRVcCB9IGZyb20gJy4vdXRpbCc7XG5cbmNvbnN0IEVTQlVJTERfTUFKT1JfVkVSU0lPTiA9ICcwJztcblxuLyoqXG4gKiBCdW5kbGluZyBwcm9wZXJ0aWVzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQnVuZGxpbmdQcm9wcyBleHRlbmRzIEJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBQYXRoIHRvIGxvY2sgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwc0xvY2tGaWxlUGF0aDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBFbnRyeSBmaWxlXG4gICAqL1xuICByZWFkb25seSBlbnRyeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZSBvZiB0aGUgbGFtYmRhIGZ1bmN0aW9uXG4gICAqL1xuICByZWFkb25seSBydW50aW1lOiBSdW50aW1lO1xuXG4gIC8qKlxuICAgKiBUaGUgc3lzdGVtIGFyY2hpdGVjdHVyZSBvZiB0aGUgbGFtYmRhIGZ1bmN0aW9uXG4gICAqL1xuICByZWFkb25seSBhcmNoaXRlY3R1cmU6IEFyY2hpdGVjdHVyZTtcblxuICAvKipcbiAgICogUGF0aCB0byBwcm9qZWN0IHJvb3RcbiAgICovXG4gIHJlYWRvbmx5IHByb2plY3RSb290OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJ1biBjb21waWxhdGlvbiB1c2luZyBgdHNjYCBiZWZvcmUgYnVuZGxpbmdcbiAgICovXG4gIHJlYWRvbmx5IHByZUNvbXBpbGF0aW9uPzogYm9vbGVhblxuXG59XG5cbi8qKlxuICogQnVuZGxpbmcgd2l0aCBlc2J1aWxkXG4gKi9cbmV4cG9ydCBjbGFzcyBCdW5kbGluZyBpbXBsZW1lbnRzIGNkay5CdW5kbGluZ09wdGlvbnMge1xuICAvKipcbiAgICogZXNidWlsZCBidW5kbGVkIExhbWJkYSBhc3NldCBjb2RlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGJ1bmRsZShvcHRpb25zOiBCdW5kbGluZ1Byb3BzKTogQXNzZXRDb2RlIHtcbiAgICByZXR1cm4gQ29kZS5mcm9tQXNzZXQob3B0aW9ucy5wcm9qZWN0Um9vdCwge1xuICAgICAgYXNzZXRIYXNoOiBvcHRpb25zLmFzc2V0SGFzaCxcbiAgICAgIGFzc2V0SGFzaFR5cGU6IG9wdGlvbnMuYXNzZXRIYXNoID8gY2RrLkFzc2V0SGFzaFR5cGUuQ1VTVE9NIDogY2RrLkFzc2V0SGFzaFR5cGUuT1VUUFVULFxuICAgICAgYnVuZGxpbmc6IG5ldyBCdW5kbGluZyhvcHRpb25zKSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgY2xlYXJFc2J1aWxkSW5zdGFsbGF0aW9uQ2FjaGUoKTogdm9pZCB7XG4gICAgdGhpcy5lc2J1aWxkSW5zdGFsbGF0aW9uID0gdW5kZWZpbmVkO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBjbGVhclRzY0luc3RhbGxhdGlvbkNhY2hlKCk6IHZvaWQge1xuICAgIHRoaXMudHNjSW5zdGFsbGF0aW9uID0gdW5kZWZpbmVkO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBjbGVhclRzY0NvbXBpbGF0aW9uQ2FjaGUoKTogdm9pZCB7XG4gICAgdGhpcy50c2NDb21waWxlZCA9IGZhbHNlO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgZXNidWlsZEluc3RhbGxhdGlvbj86IFBhY2thZ2VJbnN0YWxsYXRpb247XG4gIHByaXZhdGUgc3RhdGljIHRzY0luc3RhbGxhdGlvbj86IFBhY2thZ2VJbnN0YWxsYXRpb247XG4gIHByaXZhdGUgc3RhdGljIHRzY0NvbXBpbGVkID0gZmFsc2VcblxuICAvLyBDb3JlIGJ1bmRsaW5nIG9wdGlvbnNcbiAgcHVibGljIHJlYWRvbmx5IGltYWdlOiBjZGsuRG9ja2VySW1hZ2U7XG4gIHB1YmxpYyByZWFkb25seSBjb21tYW5kOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcHVibGljIHJlYWRvbmx5IHdvcmtpbmdEaXJlY3Rvcnk6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGxvY2FsPzogY2RrLklMb2NhbEJ1bmRsaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvamVjdFJvb3Q6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSByZWxhdGl2ZUVudHJ5UGF0aDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlVHNjb25maWdQYXRoPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlRGVwc0xvY2tGaWxlUGF0aDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGV4dGVybmFsczogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgcGFja2FnZU1hbmFnZXI6IFBhY2thZ2VNYW5hZ2VyO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEJ1bmRsaW5nUHJvcHMpIHtcbiAgICB0aGlzLnBhY2thZ2VNYW5hZ2VyID0gUGFja2FnZU1hbmFnZXIuZnJvbUxvY2tGaWxlKHByb3BzLmRlcHNMb2NrRmlsZVBhdGgpO1xuXG4gICAgQnVuZGxpbmcuZXNidWlsZEluc3RhbGxhdGlvbiA9IEJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ2VzYnVpbGQnKTtcbiAgICBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPSBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ3RzYycpO1xuXG4gICAgdGhpcy5wcm9qZWN0Um9vdCA9IHByb3BzLnByb2plY3RSb290O1xuICAgIHRoaXMucmVsYXRpdmVFbnRyeVBhdGggPSBwYXRoLnJlbGF0aXZlKHRoaXMucHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy5lbnRyeSkpO1xuICAgIHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoID0gcGF0aC5yZWxhdGl2ZSh0aGlzLnByb2plY3RSb290LCBwYXRoLnJlc29sdmUocHJvcHMuZGVwc0xvY2tGaWxlUGF0aCkpO1xuXG4gICAgaWYgKHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoLmluY2x1ZGVzKCcuLicpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkIGRlcHNMb2NrRmlsZVBhdGg6ICR7cHJvcHMuZGVwc0xvY2tGaWxlUGF0aH0gdG8gYmUgdW5kZXIgcHJvamVjdFJvb3Q6ICR7dGhpcy5wcm9qZWN0Um9vdH0gKCR7dGhpcy5yZWxhdGl2ZURlcHNMb2NrRmlsZVBhdGh9KWApO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy50c2NvbmZpZykge1xuICAgICAgdGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA9IHBhdGgucmVsYXRpdmUodGhpcy5wcm9qZWN0Um9vdCwgcGF0aC5yZXNvbHZlKHByb3BzLnRzY29uZmlnKSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnByZUNvbXBpbGF0aW9uICYmICEvXFwudHN4PyQvLnRlc3QocHJvcHMuZW50cnkpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3ByZUNvbXBpbGF0aW9uIGNhbiBvbmx5IGJlIHVzZWQgd2l0aCB0eXBlc2NyaXB0IGZpbGVzJyk7XG4gICAgfVxuXG4gICAgdGhpcy5leHRlcm5hbHMgPSBbXG4gICAgICAuLi5wcm9wcy5leHRlcm5hbE1vZHVsZXMgPz8gWydhd3Mtc2RrJ10sIC8vIE1hcmsgYXdzLXNkayBhcyBleHRlcm5hbCBieSBkZWZhdWx0IChhdmFpbGFibGUgaW4gdGhlIHJ1bnRpbWUpXG4gICAgICAuLi5wcm9wcy5ub2RlTW9kdWxlcyA/PyBbXSwgLy8gTWFyayB0aGUgbW9kdWxlcyB0aGF0IHdlIGFyZSBnb2luZyB0byBpbnN0YWxsIGFzIGV4dGVybmFscyBhbHNvXG4gICAgXTtcblxuICAgIC8vIERvY2tlciBidW5kbGluZ1xuICAgIGNvbnN0IHNob3VsZEJ1aWxkSW1hZ2UgPSBwcm9wcy5mb3JjZURvY2tlckJ1bmRsaW5nIHx8ICFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uO1xuICAgIHRoaXMuaW1hZ2UgPSBzaG91bGRCdWlsZEltYWdlID8gcHJvcHMuZG9ja2VySW1hZ2UgPz8gY2RrLkRvY2tlckltYWdlLmZyb21CdWlsZChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGliJyksXG4gICAgICB7XG4gICAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICAgIC4uLnByb3BzLmJ1aWxkQXJncyA/PyB7fSxcbiAgICAgICAgICBJTUFHRTogcHJvcHMucnVudGltZS5idW5kbGluZ0ltYWdlLmltYWdlLFxuICAgICAgICAgIEVTQlVJTERfVkVSU0lPTjogcHJvcHMuZXNidWlsZFZlcnNpb24gPz8gRVNCVUlMRF9NQUpPUl9WRVJTSU9OLFxuICAgICAgICB9LFxuICAgICAgICBwbGF0Zm9ybTogcHJvcHMuYXJjaGl0ZWN0dXJlLmRvY2tlclBsYXRmb3JtLFxuICAgICAgfSlcbiAgICAgIDogY2RrLkRvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnZHVtbXknKTsgLy8gRG8gbm90IGJ1aWxkIGlmIHdlIGRvbid0IG5lZWQgdG9cblxuICAgIGNvbnN0IGJ1bmRsaW5nQ29tbWFuZCA9IHRoaXMuY3JlYXRlQnVuZGxpbmdDb21tYW5kKHtcbiAgICAgIGlucHV0RGlyOiBjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUixcbiAgICAgIG91dHB1dERpcjogY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19PVVRQVVRfRElSLFxuICAgICAgZXNidWlsZFJ1bm5lcjogJ2VzYnVpbGQnLCAvLyBlc2J1aWxkIGlzIGluc3RhbGxlZCBnbG9iYWxseSBpbiB0aGUgZG9ja2VyIGltYWdlXG4gICAgICB0c2NSdW5uZXI6ICd0c2MnLCAvLyB0c2MgaXMgaW5zdGFsbGVkIGdsb2JhbGx5IGluIHRoZSBkb2NrZXIgaW1hZ2VcbiAgICAgIG9zUGxhdGZvcm06ICdsaW51eCcsIC8vIGxpbnV4IGRvY2tlciBpbWFnZVxuICAgIH0pO1xuICAgIHRoaXMuY29tbWFuZCA9IFsnYmFzaCcsICctYycsIGJ1bmRsaW5nQ29tbWFuZF07XG4gICAgdGhpcy5lbnZpcm9ubWVudCA9IHByb3BzLmVudmlyb25tZW50O1xuICAgIC8vIEJ1bmRsaW5nIHNldHMgdGhlIHdvcmtpbmcgZGlyZWN0b3J5IHRvIGNkay5Bc3NldFN0YWdpbmcuQlVORExJTkdfSU5QVVRfRElSXG4gICAgLy8gYW5kIHdlIHdhbnQgdG8gZm9yY2UgbnB4IHRvIHVzZSB0aGUgZ2xvYmFsbHkgaW5zdGFsbGVkIGVzYnVpbGQuXG4gICAgdGhpcy53b3JraW5nRGlyZWN0b3J5ID0gJy8nO1xuXG4gICAgLy8gTG9jYWwgYnVuZGxpbmdcbiAgICBpZiAoIXByb3BzLmZvcmNlRG9ja2VyQnVuZGxpbmcpIHsgLy8gb25seSBpZiBEb2NrZXIgaXMgbm90IGZvcmNlZFxuICAgICAgdGhpcy5sb2NhbCA9IHRoaXMuZ2V0TG9jYWxCdW5kbGluZ1Byb3ZpZGVyKCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVCdW5kbGluZ0NvbW1hbmQob3B0aW9uczogQnVuZGxpbmdDb21tYW5kT3B0aW9ucyk6IHN0cmluZyB7XG4gICAgY29uc3QgcGF0aEpvaW4gPSBvc1BhdGhKb2luKG9wdGlvbnMub3NQbGF0Zm9ybSk7XG4gICAgbGV0IHRzY0NvbW1hbmQ6IHN0cmluZyA9ICcnO1xuICAgIGxldCByZWxhdGl2ZUVudHJ5UGF0aCA9IHRoaXMucmVsYXRpdmVFbnRyeVBhdGg7XG5cbiAgICBpZiAodGhpcy5wcm9wcy5wcmVDb21waWxhdGlvbikge1xuXG4gICAgICBsZXQgdHNjb25maWcgPSB0aGlzLnJlbGF0aXZlVHNjb25maWdQYXRoO1xuICAgICAgaWYgKCF0c2NvbmZpZykge1xuICAgICAgICBjb25zdCBmaW5kQ29uZmlnID0gZmluZFVwKCd0c2NvbmZpZy5qc29uJywgcGF0aC5kaXJuYW1lKHRoaXMucHJvcHMuZW50cnkpKTtcbiAgICAgICAgaWYgKCFmaW5kQ29uZmlnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBhIHRzY29uZmlnLmpzb24sIHBsZWFzZSBzcGVjaWZ5IHRoZSBwcm9wOiB0c2NvbmZpZycpO1xuICAgICAgICB9XG4gICAgICAgIHRzY29uZmlnID0gcGF0aC5yZWxhdGl2ZSh0aGlzLnByb2plY3RSb290LCBmaW5kQ29uZmlnKTtcbiAgICAgIH1cblxuICAgICAgcmVsYXRpdmVFbnRyeVBhdGggPSByZWxhdGl2ZUVudHJ5UGF0aC5yZXBsYWNlKC9cXC50cyh4PykkLywgJy5qcyQxJyk7XG4gICAgICBpZiAoIUJ1bmRsaW5nLnRzY0NvbXBpbGVkKSB7XG4gICAgICAgIC8vIEludGVudGlvbmFsbHkgU2V0dGluZyByb290RGlyIGFuZCBvdXREaXIsIHNvIHRoYXQgdGhlIGNvbXBpbGVkIGpzIGZpbGUgYWx3YXlzIGVuZCB1cCBuZXh0IHRzIGZpbGUuXG4gICAgICAgIHRzY0NvbW1hbmQgPSBgJHtvcHRpb25zLnRzY1J1bm5lcn0gLS1wcm9qZWN0ICR7cGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgdHNjb25maWcpfSAtLXJvb3REaXIgLi8gLS1vdXREaXIgLi9gO1xuICAgICAgICBCdW5kbGluZy50c2NDb21waWxlZCA9IHRydWU7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgbG9hZGVycyA9IE9iamVjdC5lbnRyaWVzKHRoaXMucHJvcHMubG9hZGVyID8/IHt9KTtcbiAgICBjb25zdCBkZWZpbmVzID0gT2JqZWN0LmVudHJpZXModGhpcy5wcm9wcy5kZWZpbmUgPz8ge30pO1xuXG4gICAgaWYgKHRoaXMucHJvcHMuc291cmNlTWFwID09PSBmYWxzZSAmJiB0aGlzLnByb3BzLnNvdXJjZU1hcE1vZGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignc291cmNlTWFwTW9kZSBjYW5ub3QgYmUgdXNlZCB3aGVuIHNvdXJjZU1hcCBpcyBmYWxzZScpO1xuICAgIH1cblxuICAgIGNvbnN0IHNvdXJjZU1hcEVuYWJsZWQgPSB0aGlzLnByb3BzLnNvdXJjZU1hcE1vZGUgPz8gdGhpcy5wcm9wcy5zb3VyY2VNYXA7XG4gICAgY29uc3Qgc291cmNlTWFwTW9kZSA9IHRoaXMucHJvcHMuc291cmNlTWFwTW9kZSA/PyBTb3VyY2VNYXBNb2RlLkRFRkFVTFQ7XG4gICAgY29uc3Qgc291cmNlTWFwVmFsdWUgPSBzb3VyY2VNYXBNb2RlID09PSBTb3VyY2VNYXBNb2RlLkRFRkFVTFQgPyAnJyA6IGA9JHt0aGlzLnByb3BzLnNvdXJjZU1hcE1vZGV9YDtcbiAgICBjb25zdCBzb3VyY2VzQ29udGVudCA9IHRoaXMucHJvcHMuc291cmNlc0NvbnRlbnQgPz8gdHJ1ZTtcblxuICAgIGNvbnN0IGVzYnVpbGRDb21tYW5kOiBzdHJpbmdbXSA9IFtcbiAgICAgIG9wdGlvbnMuZXNidWlsZFJ1bm5lcixcbiAgICAgICctLWJ1bmRsZScsIGBcIiR7cGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgcmVsYXRpdmVFbnRyeVBhdGgpfVwiYCxcbiAgICAgIGAtLXRhcmdldD0ke3RoaXMucHJvcHMudGFyZ2V0ID8/IHRvVGFyZ2V0KHRoaXMucHJvcHMucnVudGltZSl9YCxcbiAgICAgICctLXBsYXRmb3JtPW5vZGUnLFxuICAgICAgYC0tb3V0ZmlsZT1cIiR7cGF0aEpvaW4ob3B0aW9ucy5vdXRwdXREaXIsICdpbmRleC5qcycpfVwiYCxcbiAgICAgIC4uLnRoaXMucHJvcHMubWluaWZ5ID8gWyctLW1pbmlmeSddIDogW10sXG4gICAgICAuLi5zb3VyY2VNYXBFbmFibGVkID8gW2AtLXNvdXJjZW1hcCR7c291cmNlTWFwVmFsdWV9YF0gOiBbXSxcbiAgICAgIC4uLnNvdXJjZXNDb250ZW50ID8gW10gOiBbYC0tc291cmNlcy1jb250ZW50PSR7c291cmNlc0NvbnRlbnR9YF0sXG4gICAgICAuLi50aGlzLmV4dGVybmFscy5tYXAoZXh0ZXJuYWwgPT4gYC0tZXh0ZXJuYWw6JHtleHRlcm5hbH1gKSxcbiAgICAgIC4uLmxvYWRlcnMubWFwKChbZXh0LCBuYW1lXSkgPT4gYC0tbG9hZGVyOiR7ZXh0fT0ke25hbWV9YCksXG4gICAgICAuLi5kZWZpbmVzLm1hcCgoW2tleSwgdmFsdWVdKSA9PiBgLS1kZWZpbmU6JHtrZXl9PSR7SlNPTi5zdHJpbmdpZnkodmFsdWUpfWApLFxuICAgICAgLi4udGhpcy5wcm9wcy5sb2dMZXZlbCA/IFtgLS1sb2ctbGV2ZWw9JHt0aGlzLnByb3BzLmxvZ0xldmVsfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmtlZXBOYW1lcyA/IFsnLS1rZWVwLW5hbWVzJ10gOiBbXSxcbiAgICAgIC4uLnRoaXMucmVsYXRpdmVUc2NvbmZpZ1BhdGggPyBbYC0tdHNjb25maWc9JHtwYXRoSm9pbihvcHRpb25zLmlucHV0RGlyLCB0aGlzLnJlbGF0aXZlVHNjb25maWdQYXRoKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5tZXRhZmlsZSA/IFtgLS1tZXRhZmlsZT0ke3BhdGhKb2luKG9wdGlvbnMub3V0cHV0RGlyLCAnaW5kZXgubWV0YS5qc29uJyl9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMuYmFubmVyID8gW2AtLWJhbm5lcjpqcz0ke0pTT04uc3RyaW5naWZ5KHRoaXMucHJvcHMuYmFubmVyKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5mb290ZXIgPyBbYC0tZm9vdGVyOmpzPSR7SlNPTi5zdHJpbmdpZnkodGhpcy5wcm9wcy5mb290ZXIpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmNoYXJzZXQgPyBbYC0tY2hhcnNldD0ke3RoaXMucHJvcHMuY2hhcnNldH1gXSA6IFtdLFxuICAgIF07XG5cbiAgICBsZXQgZGVwc0NvbW1hbmQgPSAnJztcbiAgICBpZiAodGhpcy5wcm9wcy5ub2RlTW9kdWxlcykge1xuICAgICAgLy8gRmluZCAncGFja2FnZS5qc29uJyBjbG9zZXN0IHRvIGVudHJ5IGZvbGRlciwgd2UgYXJlIGdvaW5nIHRvIGV4dHJhY3QgdGhlXG4gICAgICAvLyBtb2R1bGVzIHZlcnNpb25zIGZyb20gaXQuXG4gICAgICBjb25zdCBwa2dQYXRoID0gZmluZFVwKCdwYWNrYWdlLmpzb24nLCBwYXRoLmRpcm5hbWUodGhpcy5wcm9wcy5lbnRyeSkpO1xuICAgICAgaWYgKCFwa2dQYXRoKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgYSBgcGFja2FnZS5qc29uYCBpbiB0aGlzIHByb2plY3QuIFVzaW5nIGBub2RlTW9kdWxlc2AgcmVxdWlyZXMgYSBgcGFja2FnZS5qc29uYC4nKTtcbiAgICAgIH1cblxuICAgICAgLy8gRGV0ZXJtaW5lIGRlcGVuZGVuY2llcyB2ZXJzaW9ucywgbG9jayBmaWxlIGFuZCBpbnN0YWxsZXJcbiAgICAgIGNvbnN0IGRlcGVuZGVuY2llcyA9IGV4dHJhY3REZXBlbmRlbmNpZXMocGtnUGF0aCwgdGhpcy5wcm9wcy5ub2RlTW9kdWxlcyk7XG4gICAgICBjb25zdCBvc0NvbW1hbmQgPSBuZXcgT3NDb21tYW5kKG9wdGlvbnMub3NQbGF0Zm9ybSk7XG5cbiAgICAgIGNvbnN0IGxvY2tGaWxlUGF0aCA9IHBhdGhKb2luKG9wdGlvbnMuaW5wdXREaXIsIHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoID8/IHRoaXMucGFja2FnZU1hbmFnZXIubG9ja0ZpbGUpO1xuXG4gICAgICAvLyBDcmVhdGUgZHVtbXkgcGFja2FnZS5qc29uLCBjb3B5IGxvY2sgZmlsZSBpZiBhbnkgYW5kIHRoZW4gaW5zdGFsbFxuICAgICAgZGVwc0NvbW1hbmQgPSBjaGFpbihbXG4gICAgICAgIG9zQ29tbWFuZC53cml0ZUpzb24ocGF0aEpvaW4ob3B0aW9ucy5vdXRwdXREaXIsICdwYWNrYWdlLmpzb24nKSwgeyBkZXBlbmRlbmNpZXMgfSksXG4gICAgICAgIG9zQ29tbWFuZC5jb3B5KGxvY2tGaWxlUGF0aCwgcGF0aEpvaW4ob3B0aW9ucy5vdXRwdXREaXIsIHRoaXMucGFja2FnZU1hbmFnZXIubG9ja0ZpbGUpKSxcbiAgICAgICAgb3NDb21tYW5kLmNoYW5nZURpcmVjdG9yeShvcHRpb25zLm91dHB1dERpciksXG4gICAgICAgIHRoaXMucGFja2FnZU1hbmFnZXIuaW5zdGFsbENvbW1hbmQuam9pbignICcpLFxuICAgICAgXSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGNoYWluKFtcbiAgICAgIC4uLnRoaXMucHJvcHMuY29tbWFuZEhvb2tzPy5iZWZvcmVCdW5kbGluZyhvcHRpb25zLmlucHV0RGlyLCBvcHRpb25zLm91dHB1dERpcikgPz8gW10sXG4gICAgICB0c2NDb21tYW5kLFxuICAgICAgZXNidWlsZENvbW1hbmQuam9pbignICcpLFxuICAgICAgLi4uKHRoaXMucHJvcHMubm9kZU1vZHVsZXMgJiYgdGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmJlZm9yZUluc3RhbGwob3B0aW9ucy5pbnB1dERpciwgb3B0aW9ucy5vdXRwdXREaXIpKSA/PyBbXSxcbiAgICAgIGRlcHNDb21tYW5kLFxuICAgICAgLi4udGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmFmdGVyQnVuZGxpbmcob3B0aW9ucy5pbnB1dERpciwgb3B0aW9ucy5vdXRwdXREaXIpID8/IFtdLFxuICAgIF0pO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRMb2NhbEJ1bmRsaW5nUHJvdmlkZXIoKTogY2RrLklMb2NhbEJ1bmRsaW5nIHtcbiAgICBjb25zdCBvc1BsYXRmb3JtID0gb3MucGxhdGZvcm0oKTtcbiAgICBjb25zdCBjcmVhdGVMb2NhbENvbW1hbmQgPSAob3V0cHV0RGlyOiBzdHJpbmcsIGVzYnVpbGQ6IFBhY2thZ2VJbnN0YWxsYXRpb24sIHRzYz86IFBhY2thZ2VJbnN0YWxsYXRpb24pID0+IHRoaXMuY3JlYXRlQnVuZGxpbmdDb21tYW5kKHtcbiAgICAgIGlucHV0RGlyOiB0aGlzLnByb2plY3RSb290LFxuICAgICAgb3V0cHV0RGlyLFxuICAgICAgZXNidWlsZFJ1bm5lcjogZXNidWlsZC5pc0xvY2FsID8gdGhpcy5wYWNrYWdlTWFuYWdlci5ydW5CaW5Db21tYW5kKCdlc2J1aWxkJykgOiAnZXNidWlsZCcsXG4gICAgICB0c2NSdW5uZXI6IHRzYyAmJiAodHNjLmlzTG9jYWwgPyB0aGlzLnBhY2thZ2VNYW5hZ2VyLnJ1bkJpbkNvbW1hbmQoJ3RzYycpIDogJ3RzYycpLFxuICAgICAgb3NQbGF0Zm9ybSxcbiAgICB9KTtcbiAgICBjb25zdCBlbnZpcm9ubWVudCA9IHRoaXMucHJvcHMuZW52aXJvbm1lbnQgPz8ge307XG4gICAgY29uc3QgY3dkID0gdGhpcy5wcm9qZWN0Um9vdDtcblxuICAgIHJldHVybiB7XG4gICAgICB0cnlCdW5kbGUob3V0cHV0RGlyOiBzdHJpbmcpIHtcbiAgICAgICAgaWYgKCFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uKSB7XG4gICAgICAgICAgcHJvY2Vzcy5zdGRlcnIud3JpdGUoJ2VzYnVpbGQgY2Fubm90IHJ1biBsb2NhbGx5LiBTd2l0Y2hpbmcgdG8gRG9ja2VyIGJ1bmRsaW5nLlxcbicpO1xuICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICghQnVuZGxpbmcuZXNidWlsZEluc3RhbGxhdGlvbi52ZXJzaW9uLnN0YXJ0c1dpdGgoYCR7RVNCVUlMRF9NQUpPUl9WRVJTSU9OfS5gKSkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgRXhwZWN0ZWQgZXNidWlsZCB2ZXJzaW9uICR7RVNCVUlMRF9NQUpPUl9WRVJTSU9OfS54IGJ1dCBnb3QgJHtCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uLnZlcnNpb259YCk7XG4gICAgICAgIH1cblxuICAgICAgICBjb25zdCBsb2NhbENvbW1hbmQgPSBjcmVhdGVMb2NhbENvbW1hbmQob3V0cHV0RGlyLCBCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uLCBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24pO1xuXG4gICAgICAgIGV4ZWMoXG4gICAgICAgICAgb3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJyA/ICdjbWQnIDogJ2Jhc2gnLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIG9zUGxhdGZvcm0gPT09ICd3aW4zMicgPyAnL2MnIDogJy1jJyxcbiAgICAgICAgICAgIGxvY2FsQ29tbWFuZCxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGVudjogeyAuLi5wcm9jZXNzLmVudiwgLi4uZW52aXJvbm1lbnQgfSxcbiAgICAgICAgICAgIHN0ZGlvOiBbIC8vIHNob3cgb3V0cHV0XG4gICAgICAgICAgICAgICdpZ25vcmUnLCAvLyBpZ25vcmUgc3RkaW9cbiAgICAgICAgICAgICAgcHJvY2Vzcy5zdGRlcnIsIC8vIHJlZGlyZWN0IHN0ZG91dCB0byBzdGRlcnJcbiAgICAgICAgICAgICAgJ2luaGVyaXQnLCAvLyBpbmhlcml0IHN0ZGVyclxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIGN3ZCxcbiAgICAgICAgICAgIHdpbmRvd3NWZXJiYXRpbUFyZ3VtZW50czogb3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJyxcbiAgICAgICAgICB9KTtcblxuICAgICAgICByZXR1cm4gdHJ1ZTtcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgQnVuZGxpbmdDb21tYW5kT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGlucHV0RGlyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG91dHB1dERpcjogc3RyaW5nO1xuICByZWFkb25seSBlc2J1aWxkUnVubmVyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHRzY1J1bm5lcj86IHN0cmluZztcbiAgcmVhZG9ubHkgb3NQbGF0Zm9ybTogTm9kZUpTLlBsYXRmb3JtO1xufVxuXG4vKipcbiAqIE9TIGFnbm9zdGljIGNvbW1hbmRcbiAqL1xuY2xhc3MgT3NDb21tYW5kIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0pIHt9XG5cbiAgcHVibGljIHdyaXRlSnNvbihmaWxlUGF0aDogc3RyaW5nLCBkYXRhOiBhbnkpOiBzdHJpbmcge1xuICAgIGNvbnN0IHN0cmluZ2lmaWVkRGF0YSA9IEpTT04uc3RyaW5naWZ5KGRhdGEpO1xuICAgIGlmICh0aGlzLm9zUGxhdGZvcm0gPT09ICd3aW4zMicpIHtcbiAgICAgIHJldHVybiBgZWNobyBeJHtzdHJpbmdpZmllZERhdGF9XiA+ICR7ZmlsZVBhdGh9YDtcbiAgICB9XG5cbiAgICByZXR1cm4gYGVjaG8gJyR7c3RyaW5naWZpZWREYXRhfScgPiAke2ZpbGVQYXRofWA7XG4gIH1cblxuICBwdWJsaWMgY29weShzcmM6IHN0cmluZywgZGVzdDogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBpZiAodGhpcy5vc1BsYXRmb3JtID09PSAnd2luMzInKSB7XG4gICAgICByZXR1cm4gYGNvcHkgJHtzcmN9ICR7ZGVzdH1gO1xuICAgIH1cblxuICAgIHJldHVybiBgY3AgJHtzcmN9ICR7ZGVzdH1gO1xuICB9XG5cbiAgcHVibGljIGNoYW5nZURpcmVjdG9yeShkaXI6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGBjZCAke2Rpcn1gO1xuICB9XG59XG5cbi8qKlxuICogQ2hhaW4gY29tbWFuZHNcbiAqL1xuZnVuY3Rpb24gY2hhaW4oY29tbWFuZHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgcmV0dXJuIGNvbW1hbmRzLmZpbHRlcihjID0+ICEhYykuam9pbignICYmICcpO1xufVxuXG4vKipcbiAqIFBsYXRmb3JtIHNwZWNpZmljIHBhdGggam9pblxuICovXG5mdW5jdGlvbiBvc1BhdGhKb2luKHBsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0pIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKC4uLnBhdGhzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gICAgY29uc3Qgam9pbmVkID0gcGF0aC5qb2luKC4uLnBhdGhzKTtcbiAgICAvLyBJZiB3ZSBhcmUgb24gd2luMzIgYnV0IG5lZWQgcG9zaXggc3R5bGUgcGF0aHNcbiAgICBpZiAob3MucGxhdGZvcm0oKSA9PT0gJ3dpbjMyJyAmJiBwbGF0Zm9ybSAhPT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGpvaW5lZC5yZXBsYWNlKC9cXFxcL2csICcvJyk7XG4gICAgfVxuICAgIHJldHVybiBqb2luZWQ7XG4gIH07XG59XG5cbi8qKlxuICogQ29udmVydHMgYSBydW50aW1lIHRvIGFuIGVzYnVpbGQgbm9kZSB0YXJnZXRcbiAqL1xuZnVuY3Rpb24gdG9UYXJnZXQocnVudGltZTogUnVudGltZSk6IHN0cmluZyB7XG4gIGNvbnN0IG1hdGNoID0gcnVudGltZS5uYW1lLm1hdGNoKC9ub2RlanMoXFxkKykvKTtcblxuICBpZiAoIW1hdGNoKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZXh0cmFjdCB2ZXJzaW9uIGZyb20gcnVudGltZS4nKTtcbiAgfVxuXG4gIHJldHVybiBgbm9kZSR7bWF0Y2hbMV19YDtcbn1cbiJdfQ==