"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const package_manager_1 = require("./package-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using esbuild.
 *
 * @stability stable
 */
class NodejsFunction extends lambda.Function {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f, _g;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_nodejs_NodejsFunctionProps(props);
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_b = props.handler) !== null && _b !== void 0 ? _b : 'handler';
        const runtime = (_c = props.runtime) !== null && _c !== void 0 ? _c : lambda.Runtime.NODEJS_14_X;
        const architecture = (_d = props.architecture) !== null && _d !== void 0 ? _d : aws_lambda_1.Architecture.X86_64;
        const depsLockFilePath = findLockFile(props.depsLockFilePath);
        const projectRoot = (_e = props.projectRoot) !== null && _e !== void 0 ? _e : path.dirname(depsLockFilePath);
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(_f = props.bundling) !== null && _f !== void 0 ? _f : {},
                entry,
                runtime,
                architecture,
                depsLockFilePath,
                projectRoot,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if ((_g = props.awsSdkConnectionReuse) !== null && _g !== void 0 ? _g : true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
_a = JSII_RTTI_SYMBOL_1;
NodejsFunction[_a] = { fqn: "@aws-cdk/aws-lambda-nodejs.NodejsFunction", version: "1.140.0" };
/**
 * Checks given lock file or searches for a lock file
 */
function findLockFile(depsLockFilePath) {
    if (depsLockFilePath) {
        if (!fs.existsSync(depsLockFilePath)) {
            throw new Error(`Lock file at ${depsLockFilePath} doesn't exist`);
        }
        if (!fs.statSync(depsLockFilePath).isFile()) {
            throw new Error('`depsLockFilePath` should point to a file');
        }
        return path.resolve(depsLockFilePath);
    }
    const lockFiles = util_1.findUpMultiple([
        package_manager_1.PackageManager.PNPM.lockFile,
        package_manager_1.PackageManager.YARN.lockFile,
        package_manager_1.PackageManager.NPM.lockFile,
    ]);
    if (lockFiles.length === 0) {
        throw new Error('Cannot find a package lock file (`pnpm-lock.yaml`, `yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
    }
    if (lockFiles.length > 1) {
        throw new Error(`Multiple package lock files found: ${lockFiles.join(', ')}. Please specify the desired one with \`depsFileLockPath\`.`);
    }
    return lockFiles[0];
}
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 * 4. A .mjs file name as the defining file with id as suffix (defining-file.id.mjs)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?|mjs)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    const mjsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.mjs`);
    if (fs.existsSync(mjsHandlerFile)) {
        return mjsHandlerFile;
    }
    throw new Error(`Cannot find handler file ${tsHandlerFile}, ${jsHandlerFile} or ${mjsHandlerFile}`);
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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