"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_installation_1 = require("./package-installation");
const package_manager_1 = require("./package-manager");
const types_1 = require("./types");
const util_1 = require("./util");
const ESBUILD_MAJOR_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.props = props;
        this.packageManager = package_manager_1.PackageManager.fromLockFile(props.depsLockFilePath);
        Bundling.esbuildInstallation = (_a = Bundling.esbuildInstallation) !== null && _a !== void 0 ? _a : package_installation_1.PackageInstallation.detect('esbuild');
        Bundling.tscInstallation = (_b = Bundling.tscInstallation) !== null && _b !== void 0 ? _b : package_installation_1.PackageInstallation.detect('tsc');
        this.projectRoot = props.projectRoot;
        this.relativeEntryPath = path.relative(this.projectRoot, path.resolve(props.entry));
        this.relativeDepsLockFilePath = path.relative(this.projectRoot, path.resolve(props.depsLockFilePath));
        if (this.relativeDepsLockFilePath.includes('..')) {
            throw new Error(`Expected depsLockFilePath: ${props.depsLockFilePath} to be under projectRoot: ${this.projectRoot} (${this.relativeDepsLockFilePath})`);
        }
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(this.projectRoot, path.resolve(props.tsconfig));
        }
        if (props.preCompilation && !/\.tsx?$/.test(props.entry)) {
            throw new Error('preCompilation can only be used with typescript files');
        }
        if (props.format === types_1.OutputFormat.ESM
            && (props.runtime === aws_lambda_1.Runtime.NODEJS_10_X || props.runtime === aws_lambda_1.Runtime.NODEJS_12_X)) {
            throw new Error(`ECMAScript module output format is not supported by the ${props.runtime.name} runtime`);
        }
        this.externals = [
            ...(_c = props.externalModules) !== null && _c !== void 0 ? _c : ['aws-sdk'],
            ...(_d = props.nodeModules) !== null && _d !== void 0 ? _d : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.esbuildInstallation;
        this.image = shouldBuildImage ? (_e = props.dockerImage) !== null && _e !== void 0 ? _e : cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...(_f = props.buildArgs) !== null && _f !== void 0 ? _f : {},
                IMAGE: props.runtime.bundlingImage.image,
                ESBUILD_VERSION: (_g = props.esbuildVersion) !== null && _g !== void 0 ? _g : ESBUILD_MAJOR_VERSION,
            },
            platform: props.architecture.dockerPlatform,
        }) : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand({
            inputDir: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            esbuildRunner: 'esbuild',
            tscRunner: 'tsc',
            osPlatform: 'linux',
        });
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            this.local = this.getLocalBundlingProvider();
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.projectRoot, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash ? cdk.AssetHashType.CUSTOM : cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearEsbuildInstallationCache() {
        this.esbuildInstallation = undefined;
    }
    static clearTscInstallationCache() {
        this.tscInstallation = undefined;
    }
    createBundlingCommand(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        const pathJoin = osPathJoin(options.osPlatform);
        let relativeEntryPath = pathJoin(options.inputDir, this.relativeEntryPath);
        let tscCommand = '';
        if (this.props.preCompilation) {
            const tsconfig = (_a = this.props.tsconfig) !== null && _a !== void 0 ? _a : util_1.findUp('tsconfig.json', path.dirname(this.props.entry));
            if (!tsconfig) {
                throw new Error('Cannot find a `tsconfig.json` but `preCompilation` is set to `true`, please specify it via `tsconfig`');
            }
            const compilerOptions = util_1.getTsconfigCompilerOptions(tsconfig);
            tscCommand = `${options.tscRunner} "${relativeEntryPath}" ${compilerOptions}`;
            relativeEntryPath = relativeEntryPath.replace(/\.ts(x?)$/, '.js$1');
        }
        const loaders = Object.entries((_b = this.props.loader) !== null && _b !== void 0 ? _b : {});
        const defines = Object.entries((_c = this.props.define) !== null && _c !== void 0 ? _c : {});
        if (this.props.sourceMap === false && this.props.sourceMapMode) {
            throw new Error('sourceMapMode cannot be used when sourceMap is false');
        }
        const sourceMapEnabled = (_d = this.props.sourceMapMode) !== null && _d !== void 0 ? _d : this.props.sourceMap;
        const sourceMapMode = (_e = this.props.sourceMapMode) !== null && _e !== void 0 ? _e : types_1.SourceMapMode.DEFAULT;
        const sourceMapValue = sourceMapMode === types_1.SourceMapMode.DEFAULT ? '' : `=${this.props.sourceMapMode}`;
        const sourcesContent = (_f = this.props.sourcesContent) !== null && _f !== void 0 ? _f : true;
        const outFile = this.props.format === types_1.OutputFormat.ESM ? 'index.mjs' : 'index.js';
        const esbuildCommand = [
            options.esbuildRunner,
            '--bundle', `"${relativeEntryPath}"`,
            `--target=${(_g = this.props.target) !== null && _g !== void 0 ? _g : toTarget(this.props.runtime)}`,
            '--platform=node',
            ...this.props.format ? [`--format=${this.props.format}`] : [],
            `--outfile="${pathJoin(options.outputDir, outFile)}"`,
            ...this.props.minify ? ['--minify'] : [],
            ...sourceMapEnabled ? [`--sourcemap${sourceMapValue}`] : [],
            ...sourcesContent ? [] : [`--sources-content=${sourcesContent}`],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${JSON.stringify(value)}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(options.inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(options.outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner:js=${JSON.stringify(this.props.banner)}`] : [],
            ...this.props.footer ? [`--footer:js=${JSON.stringify(this.props.footer)}`] : [],
            ...this.props.charset ? [`--charset=${this.props.charset}`] : [],
            ...this.props.mainFields ? [`--main-fields=${this.props.mainFields.join(',')}`] : [],
        ];
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            const osCommand = new OsCommand(options.osPlatform);
            const lockFilePath = pathJoin(options.inputDir, (_h = this.relativeDepsLockFilePath) !== null && _h !== void 0 ? _h : this.packageManager.lockFile);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(options.outputDir, 'package.json'), { dependencies }),
                osCommand.copy(lockFilePath, pathJoin(options.outputDir, this.packageManager.lockFile)),
                osCommand.changeDirectory(options.outputDir),
                this.packageManager.installCommand.join(' '),
            ]);
        }
        return chain([
            ...(_k = (_j = this.props.commandHooks) === null || _j === void 0 ? void 0 : _j.beforeBundling(options.inputDir, options.outputDir)) !== null && _k !== void 0 ? _k : [],
            tscCommand,
            esbuildCommand.join(' '),
            ...(_m = (this.props.nodeModules && ((_l = this.props.commandHooks) === null || _l === void 0 ? void 0 : _l.beforeInstall(options.inputDir, options.outputDir)))) !== null && _m !== void 0 ? _m : [],
            depsCommand,
            ...(_p = (_o = this.props.commandHooks) === null || _o === void 0 ? void 0 : _o.afterBundling(options.inputDir, options.outputDir)) !== null && _p !== void 0 ? _p : [],
        ]);
    }
    getLocalBundlingProvider() {
        var _a;
        const osPlatform = os.platform();
        const createLocalCommand = (outputDir, esbuild, tsc) => this.createBundlingCommand({
            inputDir: this.projectRoot,
            outputDir,
            esbuildRunner: esbuild.isLocal ? this.packageManager.runBinCommand('esbuild') : 'esbuild',
            tscRunner: tsc && (tsc.isLocal ? this.packageManager.runBinCommand('tsc') : 'tsc'),
            osPlatform,
        });
        const environment = (_a = this.props.environment) !== null && _a !== void 0 ? _a : {};
        const cwd = this.projectRoot;
        return {
            tryBundle(outputDir) {
                if (!Bundling.esbuildInstallation) {
                    process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                    return false;
                }
                if (!Bundling.esbuildInstallation.version.startsWith(`${ESBUILD_MAJOR_VERSION}.`)) {
                    throw new Error(`Expected esbuild version ${ESBUILD_MAJOR_VERSION}.x but got ${Bundling.esbuildInstallation.version}`);
                }
                const localCommand = createLocalCommand(outputDir, Bundling.esbuildInstallation, Bundling.tscInstallation);
                util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                    osPlatform === 'win32' ? '/c' : '-c',
                    localCommand,
                ], {
                    env: { ...process.env, ...environment },
                    stdio: [
                        'ignore',
                        process.stderr,
                        'inherit',
                    ],
                    cwd,
                    windowsVerbatimArguments: osPlatform === 'win32',
                });
                return true;
            },
        };
    }
}
exports.Bundling = Bundling;
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > "${filePath}"`;
        }
        return `echo '${stringifiedData}' > "${filePath}"`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy "${src}" "${dest}"`;
        }
        return `cp "${src}" "${dest}"`;
    }
    changeDirectory(dir) {
        return `cd "${dir}"`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUE2RTtBQUM3RSxxQ0FBcUM7QUFDckMsaUVBQTZEO0FBQzdELHVEQUFtRDtBQUNuRCxtQ0FBdUU7QUFDdkUsaUNBQXVGO0FBRXZGLE1BQU0scUJBQXFCLEdBQUcsR0FBRyxDQUFDO0FBc0NsQzs7R0FFRztBQUNILE1BQWEsUUFBUTtJQXFDbkIsWUFBNkIsS0FBb0I7O1FBQXBCLFVBQUssR0FBTCxLQUFLLENBQWU7UUFDL0MsSUFBSSxDQUFDLGNBQWMsR0FBRyxnQ0FBYyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUUxRSxRQUFRLENBQUMsbUJBQW1CLFNBQUcsUUFBUSxDQUFDLG1CQUFtQixtQ0FBSSwwQ0FBbUIsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDckcsUUFBUSxDQUFDLGVBQWUsU0FBRyxRQUFRLENBQUMsZUFBZSxtQ0FBSSwwQ0FBbUIsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFekYsSUFBSSxDQUFDLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUNwRixJQUFJLENBQUMsd0JBQXdCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztRQUV0RyxJQUFJLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsS0FBSyxDQUFDLGdCQUFnQiw2QkFBNkIsSUFBSSxDQUFDLFdBQVcsS0FBSyxJQUFJLENBQUMsd0JBQXdCLEdBQUcsQ0FBQyxDQUFDO1NBQ3pKO1FBRUQsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2xCLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztTQUMzRjtRQUVELElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQztTQUMxRTtRQUVELElBQUksS0FBSyxDQUFDLE1BQU0sS0FBSyxvQkFBWSxDQUFDLEdBQUc7ZUFDOUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLG9CQUFPLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxPQUFPLEtBQUssb0JBQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUN2RixNQUFNLElBQUksS0FBSyxDQUFDLDJEQUEyRCxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksVUFBVSxDQUFDLENBQUM7U0FDMUc7UUFFRCxJQUFJLENBQUMsU0FBUyxHQUFHO1lBQ2YsU0FBRyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxDQUFDLFNBQVMsQ0FBQztZQUN2QyxTQUFHLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEVBQUU7U0FDM0IsQ0FBQztRQUVGLGtCQUFrQjtRQUNsQixNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxtQkFBbUIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztRQUNwRixJQUFJLENBQUMsS0FBSyxHQUFHLGdCQUFnQixDQUFDLENBQUMsT0FBQyxLQUFLLENBQUMsV0FBVyxtQ0FBSSxHQUFHLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsRUFDM0c7WUFDRSxTQUFTLEVBQUU7Z0JBQ1QsU0FBRyxLQUFLLENBQUMsU0FBUyxtQ0FBSSxFQUFFO2dCQUN4QixLQUFLLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsS0FBSztnQkFDeEMsZUFBZSxRQUFFLEtBQUssQ0FBQyxjQUFjLG1DQUFJLHFCQUFxQjthQUMvRDtZQUNELFFBQVEsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLGNBQWM7U0FDNUMsQ0FBQyxDQUNGLENBQUMsQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLG1DQUFtQztRQUU5RSxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUM7WUFDakQsUUFBUSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsa0JBQWtCO1lBQzdDLFNBQVMsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLG1CQUFtQjtZQUMvQyxhQUFhLEVBQUUsU0FBUztZQUN4QixTQUFTLEVBQUUsS0FBSztZQUNoQixVQUFVLEVBQUUsT0FBTztTQUNwQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsNkVBQTZFO1FBQzdFLGtFQUFrRTtRQUNsRSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsR0FBRyxDQUFDO1FBRTVCLGlCQUFpQjtRQUNqQixJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFLEVBQUUsK0JBQStCO1lBQy9ELElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixFQUFFLENBQUM7U0FDOUM7S0FDRjtJQWxHRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBc0I7UUFDekMsT0FBTyxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFO1lBQ3pDLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUztZQUM1QixhQUFhLEVBQUUsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsTUFBTTtZQUN0RixRQUFRLEVBQUUsSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDO1NBQ2hDLENBQUMsQ0FBQztLQUNKO0lBRU0sTUFBTSxDQUFDLDZCQUE2QjtRQUN6QyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsU0FBUyxDQUFDO0tBQ3RDO0lBRU0sTUFBTSxDQUFDLHlCQUF5QjtRQUNyQyxJQUFJLENBQUMsZUFBZSxHQUFHLFNBQVMsQ0FBQztLQUNsQztJQW1GTyxxQkFBcUIsQ0FBQyxPQUErQjs7UUFDM0QsTUFBTSxRQUFRLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUNoRCxJQUFJLGlCQUFpQixHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNFLElBQUksVUFBVSxHQUFHLEVBQUUsQ0FBQztRQUVwQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxFQUFFO1lBQzdCLE1BQU0sUUFBUSxTQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxhQUFNLENBQUMsZUFBZSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQ2hHLElBQUksQ0FBQyxRQUFRLEVBQUU7Z0JBQ2IsTUFBTSxJQUFJLEtBQUssQ0FBQyx1R0FBdUcsQ0FBQyxDQUFDO2FBQzFIO1lBQ0QsTUFBTSxlQUFlLEdBQUcsaUNBQTBCLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDN0QsVUFBVSxHQUFHLEdBQUcsT0FBTyxDQUFDLFNBQVMsS0FBSyxpQkFBaUIsS0FBSyxlQUFlLEVBQUUsQ0FBQztZQUM5RSxpQkFBaUIsR0FBRyxpQkFBaUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1NBQ3JFO1FBRUQsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sbUNBQUksRUFBRSxDQUFDLENBQUM7UUFDeEQsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sbUNBQUksRUFBRSxDQUFDLENBQUM7UUFFeEQsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsS0FBSyxLQUFLLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDOUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO1NBQ3pFO1FBRUQsTUFBTSxnQkFBZ0IsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsbUNBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFDMUUsTUFBTSxhQUFhLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLG1DQUFJLHFCQUFhLENBQUMsT0FBTyxDQUFDO1FBQ3hFLE1BQU0sY0FBYyxHQUFHLGFBQWEsS0FBSyxxQkFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxFQUFFLENBQUM7UUFDckcsTUFBTSxjQUFjLFNBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLG1DQUFJLElBQUksQ0FBQztRQUV6RCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FBSyxvQkFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUM7UUFDbEYsTUFBTSxjQUFjLEdBQWE7WUFDL0IsT0FBTyxDQUFDLGFBQWE7WUFDckIsVUFBVSxFQUFFLElBQUksaUJBQWlCLEdBQUc7WUFDcEMsWUFBWSxNQUFBLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvRCxpQkFBaUI7WUFDakIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM3RCxjQUFjLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxHQUFHO1lBQ3JELEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDeEMsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDM0QsR0FBRyxjQUFjLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxxQkFBcUIsY0FBYyxFQUFFLENBQUM7WUFDaEUsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLGNBQWMsUUFBUSxFQUFFLENBQUM7WUFDM0QsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLFlBQVksR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDO1lBQzFELEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQyxZQUFZLEdBQUcsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDNUUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNwRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQy9DLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQyxDQUFDLGNBQWMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzNHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxRQUFRLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5RixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNoRixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLGVBQWUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNoRixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLGFBQWEsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ2hFLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7U0FDckYsQ0FBQztRQUVGLElBQUksV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUNyQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQzFCLDJFQUEyRTtZQUMzRSw0QkFBNEI7WUFDNUIsTUFBTSxPQUFPLEdBQUcsYUFBTSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUN2RSxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsOEZBQThGLENBQUMsQ0FBQzthQUNqSDtZQUVELDJEQUEyRDtZQUMzRCxNQUFNLFlBQVksR0FBRywwQkFBbUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUMxRSxNQUFNLFNBQVMsR0FBRyxJQUFJLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7WUFFcEQsTUFBTSxZQUFZLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQyxRQUFRLFFBQUUsSUFBSSxDQUFDLHdCQUF3QixtQ0FBSSxJQUFJLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBRS9HLG9FQUFvRTtZQUNwRSxXQUFXLEdBQUcsS0FBSyxDQUFDO2dCQUNsQixTQUFTLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLGNBQWMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUM7Z0JBQ2xGLFNBQVMsQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3ZGLFNBQVMsQ0FBQyxlQUFlLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQztnQkFDNUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQzthQUM3QyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDO1lBQ1gsZUFBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsY0FBYyxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLFNBQVMsb0NBQUssRUFBRTtZQUNyRixVQUFVO1lBQ1YsY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUM7WUFDeEIsU0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxXQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxhQUFhLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsU0FBUyxFQUFDLENBQUMsbUNBQUksRUFBRTtZQUNoSCxXQUFXO1lBQ1gsZUFBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsYUFBYSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLFNBQVMsb0NBQUssRUFBRTtTQUNyRixDQUFDLENBQUM7S0FDSjtJQUVPLHdCQUF3Qjs7UUFDOUIsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ2pDLE1BQU0sa0JBQWtCLEdBQUcsQ0FBQyxTQUFpQixFQUFFLE9BQTRCLEVBQUUsR0FBeUIsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDO1lBQ3BJLFFBQVEsRUFBRSxJQUFJLENBQUMsV0FBVztZQUMxQixTQUFTO1lBQ1QsYUFBYSxFQUFFLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3pGLFNBQVMsRUFBRSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1lBQ2xGLFVBQVU7U0FDWCxDQUFDLENBQUM7UUFDSCxNQUFNLFdBQVcsU0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRSxDQUFDO1FBQ2pELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUM7UUFFN0IsT0FBTztZQUNMLFNBQVMsQ0FBQyxTQUFpQjtnQkFDekIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRTtvQkFDakMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsNkRBQTZELENBQUMsQ0FBQztvQkFDcEYsT0FBTyxLQUFLLENBQUM7aUJBQ2Q7Z0JBRUQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEdBQUcscUJBQXFCLEdBQUcsQ0FBQyxFQUFFO29CQUNqRixNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixxQkFBcUIsY0FBYyxRQUFRLENBQUMsbUJBQW1CLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztpQkFDeEg7Z0JBRUQsTUFBTSxZQUFZLEdBQUcsa0JBQWtCLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxtQkFBbUIsRUFBRSxRQUFRLENBQUMsZUFBZSxDQUFDLENBQUM7Z0JBRTNHLFdBQUksQ0FDRixVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLE1BQU0sRUFDdkM7b0JBQ0UsVUFBVSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxJQUFJO29CQUNwQyxZQUFZO2lCQUNiLEVBQ0Q7b0JBQ0UsR0FBRyxFQUFFLEVBQUUsR0FBRyxPQUFPLENBQUMsR0FBRyxFQUFFLEdBQUcsV0FBVyxFQUFFO29CQUN2QyxLQUFLLEVBQUU7d0JBQ0wsUUFBUTt3QkFDUixPQUFPLENBQUMsTUFBTTt3QkFDZCxTQUFTO3FCQUNWO29CQUNELEdBQUc7b0JBQ0gsd0JBQXdCLEVBQUUsVUFBVSxLQUFLLE9BQU87aUJBQ2pELENBQUMsQ0FBQztnQkFFTCxPQUFPLElBQUksQ0FBQztZQUNkLENBQUM7U0FDRixDQUFDO0tBQ0g7Q0FDRjtBQXhPRCw0QkF3T0M7QUFVRDs7R0FFRztBQUNILE1BQU0sU0FBUztJQUNiLFlBQTZCLFVBQTJCO1FBQTNCLGVBQVUsR0FBVixVQUFVLENBQWlCO0tBQUk7SUFFckQsU0FBUyxDQUFDLFFBQWdCLEVBQUUsSUFBUztRQUMxQyxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdDLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxPQUFPLEVBQUU7WUFDL0IsT0FBTyxTQUFTLGVBQWUsUUFBUSxRQUFRLEdBQUcsQ0FBQztTQUNwRDtRQUVELE9BQU8sU0FBUyxlQUFlLFFBQVEsUUFBUSxHQUFHLENBQUM7S0FDcEQ7SUFFTSxJQUFJLENBQUMsR0FBVyxFQUFFLElBQVk7UUFDbkMsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLE9BQU8sRUFBRTtZQUMvQixPQUFPLFNBQVMsR0FBRyxNQUFNLElBQUksR0FBRyxDQUFDO1NBQ2xDO1FBRUQsT0FBTyxPQUFPLEdBQUcsTUFBTSxJQUFJLEdBQUcsQ0FBQztLQUNoQztJQUVNLGVBQWUsQ0FBQyxHQUFXO1FBQ2hDLE9BQU8sT0FBTyxHQUFHLEdBQUcsQ0FBQztLQUN0QjtDQUNGO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLEtBQUssQ0FBQyxRQUFrQjtJQUMvQixPQUFPLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQ2hELENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsVUFBVSxDQUFDLFFBQXlCO0lBQzNDLE9BQU8sVUFBUyxHQUFHLEtBQWU7UUFDaEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLEtBQUssQ0FBQyxDQUFDO1FBQ25DLGdEQUFnRDtRQUNoRCxJQUFJLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxPQUFPLElBQUksUUFBUSxLQUFLLE9BQU8sRUFBRTtZQUNyRCxPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1NBQ25DO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQyxDQUFDO0FBQ0osQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxRQUFRLENBQUMsT0FBZ0I7SUFDaEMsTUFBTSxLQUFLLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7SUFFaEQsSUFBSSxDQUFDLEtBQUssRUFBRTtRQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsc0NBQXNDLENBQUMsQ0FBQztLQUN6RDtJQUVELE9BQU8sT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztBQUMzQixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgb3MgZnJvbSAnb3MnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IEFyY2hpdGVjdHVyZSwgQXNzZXRDb2RlLCBDb2RlLCBSdW50aW1lIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBQYWNrYWdlSW5zdGFsbGF0aW9uIH0gZnJvbSAnLi9wYWNrYWdlLWluc3RhbGxhdGlvbic7XG5pbXBvcnQgeyBQYWNrYWdlTWFuYWdlciB9IGZyb20gJy4vcGFja2FnZS1tYW5hZ2VyJztcbmltcG9ydCB7IEJ1bmRsaW5nT3B0aW9ucywgT3V0cHV0Rm9ybWF0LCBTb3VyY2VNYXBNb2RlIH0gZnJvbSAnLi90eXBlcyc7XG5pbXBvcnQgeyBleGVjLCBleHRyYWN0RGVwZW5kZW5jaWVzLCBmaW5kVXAsIGdldFRzY29uZmlnQ29tcGlsZXJPcHRpb25zIH0gZnJvbSAnLi91dGlsJztcblxuY29uc3QgRVNCVUlMRF9NQUpPUl9WRVJTSU9OID0gJzAnO1xuXG4vKipcbiAqIEJ1bmRsaW5nIHByb3BlcnRpZXNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCdW5kbGluZ1Byb3BzIGV4dGVuZHMgQnVuZGxpbmdPcHRpb25zIHtcbiAgLyoqXG4gICAqIFBhdGggdG8gbG9jayBmaWxlXG4gICAqL1xuICByZWFkb25seSBkZXBzTG9ja0ZpbGVQYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEVudHJ5IGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IGVudHJ5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBydW50aW1lIG9mIHRoZSBsYW1iZGEgZnVuY3Rpb25cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU6IFJ1bnRpbWU7XG5cbiAgLyoqXG4gICAqIFRoZSBzeXN0ZW0gYXJjaGl0ZWN0dXJlIG9mIHRoZSBsYW1iZGEgZnVuY3Rpb25cbiAgICovXG4gIHJlYWRvbmx5IGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlO1xuXG4gIC8qKlxuICAgKiBQYXRoIHRvIHByb2plY3Qgcm9vdFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvamVjdFJvb3Q6IHN0cmluZztcblxuICAvKipcbiAgICogUnVuIGNvbXBpbGF0aW9uIHVzaW5nIGB0c2NgIGJlZm9yZSBidW5kbGluZ1xuICAgKi9cbiAgcmVhZG9ubHkgcHJlQ29tcGlsYXRpb24/OiBib29sZWFuXG5cbn1cblxuLyoqXG4gKiBCdW5kbGluZyB3aXRoIGVzYnVpbGRcbiAqL1xuZXhwb3J0IGNsYXNzIEJ1bmRsaW5nIGltcGxlbWVudHMgY2RrLkJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBlc2J1aWxkIGJ1bmRsZWQgTGFtYmRhIGFzc2V0IGNvZGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nUHJvcHMpOiBBc3NldENvZGUge1xuICAgIHJldHVybiBDb2RlLmZyb21Bc3NldChvcHRpb25zLnByb2plY3RSb290LCB7XG4gICAgICBhc3NldEhhc2g6IG9wdGlvbnMuYXNzZXRIYXNoLFxuICAgICAgYXNzZXRIYXNoVHlwZTogb3B0aW9ucy5hc3NldEhhc2ggPyBjZGsuQXNzZXRIYXNoVHlwZS5DVVNUT00gOiBjZGsuQXNzZXRIYXNoVHlwZS5PVVRQVVQsXG4gICAgICBidW5kbGluZzogbmV3IEJ1bmRsaW5nKG9wdGlvbnMpLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBjbGVhckVzYnVpbGRJbnN0YWxsYXRpb25DYWNoZSgpOiB2b2lkIHtcbiAgICB0aGlzLmVzYnVpbGRJbnN0YWxsYXRpb24gPSB1bmRlZmluZWQ7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGNsZWFyVHNjSW5zdGFsbGF0aW9uQ2FjaGUoKTogdm9pZCB7XG4gICAgdGhpcy50c2NJbnN0YWxsYXRpb24gPSB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBlc2J1aWxkSW5zdGFsbGF0aW9uPzogUGFja2FnZUluc3RhbGxhdGlvbjtcbiAgcHJpdmF0ZSBzdGF0aWMgdHNjSW5zdGFsbGF0aW9uPzogUGFja2FnZUluc3RhbGxhdGlvbjtcblxuICAvLyBDb3JlIGJ1bmRsaW5nIG9wdGlvbnNcbiAgcHVibGljIHJlYWRvbmx5IGltYWdlOiBjZGsuRG9ja2VySW1hZ2U7XG4gIHB1YmxpYyByZWFkb25seSBjb21tYW5kOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcHVibGljIHJlYWRvbmx5IHdvcmtpbmdEaXJlY3Rvcnk6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGxvY2FsPzogY2RrLklMb2NhbEJ1bmRsaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcHJvamVjdFJvb3Q6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSByZWxhdGl2ZUVudHJ5UGF0aDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlVHNjb25maWdQYXRoPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHJlbGF0aXZlRGVwc0xvY2tGaWxlUGF0aDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGV4dGVybmFsczogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgcGFja2FnZU1hbmFnZXI6IFBhY2thZ2VNYW5hZ2VyO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEJ1bmRsaW5nUHJvcHMpIHtcbiAgICB0aGlzLnBhY2thZ2VNYW5hZ2VyID0gUGFja2FnZU1hbmFnZXIuZnJvbUxvY2tGaWxlKHByb3BzLmRlcHNMb2NrRmlsZVBhdGgpO1xuXG4gICAgQnVuZGxpbmcuZXNidWlsZEluc3RhbGxhdGlvbiA9IEJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ2VzYnVpbGQnKTtcbiAgICBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPSBCdW5kbGluZy50c2NJbnN0YWxsYXRpb24gPz8gUGFja2FnZUluc3RhbGxhdGlvbi5kZXRlY3QoJ3RzYycpO1xuXG4gICAgdGhpcy5wcm9qZWN0Um9vdCA9IHByb3BzLnByb2plY3RSb290O1xuICAgIHRoaXMucmVsYXRpdmVFbnRyeVBhdGggPSBwYXRoLnJlbGF0aXZlKHRoaXMucHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy5lbnRyeSkpO1xuICAgIHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoID0gcGF0aC5yZWxhdGl2ZSh0aGlzLnByb2plY3RSb290LCBwYXRoLnJlc29sdmUocHJvcHMuZGVwc0xvY2tGaWxlUGF0aCkpO1xuXG4gICAgaWYgKHRoaXMucmVsYXRpdmVEZXBzTG9ja0ZpbGVQYXRoLmluY2x1ZGVzKCcuLicpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEV4cGVjdGVkIGRlcHNMb2NrRmlsZVBhdGg6ICR7cHJvcHMuZGVwc0xvY2tGaWxlUGF0aH0gdG8gYmUgdW5kZXIgcHJvamVjdFJvb3Q6ICR7dGhpcy5wcm9qZWN0Um9vdH0gKCR7dGhpcy5yZWxhdGl2ZURlcHNMb2NrRmlsZVBhdGh9KWApO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy50c2NvbmZpZykge1xuICAgICAgdGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA9IHBhdGgucmVsYXRpdmUodGhpcy5wcm9qZWN0Um9vdCwgcGF0aC5yZXNvbHZlKHByb3BzLnRzY29uZmlnKSk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnByZUNvbXBpbGF0aW9uICYmICEvXFwudHN4PyQvLnRlc3QocHJvcHMuZW50cnkpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3ByZUNvbXBpbGF0aW9uIGNhbiBvbmx5IGJlIHVzZWQgd2l0aCB0eXBlc2NyaXB0IGZpbGVzJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmZvcm1hdCA9PT0gT3V0cHV0Rm9ybWF0LkVTTVxuICAgICAgICAmJiAocHJvcHMucnVudGltZSA9PT0gUnVudGltZS5OT0RFSlNfMTBfWCB8fCBwcm9wcy5ydW50aW1lID09PSBSdW50aW1lLk5PREVKU18xMl9YKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBFQ01BU2NyaXB0IG1vZHVsZSBvdXRwdXQgZm9ybWF0IGlzIG5vdCBzdXBwb3J0ZWQgYnkgdGhlICR7cHJvcHMucnVudGltZS5uYW1lfSBydW50aW1lYCk7XG4gICAgfVxuXG4gICAgdGhpcy5leHRlcm5hbHMgPSBbXG4gICAgICAuLi5wcm9wcy5leHRlcm5hbE1vZHVsZXMgPz8gWydhd3Mtc2RrJ10sIC8vIE1hcmsgYXdzLXNkayBhcyBleHRlcm5hbCBieSBkZWZhdWx0IChhdmFpbGFibGUgaW4gdGhlIHJ1bnRpbWUpXG4gICAgICAuLi5wcm9wcy5ub2RlTW9kdWxlcyA/PyBbXSwgLy8gTWFyayB0aGUgbW9kdWxlcyB0aGF0IHdlIGFyZSBnb2luZyB0byBpbnN0YWxsIGFzIGV4dGVybmFscyBhbHNvXG4gICAgXTtcblxuICAgIC8vIERvY2tlciBidW5kbGluZ1xuICAgIGNvbnN0IHNob3VsZEJ1aWxkSW1hZ2UgPSBwcm9wcy5mb3JjZURvY2tlckJ1bmRsaW5nIHx8ICFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uO1xuICAgIHRoaXMuaW1hZ2UgPSBzaG91bGRCdWlsZEltYWdlID8gcHJvcHMuZG9ja2VySW1hZ2UgPz8gY2RrLkRvY2tlckltYWdlLmZyb21CdWlsZChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGliJyksXG4gICAgICB7XG4gICAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICAgIC4uLnByb3BzLmJ1aWxkQXJncyA/PyB7fSxcbiAgICAgICAgICBJTUFHRTogcHJvcHMucnVudGltZS5idW5kbGluZ0ltYWdlLmltYWdlLFxuICAgICAgICAgIEVTQlVJTERfVkVSU0lPTjogcHJvcHMuZXNidWlsZFZlcnNpb24gPz8gRVNCVUlMRF9NQUpPUl9WRVJTSU9OLFxuICAgICAgICB9LFxuICAgICAgICBwbGF0Zm9ybTogcHJvcHMuYXJjaGl0ZWN0dXJlLmRvY2tlclBsYXRmb3JtLFxuICAgICAgfSlcbiAgICAgIDogY2RrLkRvY2tlckltYWdlLmZyb21SZWdpc3RyeSgnZHVtbXknKTsgLy8gRG8gbm90IGJ1aWxkIGlmIHdlIGRvbid0IG5lZWQgdG9cblxuICAgIGNvbnN0IGJ1bmRsaW5nQ29tbWFuZCA9IHRoaXMuY3JlYXRlQnVuZGxpbmdDb21tYW5kKHtcbiAgICAgIGlucHV0RGlyOiBjZGsuQXNzZXRTdGFnaW5nLkJVTkRMSU5HX0lOUFVUX0RJUixcbiAgICAgIG91dHB1dERpcjogY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19PVVRQVVRfRElSLFxuICAgICAgZXNidWlsZFJ1bm5lcjogJ2VzYnVpbGQnLCAvLyBlc2J1aWxkIGlzIGluc3RhbGxlZCBnbG9iYWxseSBpbiB0aGUgZG9ja2VyIGltYWdlXG4gICAgICB0c2NSdW5uZXI6ICd0c2MnLCAvLyB0c2MgaXMgaW5zdGFsbGVkIGdsb2JhbGx5IGluIHRoZSBkb2NrZXIgaW1hZ2VcbiAgICAgIG9zUGxhdGZvcm06ICdsaW51eCcsIC8vIGxpbnV4IGRvY2tlciBpbWFnZVxuICAgIH0pO1xuICAgIHRoaXMuY29tbWFuZCA9IFsnYmFzaCcsICctYycsIGJ1bmRsaW5nQ29tbWFuZF07XG4gICAgdGhpcy5lbnZpcm9ubWVudCA9IHByb3BzLmVudmlyb25tZW50O1xuICAgIC8vIEJ1bmRsaW5nIHNldHMgdGhlIHdvcmtpbmcgZGlyZWN0b3J5IHRvIGNkay5Bc3NldFN0YWdpbmcuQlVORExJTkdfSU5QVVRfRElSXG4gICAgLy8gYW5kIHdlIHdhbnQgdG8gZm9yY2UgbnB4IHRvIHVzZSB0aGUgZ2xvYmFsbHkgaW5zdGFsbGVkIGVzYnVpbGQuXG4gICAgdGhpcy53b3JraW5nRGlyZWN0b3J5ID0gJy8nO1xuXG4gICAgLy8gTG9jYWwgYnVuZGxpbmdcbiAgICBpZiAoIXByb3BzLmZvcmNlRG9ja2VyQnVuZGxpbmcpIHsgLy8gb25seSBpZiBEb2NrZXIgaXMgbm90IGZvcmNlZFxuICAgICAgdGhpcy5sb2NhbCA9IHRoaXMuZ2V0TG9jYWxCdW5kbGluZ1Byb3ZpZGVyKCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVCdW5kbGluZ0NvbW1hbmQob3B0aW9uczogQnVuZGxpbmdDb21tYW5kT3B0aW9ucyk6IHN0cmluZyB7XG4gICAgY29uc3QgcGF0aEpvaW4gPSBvc1BhdGhKb2luKG9wdGlvbnMub3NQbGF0Zm9ybSk7XG4gICAgbGV0IHJlbGF0aXZlRW50cnlQYXRoID0gcGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgdGhpcy5yZWxhdGl2ZUVudHJ5UGF0aCk7XG4gICAgbGV0IHRzY0NvbW1hbmQgPSAnJztcblxuICAgIGlmICh0aGlzLnByb3BzLnByZUNvbXBpbGF0aW9uKSB7XG4gICAgICBjb25zdCB0c2NvbmZpZyA9IHRoaXMucHJvcHMudHNjb25maWcgPz8gZmluZFVwKCd0c2NvbmZpZy5qc29uJywgcGF0aC5kaXJuYW1lKHRoaXMucHJvcHMuZW50cnkpKTtcbiAgICAgIGlmICghdHNjb25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBhIGB0c2NvbmZpZy5qc29uYCBidXQgYHByZUNvbXBpbGF0aW9uYCBpcyBzZXQgdG8gYHRydWVgLCBwbGVhc2Ugc3BlY2lmeSBpdCB2aWEgYHRzY29uZmlnYCcpO1xuICAgICAgfVxuICAgICAgY29uc3QgY29tcGlsZXJPcHRpb25zID0gZ2V0VHNjb25maWdDb21waWxlck9wdGlvbnModHNjb25maWcpO1xuICAgICAgdHNjQ29tbWFuZCA9IGAke29wdGlvbnMudHNjUnVubmVyfSBcIiR7cmVsYXRpdmVFbnRyeVBhdGh9XCIgJHtjb21waWxlck9wdGlvbnN9YDtcbiAgICAgIHJlbGF0aXZlRW50cnlQYXRoID0gcmVsYXRpdmVFbnRyeVBhdGgucmVwbGFjZSgvXFwudHMoeD8pJC8sICcuanMkMScpO1xuICAgIH1cblxuICAgIGNvbnN0IGxvYWRlcnMgPSBPYmplY3QuZW50cmllcyh0aGlzLnByb3BzLmxvYWRlciA/PyB7fSk7XG4gICAgY29uc3QgZGVmaW5lcyA9IE9iamVjdC5lbnRyaWVzKHRoaXMucHJvcHMuZGVmaW5lID8/IHt9KTtcblxuICAgIGlmICh0aGlzLnByb3BzLnNvdXJjZU1hcCA9PT0gZmFsc2UgJiYgdGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3NvdXJjZU1hcE1vZGUgY2Fubm90IGJlIHVzZWQgd2hlbiBzb3VyY2VNYXAgaXMgZmFsc2UnKTtcbiAgICB9XG5cbiAgICBjb25zdCBzb3VyY2VNYXBFbmFibGVkID0gdGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlID8/IHRoaXMucHJvcHMuc291cmNlTWFwO1xuICAgIGNvbnN0IHNvdXJjZU1hcE1vZGUgPSB0aGlzLnByb3BzLnNvdXJjZU1hcE1vZGUgPz8gU291cmNlTWFwTW9kZS5ERUZBVUxUO1xuICAgIGNvbnN0IHNvdXJjZU1hcFZhbHVlID0gc291cmNlTWFwTW9kZSA9PT0gU291cmNlTWFwTW9kZS5ERUZBVUxUID8gJycgOiBgPSR7dGhpcy5wcm9wcy5zb3VyY2VNYXBNb2RlfWA7XG4gICAgY29uc3Qgc291cmNlc0NvbnRlbnQgPSB0aGlzLnByb3BzLnNvdXJjZXNDb250ZW50ID8/IHRydWU7XG5cbiAgICBjb25zdCBvdXRGaWxlID0gdGhpcy5wcm9wcy5mb3JtYXQgPT09IE91dHB1dEZvcm1hdC5FU00gPyAnaW5kZXgubWpzJyA6ICdpbmRleC5qcyc7XG4gICAgY29uc3QgZXNidWlsZENvbW1hbmQ6IHN0cmluZ1tdID0gW1xuICAgICAgb3B0aW9ucy5lc2J1aWxkUnVubmVyLFxuICAgICAgJy0tYnVuZGxlJywgYFwiJHtyZWxhdGl2ZUVudHJ5UGF0aH1cImAsXG4gICAgICBgLS10YXJnZXQ9JHt0aGlzLnByb3BzLnRhcmdldCA/PyB0b1RhcmdldCh0aGlzLnByb3BzLnJ1bnRpbWUpfWAsXG4gICAgICAnLS1wbGF0Zm9ybT1ub2RlJyxcbiAgICAgIC4uLnRoaXMucHJvcHMuZm9ybWF0ID8gW2AtLWZvcm1hdD0ke3RoaXMucHJvcHMuZm9ybWF0fWBdIDogW10sXG4gICAgICBgLS1vdXRmaWxlPVwiJHtwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgb3V0RmlsZSl9XCJgLFxuICAgICAgLi4udGhpcy5wcm9wcy5taW5pZnkgPyBbJy0tbWluaWZ5J10gOiBbXSxcbiAgICAgIC4uLnNvdXJjZU1hcEVuYWJsZWQgPyBbYC0tc291cmNlbWFwJHtzb3VyY2VNYXBWYWx1ZX1gXSA6IFtdLFxuICAgICAgLi4uc291cmNlc0NvbnRlbnQgPyBbXSA6IFtgLS1zb3VyY2VzLWNvbnRlbnQ9JHtzb3VyY2VzQ29udGVudH1gXSxcbiAgICAgIC4uLnRoaXMuZXh0ZXJuYWxzLm1hcChleHRlcm5hbCA9PiBgLS1leHRlcm5hbDoke2V4dGVybmFsfWApLFxuICAgICAgLi4ubG9hZGVycy5tYXAoKFtleHQsIG5hbWVdKSA9PiBgLS1sb2FkZXI6JHtleHR9PSR7bmFtZX1gKSxcbiAgICAgIC4uLmRlZmluZXMubWFwKChba2V5LCB2YWx1ZV0pID0+IGAtLWRlZmluZToke2tleX09JHtKU09OLnN0cmluZ2lmeSh2YWx1ZSl9YCksXG4gICAgICAuLi50aGlzLnByb3BzLmxvZ0xldmVsID8gW2AtLWxvZy1sZXZlbD0ke3RoaXMucHJvcHMubG9nTGV2ZWx9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMua2VlcE5hbWVzID8gWyctLWtlZXAtbmFtZXMnXSA6IFtdLFxuICAgICAgLi4udGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA/IFtgLS10c2NvbmZpZz0ke3BhdGhKb2luKG9wdGlvbnMuaW5wdXREaXIsIHRoaXMucmVsYXRpdmVUc2NvbmZpZ1BhdGgpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLm1ldGFmaWxlID8gW2AtLW1ldGFmaWxlPSR7cGF0aEpvaW4ob3B0aW9ucy5vdXRwdXREaXIsICdpbmRleC5tZXRhLmpzb24nKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5iYW5uZXIgPyBbYC0tYmFubmVyOmpzPSR7SlNPTi5zdHJpbmdpZnkodGhpcy5wcm9wcy5iYW5uZXIpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmZvb3RlciA/IFtgLS1mb290ZXI6anM9JHtKU09OLnN0cmluZ2lmeSh0aGlzLnByb3BzLmZvb3Rlcil9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMuY2hhcnNldCA/IFtgLS1jaGFyc2V0PSR7dGhpcy5wcm9wcy5jaGFyc2V0fWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLm1haW5GaWVsZHMgPyBbYC0tbWFpbi1maWVsZHM9JHt0aGlzLnByb3BzLm1haW5GaWVsZHMuam9pbignLCcpfWBdIDogW10sXG4gICAgXTtcblxuICAgIGxldCBkZXBzQ29tbWFuZCA9ICcnO1xuICAgIGlmICh0aGlzLnByb3BzLm5vZGVNb2R1bGVzKSB7XG4gICAgICAvLyBGaW5kICdwYWNrYWdlLmpzb24nIGNsb3Nlc3QgdG8gZW50cnkgZm9sZGVyLCB3ZSBhcmUgZ29pbmcgdG8gZXh0cmFjdCB0aGVcbiAgICAgIC8vIG1vZHVsZXMgdmVyc2lvbnMgZnJvbSBpdC5cbiAgICAgIGNvbnN0IHBrZ1BhdGggPSBmaW5kVXAoJ3BhY2thZ2UuanNvbicsIHBhdGguZGlybmFtZSh0aGlzLnByb3BzLmVudHJ5KSk7XG4gICAgICBpZiAoIXBrZ1BhdGgpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBhIGBwYWNrYWdlLmpzb25gIGluIHRoaXMgcHJvamVjdC4gVXNpbmcgYG5vZGVNb2R1bGVzYCByZXF1aXJlcyBhIGBwYWNrYWdlLmpzb25gLicpO1xuICAgICAgfVxuXG4gICAgICAvLyBEZXRlcm1pbmUgZGVwZW5kZW5jaWVzIHZlcnNpb25zLCBsb2NrIGZpbGUgYW5kIGluc3RhbGxlclxuICAgICAgY29uc3QgZGVwZW5kZW5jaWVzID0gZXh0cmFjdERlcGVuZGVuY2llcyhwa2dQYXRoLCB0aGlzLnByb3BzLm5vZGVNb2R1bGVzKTtcbiAgICAgIGNvbnN0IG9zQ29tbWFuZCA9IG5ldyBPc0NvbW1hbmQob3B0aW9ucy5vc1BsYXRmb3JtKTtcblxuICAgICAgY29uc3QgbG9ja0ZpbGVQYXRoID0gcGF0aEpvaW4ob3B0aW9ucy5pbnB1dERpciwgdGhpcy5yZWxhdGl2ZURlcHNMb2NrRmlsZVBhdGggPz8gdGhpcy5wYWNrYWdlTWFuYWdlci5sb2NrRmlsZSk7XG5cbiAgICAgIC8vIENyZWF0ZSBkdW1teSBwYWNrYWdlLmpzb24sIGNvcHkgbG9jayBmaWxlIGlmIGFueSBhbmQgdGhlbiBpbnN0YWxsXG4gICAgICBkZXBzQ29tbWFuZCA9IGNoYWluKFtcbiAgICAgICAgb3NDb21tYW5kLndyaXRlSnNvbihwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgJ3BhY2thZ2UuanNvbicpLCB7IGRlcGVuZGVuY2llcyB9KSxcbiAgICAgICAgb3NDb21tYW5kLmNvcHkobG9ja0ZpbGVQYXRoLCBwYXRoSm9pbihvcHRpb25zLm91dHB1dERpciwgdGhpcy5wYWNrYWdlTWFuYWdlci5sb2NrRmlsZSkpLFxuICAgICAgICBvc0NvbW1hbmQuY2hhbmdlRGlyZWN0b3J5KG9wdGlvbnMub3V0cHV0RGlyKSxcbiAgICAgICAgdGhpcy5wYWNrYWdlTWFuYWdlci5pbnN0YWxsQ29tbWFuZC5qb2luKCcgJyksXG4gICAgICBdKTtcbiAgICB9XG5cbiAgICByZXR1cm4gY2hhaW4oW1xuICAgICAgLi4udGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmJlZm9yZUJ1bmRsaW5nKG9wdGlvbnMuaW5wdXREaXIsIG9wdGlvbnMub3V0cHV0RGlyKSA/PyBbXSxcbiAgICAgIHRzY0NvbW1hbmQsXG4gICAgICBlc2J1aWxkQ29tbWFuZC5qb2luKCcgJyksXG4gICAgICAuLi4odGhpcy5wcm9wcy5ub2RlTW9kdWxlcyAmJiB0aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYmVmb3JlSW5zdGFsbChvcHRpb25zLmlucHV0RGlyLCBvcHRpb25zLm91dHB1dERpcikpID8/IFtdLFxuICAgICAgZGVwc0NvbW1hbmQsXG4gICAgICAuLi50aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYWZ0ZXJCdW5kbGluZyhvcHRpb25zLmlucHV0RGlyLCBvcHRpb25zLm91dHB1dERpcikgPz8gW10sXG4gICAgXSk7XG4gIH1cblxuICBwcml2YXRlIGdldExvY2FsQnVuZGxpbmdQcm92aWRlcigpOiBjZGsuSUxvY2FsQnVuZGxpbmcge1xuICAgIGNvbnN0IG9zUGxhdGZvcm0gPSBvcy5wbGF0Zm9ybSgpO1xuICAgIGNvbnN0IGNyZWF0ZUxvY2FsQ29tbWFuZCA9IChvdXRwdXREaXI6IHN0cmluZywgZXNidWlsZDogUGFja2FnZUluc3RhbGxhdGlvbiwgdHNjPzogUGFja2FnZUluc3RhbGxhdGlvbikgPT4gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQoe1xuICAgICAgaW5wdXREaXI6IHRoaXMucHJvamVjdFJvb3QsXG4gICAgICBvdXRwdXREaXIsXG4gICAgICBlc2J1aWxkUnVubmVyOiBlc2J1aWxkLmlzTG9jYWwgPyB0aGlzLnBhY2thZ2VNYW5hZ2VyLnJ1bkJpbkNvbW1hbmQoJ2VzYnVpbGQnKSA6ICdlc2J1aWxkJyxcbiAgICAgIHRzY1J1bm5lcjogdHNjICYmICh0c2MuaXNMb2NhbCA/IHRoaXMucGFja2FnZU1hbmFnZXIucnVuQmluQ29tbWFuZCgndHNjJykgOiAndHNjJyksXG4gICAgICBvc1BsYXRmb3JtLFxuICAgIH0pO1xuICAgIGNvbnN0IGVudmlyb25tZW50ID0gdGhpcy5wcm9wcy5lbnZpcm9ubWVudCA/PyB7fTtcbiAgICBjb25zdCBjd2QgPSB0aGlzLnByb2plY3RSb290O1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHRyeUJ1bmRsZShvdXRwdXREaXI6IHN0cmluZykge1xuICAgICAgICBpZiAoIUJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24pIHtcbiAgICAgICAgICBwcm9jZXNzLnN0ZGVyci53cml0ZSgnZXNidWlsZCBjYW5ub3QgcnVuIGxvY2FsbHkuIFN3aXRjaGluZyB0byBEb2NrZXIgYnVuZGxpbmcuXFxuJyk7XG4gICAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKCFCdW5kbGluZy5lc2J1aWxkSW5zdGFsbGF0aW9uLnZlcnNpb24uc3RhcnRzV2l0aChgJHtFU0JVSUxEX01BSk9SX1ZFUlNJT059LmApKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBFeHBlY3RlZCBlc2J1aWxkIHZlcnNpb24gJHtFU0JVSUxEX01BSk9SX1ZFUlNJT059LnggYnV0IGdvdCAke0J1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24udmVyc2lvbn1gKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IGxvY2FsQ29tbWFuZCA9IGNyZWF0ZUxvY2FsQ29tbWFuZChvdXRwdXREaXIsIEJ1bmRsaW5nLmVzYnVpbGRJbnN0YWxsYXRpb24sIEJ1bmRsaW5nLnRzY0luc3RhbGxhdGlvbik7XG5cbiAgICAgICAgZXhlYyhcbiAgICAgICAgICBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ2NtZCcgOiAnYmFzaCcsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgb3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJyA/ICcvYycgOiAnLWMnLFxuICAgICAgICAgICAgbG9jYWxDb21tYW5kLFxuICAgICAgICAgIF0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgZW52OiB7IC4uLnByb2Nlc3MuZW52LCAuLi5lbnZpcm9ubWVudCB9LFxuICAgICAgICAgICAgc3RkaW86IFsgLy8gc2hvdyBvdXRwdXRcbiAgICAgICAgICAgICAgJ2lnbm9yZScsIC8vIGlnbm9yZSBzdGRpb1xuICAgICAgICAgICAgICBwcm9jZXNzLnN0ZGVyciwgLy8gcmVkaXJlY3Qgc3Rkb3V0IHRvIHN0ZGVyclxuICAgICAgICAgICAgICAnaW5oZXJpdCcsIC8vIGluaGVyaXQgc3RkZXJyXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgY3dkLFxuICAgICAgICAgICAgd2luZG93c1ZlcmJhdGltQXJndW1lbnRzOiBvc1BsYXRmb3JtID09PSAnd2luMzInLFxuICAgICAgICAgIH0pO1xuXG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbmludGVyZmFjZSBCdW5kbGluZ0NvbW1hbmRPcHRpb25zIHtcbiAgcmVhZG9ubHkgaW5wdXREaXI6IHN0cmluZztcbiAgcmVhZG9ubHkgb3V0cHV0RGlyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGVzYnVpbGRSdW5uZXI6IHN0cmluZztcbiAgcmVhZG9ubHkgdHNjUnVubmVyPzogc3RyaW5nO1xuICByZWFkb25seSBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm07XG59XG5cbi8qKlxuICogT1MgYWdub3N0aWMgY29tbWFuZFxuICovXG5jbGFzcyBPc0NvbW1hbmQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IG9zUGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge31cblxuICBwdWJsaWMgd3JpdGVKc29uKGZpbGVQYXRoOiBzdHJpbmcsIGRhdGE6IGFueSk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RyaW5naWZpZWREYXRhID0gSlNPTi5zdHJpbmdpZnkoZGF0YSk7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBlY2hvIF4ke3N0cmluZ2lmaWVkRGF0YX1eID4gXCIke2ZpbGVQYXRofVwiYDtcbiAgICB9XG5cbiAgICByZXR1cm4gYGVjaG8gJyR7c3RyaW5naWZpZWREYXRhfScgPiBcIiR7ZmlsZVBhdGh9XCJgO1xuICB9XG5cbiAgcHVibGljIGNvcHkoc3JjOiBzdHJpbmcsIGRlc3Q6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBjb3B5IFwiJHtzcmN9XCIgXCIke2Rlc3R9XCJgO1xuICAgIH1cblxuICAgIHJldHVybiBgY3AgXCIke3NyY31cIiBcIiR7ZGVzdH1cImA7XG4gIH1cblxuICBwdWJsaWMgY2hhbmdlRGlyZWN0b3J5KGRpcjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gYGNkIFwiJHtkaXJ9XCJgO1xuICB9XG59XG5cbi8qKlxuICogQ2hhaW4gY29tbWFuZHNcbiAqL1xuZnVuY3Rpb24gY2hhaW4oY29tbWFuZHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgcmV0dXJuIGNvbW1hbmRzLmZpbHRlcihjID0+ICEhYykuam9pbignICYmICcpO1xufVxuXG4vKipcbiAqIFBsYXRmb3JtIHNwZWNpZmljIHBhdGggam9pblxuICovXG5mdW5jdGlvbiBvc1BhdGhKb2luKHBsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0pIHtcbiAgcmV0dXJuIGZ1bmN0aW9uKC4uLnBhdGhzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gICAgY29uc3Qgam9pbmVkID0gcGF0aC5qb2luKC4uLnBhdGhzKTtcbiAgICAvLyBJZiB3ZSBhcmUgb24gd2luMzIgYnV0IG5lZWQgcG9zaXggc3R5bGUgcGF0aHNcbiAgICBpZiAob3MucGxhdGZvcm0oKSA9PT0gJ3dpbjMyJyAmJiBwbGF0Zm9ybSAhPT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGpvaW5lZC5yZXBsYWNlKC9cXFxcL2csICcvJyk7XG4gICAgfVxuICAgIHJldHVybiBqb2luZWQ7XG4gIH07XG59XG5cbi8qKlxuICogQ29udmVydHMgYSBydW50aW1lIHRvIGFuIGVzYnVpbGQgbm9kZSB0YXJnZXRcbiAqL1xuZnVuY3Rpb24gdG9UYXJnZXQocnVudGltZTogUnVudGltZSk6IHN0cmluZyB7XG4gIGNvbnN0IG1hdGNoID0gcnVudGltZS5uYW1lLm1hdGNoKC9ub2RlanMoXFxkKykvKTtcblxuICBpZiAoIW1hdGNoKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZXh0cmFjdCB2ZXJzaW9uIGZyb20gcnVudGltZS4nKTtcbiAgfVxuXG4gIHJldHVybiBgbm9kZSR7bWF0Y2hbMV19YDtcbn1cbiJdfQ==