"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageManager = exports.LockFile = void 0;
const os = require("os");
const path = require("path");
const types_1 = require("./types");
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
    LockFile["PNPM"] = "pnpm-lock.yaml";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
/**
 * A node package manager
 */
class PackageManager {
    constructor(props) {
        this.lockFile = props.lockFile;
        this.installCommand = props.installCommand;
        this.runCommand = props.runCommand;
        this.argsSeparator = props.argsSeparator;
    }
    /**
     * Use a lock file path to determine the package manager to use. Optionally, specify a log level to
     * control its verbosity.
     * @param lockFilePath Path of the lock file
     * @param logLevel optional log level @default LogLevel.INFO
     * @returns the right PackageManager for that lock file
     */
    static fromLockFile(lockFilePath, logLevel) {
        const lockFile = path.basename(lockFilePath);
        switch (lockFile) {
            case LockFile.YARN:
                return new PackageManager({
                    lockFile: LockFile.YARN,
                    installCommand: logLevel && logLevel !== types_1.LogLevel.INFO ? ['yarn', 'install', '--no-immutable', '--silent'] : ['yarn', 'install', '--no-immutable'],
                    runCommand: ['yarn', 'run'],
                });
            case LockFile.PNPM:
                return new PackageManager({
                    lockFile: LockFile.PNPM,
                    installCommand: logLevel && logLevel !== types_1.LogLevel.INFO ? ['pnpm', 'install', '--reporter', 'silent'] : ['pnpm', 'install'],
                    runCommand: ['pnpm', 'exec'],
                    argsSeparator: '--',
                });
            default:
                return new PackageManager({
                    lockFile: LockFile.NPM,
                    installCommand: logLevel ? ['npm', 'ci', '--loglevel', logLevel] : ['npm', 'ci'],
                    runCommand: ['npx', '--no-install'],
                });
        }
    }
    runBinCommand(bin) {
        const [runCommand, ...runArgs] = this.runCommand;
        return [
            os.platform() === 'win32' ? `${runCommand}.cmd` : runCommand,
            ...runArgs,
            ...(this.argsSeparator ? [this.argsSeparator] : []),
            bin,
        ].join(' ');
    }
}
exports.PackageManager = PackageManager;
//# sourceMappingURL=data:application/json;base64,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