"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const build_1 = require("./build");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'handler';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = props.runtime || defaultRunTime;
        // Build with Parcel
        build_1.build({
            entry,
            outDir: handlerDir,
            global: handler,
            minify: props.minify,
            sourceMaps: props.sourceMaps,
            cacheDir: props.cacheDir,
            nodeVersion: extractVersion(runtime),
        });
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        return undefined;
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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