"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Builder
 */
class Builder {
    constructor(options) {
        this.options = options;
        let parcelPkgPath;
        try {
            parcelPkgPath = require.resolve('parcel-bundler/package.json'); // This will throw if `parcel-bundler` cannot be found
        }
        catch (err) {
            throw new Error('It looks like parcel-bundler is not installed. Please install v1.x of parcel-bundler with yarn or npm.');
        }
        const parcelDir = path.dirname(parcelPkgPath);
        const parcelPkg = JSON.parse(fs.readFileSync(parcelPkgPath, 'utf8'));
        if (!parcelPkg.version || !/^1\./.test(parcelPkg.version)) { // Peer dependency on parcel v1.x
            throw new Error(`This module has a peer dependency on parcel-bundler v1.x. Got v${parcelPkg.version}.`);
        }
        this.parcelBinPath = path.join(parcelDir, parcelPkg.bin.parcel);
    }
    build() {
        const pkgPath = util_1.findPkgPath();
        let originalPkg;
        try {
            if (this.options.nodeVersion && pkgPath) {
                // Update engines.node (Babel target)
                originalPkg = util_1.updatePkg(pkgPath, {
                    engines: { node: `>= ${this.options.nodeVersion}` }
                });
            }
            const args = [
                'build', this.options.entry,
                '--out-dir', this.options.outDir,
                '--out-file', 'index.js',
                '--global', this.options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !this.options.minify && '--no-minify',
                !this.options.sourceMaps && '--no-source-maps',
                ...this.options.cacheDir
                    ? ['--cache-dir', this.options.cacheDir]
                    : [],
            ].filter(Boolean);
            const parcel = child_process_1.spawnSync(this.parcelBinPath, args);
            if (parcel.error) {
                throw parcel.error;
            }
            if (parcel.status !== 0) {
                throw new Error(parcel.stdout.toString().trim());
            }
        }
        catch (err) {
            throw new Error(`Failed to build file at ${this.options.entry}: ${err}`);
        }
        finally { // Always restore package.json to original
            if (pkgPath && originalPkg) {
                fs.writeFileSync(pkgPath, originalPkg);
            }
        }
    }
}
exports.Builder = Builder;
//# sourceMappingURL=data:application/json;base64,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