"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const builder_1 = require("../lib/builder");
let parcelPkgPath;
let parcelPkg;
beforeAll(() => {
    parcelPkgPath = require.resolve('parcel-bundler/package.json');
    parcelPkg = fs.readFileSync(parcelPkgPath);
});
afterEach(() => {
    fs.writeFileSync(parcelPkgPath, parcelPkg);
});
jest.mock('child_process', () => ({
    spawnSync: jest.fn((_cmd, args) => {
        if (args[1] === 'error') {
            return { error: 'parcel-error' };
        }
        if (args[1] === 'status') {
            return { status: 1, stdout: Buffer.from('status-error') };
        }
        return { error: null, status: 0 };
    })
}));
test('calls parcel with the correct args', () => {
    const builder = new builder_1.Builder({
        entry: 'entry',
        global: 'handler',
        outDir: 'out-dir',
        cacheDir: 'cache-dir',
    });
    builder.build();
    expect(child_process_1.spawnSync).toHaveBeenCalledWith(expect.stringContaining('parcel-bundler'), expect.arrayContaining([
        'build', 'entry',
        '--out-dir', 'out-dir',
        '--out-file', 'index.js',
        '--global', 'handler',
        '--target', 'node',
        '--bundle-node-modules',
        '--log-level', '2',
        '--no-minify',
        '--no-source-maps',
        '--cache-dir', 'cache-dir'
    ]));
});
test('throws in case of error', () => {
    const builder = new builder_1.Builder({
        entry: 'error',
        global: 'handler',
        outDir: 'out-dir'
    });
    expect(() => builder.build()).toThrow('parcel-error');
});
test('throws if status is not 0', () => {
    const builder = new builder_1.Builder({
        entry: 'status',
        global: 'handler',
        outDir: 'out-dir'
    });
    expect(() => builder.build()).toThrow('status-error');
});
test('throws when parcel-bundler is not 1.x', () => {
    fs.writeFileSync(parcelPkgPath, JSON.stringify({
        ...JSON.parse(parcelPkg.toString()),
        version: '2.3.4'
    }));
    expect(() => new builder_1.Builder({
        entry: 'entry',
        global: 'handler',
        outDir: 'out-dur'
    })).toThrow(/This module has a peer dependency on parcel-bundler v1.x. Got v2.3.4./);
});
//# sourceMappingURL=data:application/json;base64,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