"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const builder_1 = require("./builder");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'handler';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = props.runtime || defaultRunTime;
        // Build with Parcel
        const builder = new builder_1.Builder({
            entry,
            outDir: handlerDir,
            global: handler,
            minify: props.minify,
            sourceMaps: props.sourceMaps,
            cacheDir: props.cacheDir,
            nodeVersion: extractVersion(runtime),
        });
        builder.build();
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        return undefined;
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmdW5jdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLDhDQUE4QztBQUU5QyxpQ0FBaUM7QUFDakMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3Qix1Q0FBb0M7QUFDcEMsaUNBQTJEO0FBK0QzRDs7R0FFRztBQUNILE1BQWEsY0FBZSxTQUFRLE1BQU0sQ0FBQyxRQUFRO0lBQ2pELFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsUUFBNkIsRUFBRTtRQUMzRSxJQUFJLEtBQUssQ0FBQyxPQUFPLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEtBQUssTUFBTSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEVBQUU7WUFDekUsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekMsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxTQUFTLENBQUM7UUFDM0MsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDNUUsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDaEcsTUFBTSxjQUFjLEdBQUcsdUJBQWdCLEVBQUUsSUFBSSxFQUFFO1lBQzdDLENBQUMsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDNUIsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQy9CLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksY0FBYyxDQUFDO1FBRWhELG9CQUFvQjtRQUNwQixNQUFNLE9BQU8sR0FBRyxJQUFJLGlCQUFPLENBQUM7WUFDMUIsS0FBSztZQUNMLE1BQU0sRUFBRSxVQUFVO1lBQ2xCLE1BQU0sRUFBRSxPQUFPO1lBQ2YsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3BCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsV0FBVyxFQUFFLGNBQWMsQ0FBQyxPQUFPLENBQUM7U0FDckMsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRWhCLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFO1lBQ2YsR0FBRyxLQUFLO1lBQ1IsT0FBTztZQUNQLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7WUFDdkMsT0FBTyxFQUFFLFNBQVMsT0FBTyxFQUFFO1NBQzVCLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQWxDRCx3Q0FrQ0M7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsU0FBUyxDQUFDLEVBQVUsRUFBRSxLQUFjO0lBQzNDLElBQUksS0FBSyxFQUFFO1FBQ1QsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDN0IsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsQ0FBQyxDQUFDO1NBQzdFO1FBQ0QsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDekIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsS0FBSyxFQUFFLENBQUMsQ0FBQztTQUN2RDtRQUNELE9BQU8sS0FBSyxDQUFDO0tBQ2Q7SUFFRCxNQUFNLFlBQVksR0FBRyxnQkFBZ0IsRUFBRSxDQUFDO0lBQ3hDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFM0MsTUFBTSxhQUFhLEdBQUcsWUFBWSxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLE9BQU8sR0FBRyxDQUFDLEVBQUUsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ25GLElBQUksRUFBRSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsRUFBRTtRQUNoQyxPQUFPLGFBQWEsQ0FBQztLQUN0QjtJQUVELE1BQU0sYUFBYSxHQUFHLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyxPQUFPLEdBQUcsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNuRixJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLEVBQUU7UUFDaEMsT0FBTyxhQUFhLENBQUM7S0FDdEI7SUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLHlCQUF5QixDQUFDLENBQUM7QUFDN0MsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBUyxnQkFBZ0I7SUFDdkIsTUFBTSxVQUFVLEdBQUcsc0JBQWUsRUFBRSxDQUFDO0lBQ3JDLE1BQU0sYUFBYSxHQUFHLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBRTNGLElBQUksYUFBYSxLQUFLLENBQUMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsR0FBRyxDQUFDLENBQUMsRUFBRTtRQUMxRCxNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixDQUFDLENBQUM7S0FDL0M7SUFFRCxPQUFPLFVBQVUsQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO0FBQzVDLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsY0FBYyxDQUFDLE9BQXVCO0lBQzdDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRWhELElBQUksQ0FBQyxLQUFLLEVBQUU7UUFDVixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUVELE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ2xCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBCdWlsZGVyIH0gZnJvbSAnLi9idWlsZGVyJztcbmltcG9ydCB7IG5vZGVNYWpvclZlcnNpb24sIHBhcnNlU3RhY2tUcmFjZSB9IGZyb20gJy4vdXRpbCc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYSBOb2RlanNGdW5jdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5vZGVqc0Z1bmN0aW9uUHJvcHMgZXh0ZW5kcyBsYW1iZGEuRnVuY3Rpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIFBhdGggdG8gdGhlIGVudHJ5IGZpbGUgKEphdmFTY3JpcHQgb3IgVHlwZVNjcmlwdCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRGVyaXZlZCBmcm9tIHRoZSBuYW1lIG9mIHRoZSBkZWZpbmluZyBmaWxlIGFuZCB0aGUgY29uc3RydWN0J3MgaWQuXG4gICAqIElmIHRoZSBgTm9kZWpzRnVuY3Rpb25gIGlzIGRlZmluZWQgaW4gYHN0YWNrLnRzYCB3aXRoIGBteS1oYW5kbGVyYCBhcyBpZFxuICAgKiAoYG5ldyBOb2RlanNGdW5jdGlvbih0aGlzLCAnbXktaGFuZGxlcicpYCksIHRoZSBjb25zdHJ1Y3Qgd2lsbCBsb29rIGF0IGBzdGFjay5teS1oYW5kbGVyLnRzYFxuICAgKiBhbmQgYHN0YWNrLm15LWhhbmRsZXIuanNgLlxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBleHBvcnRlZCBoYW5kbGVyIGluIHRoZSBlbnRyeSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBoYW5kbGVyXG4gICAqL1xuICByZWFkb25seSBoYW5kbGVyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZSBlbnZpcm9ubWVudC4gT25seSBydW50aW1lcyBvZiB0aGUgTm9kZS5qcyBmYW1pbHkgYXJlXG4gICAqIHN1cHBvcnRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgTk9ERUpTXzEyX1hgIGlmIGBwcm9jZXNzLnZlcnNpb25zLm5vZGVgID49ICcxMi4wLjAnLFxuICAgKiBgTk9ERUpTXzEwX1hgIG90aGVyd2lzZS5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU/OiBsYW1iZGEuUnVudGltZTtcblxuICAvKipcbiAgICogV2hldGhlciB0byBtaW5pZnkgZmlsZXMgd2hlbiBidW5kbGluZy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG1pbmlmeT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gaW5jbHVkZSBzb3VyY2UgbWFwcyB3aGVuIGJ1bmRsaW5nLlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgc291cmNlTWFwcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBidWlsZCBkaXJlY3RvcnlcbiAgICpcbiAgICogQGRlZmF1bHQgLSBgLmJ1aWxkYCBpbiB0aGUgZW50cnkgZmlsZSBkaXJlY3RvcnlcbiAgICovXG4gIHJlYWRvbmx5IGJ1aWxkRGlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY2FjaGUgZGlyZWN0b3J5XG4gICAqXG4gICAqIFBhcmNlbCB1c2VzIGEgZmlsZXN5c3RlbSBjYWNoZSBmb3IgZmFzdCByZWJ1aWxkcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBgLmNhY2hlYCBpbiB0aGUgcm9vdCBkaXJlY3RvcnlcbiAgICovXG4gIHJlYWRvbmx5IGNhY2hlRGlyPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEEgTm9kZS5qcyBMYW1iZGEgZnVuY3Rpb24gYnVuZGxlZCB1c2luZyBQYXJjZWxcbiAqL1xuZXhwb3J0IGNsYXNzIE5vZGVqc0Z1bmN0aW9uIGV4dGVuZHMgbGFtYmRhLkZ1bmN0aW9uIHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBOb2RlanNGdW5jdGlvblByb3BzID0ge30pIHtcbiAgICBpZiAocHJvcHMucnVudGltZSAmJiBwcm9wcy5ydW50aW1lLmZhbWlseSAhPT0gbGFtYmRhLlJ1bnRpbWVGYW1pbHkuTk9ERUpTKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgYE5PREVKU2AgcnVudGltZXMgYXJlIHN1cHBvcnRlZC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBlbnRyeSA9IGZpbmRFbnRyeShpZCwgcHJvcHMuZW50cnkpO1xuICAgIGNvbnN0IGhhbmRsZXIgPSBwcm9wcy5oYW5kbGVyIHx8ICdoYW5kbGVyJztcbiAgICBjb25zdCBidWlsZERpciA9IHByb3BzLmJ1aWxkRGlyIHx8IHBhdGguam9pbihwYXRoLmRpcm5hbWUoZW50cnkpLCAnLmJ1aWxkJyk7XG4gICAgY29uc3QgaGFuZGxlckRpciA9IHBhdGguam9pbihidWlsZERpciwgY3J5cHRvLmNyZWF0ZUhhc2goJ3NoYTI1NicpLnVwZGF0ZShlbnRyeSkuZGlnZXN0KCdoZXgnKSk7XG4gICAgY29uc3QgZGVmYXVsdFJ1blRpbWUgPSBub2RlTWFqb3JWZXJzaW9uKCkgPj0gMTJcbiAgICAgID8gbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1hcbiAgICAgIDogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1g7XG4gICAgY29uc3QgcnVudGltZSA9IHByb3BzLnJ1bnRpbWUgfHwgZGVmYXVsdFJ1blRpbWU7XG5cbiAgICAvLyBCdWlsZCB3aXRoIFBhcmNlbFxuICAgIGNvbnN0IGJ1aWxkZXIgPSBuZXcgQnVpbGRlcih7XG4gICAgICBlbnRyeSxcbiAgICAgIG91dERpcjogaGFuZGxlckRpcixcbiAgICAgIGdsb2JhbDogaGFuZGxlcixcbiAgICAgIG1pbmlmeTogcHJvcHMubWluaWZ5LFxuICAgICAgc291cmNlTWFwczogcHJvcHMuc291cmNlTWFwcyxcbiAgICAgIGNhY2hlRGlyOiBwcm9wcy5jYWNoZURpcixcbiAgICAgIG5vZGVWZXJzaW9uOiBleHRyYWN0VmVyc2lvbihydW50aW1lKSxcbiAgICB9KTtcbiAgICBidWlsZGVyLmJ1aWxkKCk7XG5cbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgcnVudGltZSxcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChoYW5kbGVyRGlyKSxcbiAgICAgIGhhbmRsZXI6IGBpbmRleC4ke2hhbmRsZXJ9YCxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIFNlYXJjaGVzIGZvciBhbiBlbnRyeSBmaWxlLiBQcmVmZXJlbmNlIG9yZGVyIGlzIHRoZSBmb2xsb3dpbmc6XG4gKiAxLiBHaXZlbiBlbnRyeSBmaWxlXG4gKiAyLiBBIC50cyBmaWxlIG5hbWVkIGFzIHRoZSBkZWZpbmluZyBmaWxlIHdpdGggaWQgYXMgc3VmZml4IChkZWZpbmluZy1maWxlLmlkLnRzKVxuICogMy4gQSAuanMgZmlsZSBuYW1lIGFzIHRoZSBkZWZpbmluZyBmaWxlIHdpdGggaWQgYXMgc3VmZml4IChkZWZpbmluZy1maWxlLmlkLmpzKVxuICovXG5mdW5jdGlvbiBmaW5kRW50cnkoaWQ6IHN0cmluZywgZW50cnk/OiBzdHJpbmcpOiBzdHJpbmcge1xuICBpZiAoZW50cnkpIHtcbiAgICBpZiAoIS9cXC4oanN8dHMpJC8udGVzdChlbnRyeSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBKYXZhU2NyaXB0IG9yIFR5cGVTY3JpcHQgZW50cnkgZmlsZXMgYXJlIHN1cHBvcnRlZC4nKTtcbiAgICB9XG4gICAgaWYgKCFmcy5leGlzdHNTeW5jKGVudHJ5KSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgZmluZCBlbnRyeSBmaWxlIGF0ICR7ZW50cnl9YCk7XG4gICAgfVxuICAgIHJldHVybiBlbnRyeTtcbiAgfVxuXG4gIGNvbnN0IGRlZmluaW5nRmlsZSA9IGZpbmREZWZpbmluZ0ZpbGUoKTtcbiAgY29uc3QgZXh0bmFtZSA9IHBhdGguZXh0bmFtZShkZWZpbmluZ0ZpbGUpO1xuXG4gIGNvbnN0IHRzSGFuZGxlckZpbGUgPSBkZWZpbmluZ0ZpbGUucmVwbGFjZShuZXcgUmVnRXhwKGAke2V4dG5hbWV9JGApLCBgLiR7aWR9LnRzYCk7XG4gIGlmIChmcy5leGlzdHNTeW5jKHRzSGFuZGxlckZpbGUpKSB7XG4gICAgcmV0dXJuIHRzSGFuZGxlckZpbGU7XG4gIH1cblxuICBjb25zdCBqc0hhbmRsZXJGaWxlID0gZGVmaW5pbmdGaWxlLnJlcGxhY2UobmV3IFJlZ0V4cChgJHtleHRuYW1lfSRgKSwgYC4ke2lkfS5qc2ApO1xuICBpZiAoZnMuZXhpc3RzU3luYyhqc0hhbmRsZXJGaWxlKSkge1xuICAgIHJldHVybiBqc0hhbmRsZXJGaWxlO1xuICB9XG5cbiAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBlbnRyeSBmaWxlLicpO1xufVxuXG4vKipcbiAqIEZpbmRzIHRoZSBuYW1lIG9mIHRoZSBmaWxlIHdoZXJlIHRoZSBgTm9kZWpzRnVuY3Rpb25gIGlzIGRlZmluZWRcbiAqL1xuZnVuY3Rpb24gZmluZERlZmluaW5nRmlsZSgpOiBzdHJpbmcge1xuICBjb25zdCBzdGFja1RyYWNlID0gcGFyc2VTdGFja1RyYWNlKCk7XG4gIGNvbnN0IGZ1bmN0aW9uSW5kZXggPSBzdGFja1RyYWNlLmZpbmRJbmRleChzID0+IC9Ob2RlanNGdW5jdGlvbi8udGVzdChzLm1ldGhvZE5hbWUgfHwgJycpKTtcblxuICBpZiAoZnVuY3Rpb25JbmRleCA9PT0gLTEgfHwgIXN0YWNrVHJhY2VbZnVuY3Rpb25JbmRleCArIDFdKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgZmluZCBkZWZpbmluZyBmaWxlLicpO1xuICB9XG5cbiAgcmV0dXJuIHN0YWNrVHJhY2VbZnVuY3Rpb25JbmRleCArIDFdLmZpbGU7XG59XG5cbi8qKlxuICogRXh0cmFjdHMgdGhlIHZlcnNpb24gZnJvbSB0aGUgcnVudGltZVxuICovXG5mdW5jdGlvbiBleHRyYWN0VmVyc2lvbihydW50aW1lOiBsYW1iZGEuUnVudGltZSk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gIGNvbnN0IG1hdGNoID0gcnVudGltZS5uYW1lLm1hdGNoKC9ub2RlanMoXFxkKykvKTtcblxuICBpZiAoIW1hdGNoKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIHJldHVybiBtYXRjaFsxXTtcbn1cbiJdfQ==