"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const path = require("path");
const builder_1 = require("../lib/builder");
jest.mock('child_process', () => ({
    spawnSync: jest.fn((_cmd, args) => {
        if (args.includes('/project/folder/error')) {
            return { error: 'parcel-error' };
        }
        if (args.includes('/project/folder/status')) {
            return { status: 1, stdout: Buffer.from('status-error') };
        }
        if (args.includes('/project/folder/no-docker')) {
            return { error: 'Error: spawnSync docker ENOENT' };
        }
        return { error: null, status: 0 };
    }),
}));
beforeEach(() => {
    jest.clearAllMocks();
});
test('calls docker with the correct args', () => {
    const builder = new builder_1.Builder({
        entry: '/project/folder/entry.ts',
        global: 'handler',
        outDir: '/out-dir',
        cacheDir: '/cache-dir',
        nodeDockerTag: 'lts-alpine',
        nodeVersion: '12',
        projectRoot: '/project',
    });
    builder.build();
    // docker build
    expect(child_process_1.spawnSync).toHaveBeenNthCalledWith(1, 'docker', [
        'build', '--build-arg', 'NODE_TAG=lts-alpine', '-t', 'parcel-bundler', path.join(__dirname, '../parcel-bundler'),
    ]);
    // docker run
    expect(child_process_1.spawnSync).toHaveBeenNthCalledWith(2, 'docker', [
        'run', '--rm',
        '-v', '/project:/project',
        '-v', '/out-dir:/out',
        '-v', '/cache-dir:/cache',
        '-w', '/project/folder',
        'parcel-bundler',
        'parcel', 'build', '/project/folder/entry.ts',
        '--out-dir', '/out',
        '--out-file', 'index.js',
        '--global', 'handler',
        '--target', 'node',
        '--bundle-node-modules',
        '--log-level', '2',
        '--no-minify',
        '--no-source-maps',
        '--cache-dir', '/cache',
    ]);
});
test('with Windows paths', () => {
    const builder = new builder_1.Builder({
        entry: 'C:\\my-project\\lib\\entry.ts',
        global: 'handler',
        outDir: '/out-dir',
        cacheDir: '/cache-dir',
        nodeDockerTag: 'lts-alpine',
        nodeVersion: '12',
        projectRoot: 'C:\\my-project',
    });
    builder.build();
    // docker run
    expect(child_process_1.spawnSync).toHaveBeenCalledWith('docker', expect.arrayContaining([
        'parcel', 'build', expect.stringContaining('/lib/entry.ts'),
    ]));
});
test('throws in case of error', () => {
    const builder = new builder_1.Builder({
        entry: '/project/folder/error',
        global: 'handler',
        outDir: 'out-dir',
        nodeDockerTag: 'lts-alpine',
        nodeVersion: '12',
        projectRoot: '/project',
    });
    expect(() => builder.build()).toThrow('parcel-error');
});
test('throws if status is not 0', () => {
    const builder = new builder_1.Builder({
        entry: '/project/folder/status',
        global: 'handler',
        outDir: 'out-dir',
        nodeDockerTag: 'lts-alpine',
        nodeVersion: '12',
        projectRoot: '/project',
    });
    expect(() => builder.build()).toThrow('status-error');
});
test('throws if docker is not installed', () => {
    const builder = new builder_1.Builder({
        entry: '/project/folder/no-docker',
        global: 'handler',
        outDir: 'out-dir',
        nodeDockerTag: 'lts-alpine',
        nodeVersion: '12',
        projectRoot: '/project',
    });
    expect(() => builder.build()).toThrow('Error: spawnSync docker ENOENT');
});
//# sourceMappingURL=data:application/json;base64,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