"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const util_1 = require("./util");
/**
 * Builder
 */
class Builder {
    constructor(options) {
        this.options = options;
        // Original package.json
        const pkgPath = util_1.findPkgPath();
        if (!pkgPath) {
            throw new Error('Cannot find a `package.json` in this project.');
        }
        this.pkgPath = path.join(pkgPath, 'package.json');
        this.originalPkg = fs.readFileSync(this.pkgPath);
        this.originalPkgJson = JSON.parse(this.originalPkg.toString());
    }
    /**
     * Build with parcel in a Docker container
     */
    build() {
        var _a, _b, _c, _d;
        try {
            this.updatePkg();
            const dockerBuildArgs = [
                'build',
                '--build-arg', `NODE_TAG=${this.options.nodeDockerTag}`,
                '-t', 'parcel-bundler',
                path.join(__dirname, '../parcel-bundler'),
            ];
            const build = child_process_1.spawnSync('docker', dockerBuildArgs);
            if (build.error) {
                throw build.error;
            }
            if (build.status !== 0) {
                throw new Error(`[Status ${build.status}] stdout: ${(_a = build.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = build.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
            }
            const containerProjectRoot = '/project';
            const containerOutDir = '/out';
            const containerCacheDir = '/cache';
            const containerEntryPath = path.join(containerProjectRoot, path.relative(this.options.projectRoot, path.resolve(this.options.entry)));
            const dockerRunArgs = [
                'run', '--rm',
                '-v', `${this.options.projectRoot}:${containerProjectRoot}`,
                '-v', `${path.resolve(this.options.outDir)}:${containerOutDir}`,
                ...(this.options.cacheDir ? ['-v', `${path.resolve(this.options.cacheDir)}:${containerCacheDir}`] : []),
                '-w', path.dirname(containerEntryPath).replace(/\\/g, '/'),
                'parcel-bundler',
            ];
            const parcelArgs = [
                'parcel', 'build', containerEntryPath.replace(/\\/g, '/'),
                '--out-dir', containerOutDir,
                '--out-file', 'index.js',
                '--global', this.options.global,
                '--target', 'node',
                '--bundle-node-modules',
                '--log-level', '2',
                !this.options.minify && '--no-minify',
                !this.options.sourceMaps && '--no-source-maps',
                ...(this.options.cacheDir ? ['--cache-dir', containerCacheDir] : []),
            ].filter(Boolean);
            const parcel = child_process_1.spawnSync('docker', [...dockerRunArgs, ...parcelArgs]);
            if (parcel.error) {
                throw parcel.error;
            }
            if (parcel.status !== 0) {
                throw new Error(`[Status ${parcel.status}] stdout: ${(_c = parcel.stdout) === null || _c === void 0 ? void 0 : _c.toString().trim()}\n\n\nstderr: ${(_d = parcel.stderr) === null || _d === void 0 ? void 0 : _d.toString().trim()}`);
            }
        }
        catch (err) {
            throw new Error(`Failed to build file at ${this.options.entry}: ${err}`);
        }
        finally { // Always restore package.json to original
            this.restorePkg();
        }
    }
    /**
     * Updates the package.json to configure Parcel
     */
    updatePkg() {
        const updateData = {};
        // Update engines.node (Babel target)
        updateData.engines = { node: `>= ${this.options.nodeVersion}` };
        // Write new package.json
        if (Object.keys(updateData).length !== 0) {
            fs.writeFileSync(this.pkgPath, JSON.stringify({
                ...this.originalPkgJson,
                ...updateData,
            }, null, 2));
        }
    }
    restorePkg() {
        fs.writeFileSync(this.pkgPath, this.originalPkg);
    }
}
exports.Builder = Builder;
//# sourceMappingURL=data:application/json;base64,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