"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const builder_1 = require("./builder");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'handler';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = props.runtime || defaultRunTime;
        const projectRoot = (_a = props.projectRoot) !== null && _a !== void 0 ? _a : util_1.findGitPath();
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Build with Parcel
        const builder = new builder_1.Builder({
            entry,
            outDir: handlerDir,
            global: handler,
            minify: props.minify,
            sourceMaps: props.sourceMaps,
            cacheDir: props.cacheDir,
            nodeVersion: extractVersion(runtime),
            nodeDockerTag: props.nodeDockerTag || `${process.versions.node}-alpine`,
            projectRoot: path.resolve(projectRoot),
            environment: props.containerEnvironment,
        });
        builder.build();
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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