"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const fs = require("fs");
const path = require("path");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        const projectRoot = (_c = props.projectRoot) !== null && _c !== void 0 ? _c : util_1.findGitPath();
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const nodeDockerTag = (_d = props.nodeDockerTag) !== null && _d !== void 0 ? _d : `${process.versions.node}-alpine`;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.parcel({
                entry,
                global: handler,
                minify: props.minify,
                sourceMaps: props.sourceMaps,
                cacheDir: props.cacheDir,
                nodeVersion: extractVersion(runtime),
                nodeDockerTag,
                projectRoot: path.resolve(projectRoot),
                environment: props.containerEnvironment,
            }),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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