"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const { root } = path.parse(directory);
    if (directory === root && !fs.existsSync(path.join(directory, name))) {
        return undefined;
    }
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    return findUp(name, path.dirname(directory));
}
exports.findUp = findUp;
//# sourceMappingURL=data:application/json;base64,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