"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        // Find project root
        const projectRoot = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`);
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Bundling image derived from runtime bundling image (AWS SAM docker image)
        const image = cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
            buildArgs: {
                ...(_b = options.buildArgs) !== null && _b !== void 0 ? _b : {},
                IMAGE: options.runtime.bundlingDockerImage.image,
                PARCEL_VERSION: (_c = options.parcelVersion) !== null && _c !== void 0 ? _c : '2.0.0-beta.1',
            },
        });
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_d = options.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_e = options.nodeModules) !== null && _e !== void 0 ? _e : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            'cdk-lambda': `${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/index.js`,
            'targets': {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_f = options.sourceMaps) !== null && _f !== void 0 ? _f : false,
                    minify: (_g = options.minify) !== null && _g !== void 0 ? _g : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Entry file path relative to container path
        const containerEntryPath = path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, path.relative(projectRoot, path.resolve(options.entry)));
        const parcelCommand = [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', containerEntryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', '/parcel-cache']
                : [],
        ].join(' ');
        let installer = Installer.NPM;
        let lockfile;
        let depsCommand = '';
        if (dependencies) {
            // Create a dummy package.json for dependencies that we need to install
            fs.writeFileSync(path.join(projectRoot, '.package.json'), JSON.stringify({ dependencies }));
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, LockFile.YARN))) {
                installer = Installer.YARN;
                lockfile = LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, LockFile.NPM))) {
                lockfile = LockFile.NPM;
            }
            // Move dummy package.json and lock file then install
            depsCommand = chain([
                `mv ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/.package.json ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/package.json`,
                lockfile ? `cp ${cdk.AssetStaging.BUNDLING_INPUT_DIR}/${lockfile} ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${lockfile}` : '',
                `cd ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR} && ${installer} install`,
            ]);
        }
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                image,
                command: ['bash', '-c', chain([parcelCommand, depsCommand])],
                environment: options.parcelEnvironment,
                volumes: options.cacheDir
                    ? [{ containerPath: '/parcel-cache', hostPath: options.cacheDir }]
                    : [],
                workingDirectory: path.dirname(containerEntryPath).replace(/\\/g, '/'),
            },
        });
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile || (LockFile = {}));
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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