"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            LocalBundler._runsLocally = /^2/.test(parcel.stdout.toString().trim()); // Cache result to avoid unnecessary spawns
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.ts$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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