"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process = require("child_process");
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const bundlers_1 = require("../lib/bundlers");
const bundling_1 = require("../lib/bundling");
const util = require("../lib/util");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
const originalFindUp = util.findUp;
const findUpMock = jest.spyOn(util, 'findUp').mockImplementation((name, directory) => {
    if (name === 'package.json') {
        return path.join(__dirname, '..');
    }
    return originalFindUp(name, directory);
});
const fromAssetMock = jest.spyOn(core_1.BundlingDockerImage, 'fromAsset');
beforeEach(() => {
    jest.clearAllMocks();
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: 'cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            local: {
                props: expect.objectContaining({
                    projectRoot: '/project',
                }),
            },
            environment: {
                KEY: 'value',
            },
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist --cache-dir /asset-input/cache-dir',
                    'mv /asset-output/entry.js /asset-output/index.js',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
    // Searches for the package.json starting in the directory of the entry file
    expect(findUpMock).toHaveBeenCalledWith('package.json', '/project/folder');
});
test('Parcel bundling with handler named index.ts', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/index.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/index.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
            ],
        }),
    });
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
                    'mv /asset-output/entry.js /asset-output/index.js',
                    `echo \'{\"dependencies\":{\"delay\":\"${package_json_1.version}\"}}\' > /asset-output/package.json`,
                    'cd /asset-output',
                    'npm install',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
test('with Docker build args', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        buildArgs: {
            HELLO: 'WORLD',
        },
        forceDockerBundling: true,
    });
    expect(fromAssetMock).toHaveBeenCalledWith(expect.stringMatching(/parcel$/), expect.objectContaining({
        buildArgs: expect.objectContaining({
            HELLO: 'WORLD',
        }),
    }));
});
test('Local bundling', () => {
    const spawnSyncMock = jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('stdout'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    const bundler = new bundlers_1.LocalBundler({
        installer: bundlers_1.Installer.NPM,
        projectRoot: '/project',
        relativeEntryPath: 'folder/entry.ts',
        dependencies: {
            dep: 'version',
        },
        environment: {
            KEY: 'value',
        },
        lockFile: bundlers_1.LockFile.NPM,
    });
    bundler.tryBundle('/outdir');
    expect(spawnSyncMock).toHaveBeenCalledWith('bash', [
        '-c',
        [
            '$(node -p \"require.resolve(\'parcel\')\") build /project/folder/entry.ts --target cdk-lambda --dist-dir /outdir --no-autoinstall --no-scope-hoist',
            'mv /outdir/entry.js /outdir/index.js',
            'echo \'{\"dependencies\":{\"dep\":\"version\"}}\' > /outdir/package.json',
            'cp /project/package-lock.json /outdir/package-lock.json',
            'cd /outdir',
            'npm install',
        ].join(' && '),
    ], expect.objectContaining({
        env: expect.objectContaining({ KEY: 'value' }),
    }));
    // Docker image is not built
    expect(fromAssetMock).not.toHaveBeenCalled();
});
//# sourceMappingURL=data:application/json;base64,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