"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            LocalBundler._runsLocally = /^2/.test(parcel.stdout.toString().trim()); // Cache result to avoid unnecessary spawns
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : '2.0.0-beta.1',
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.ts$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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