"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const PARCEL_VERSION = '2.0.0-beta.1';
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            const version = parcel.stdout.toString().trim();
            LocalBundler._runsLocally = new RegExp(`^${PARCEL_VERSION}`).test(version); // Cache result to avoid unnecessary spawns
            if (!LocalBundler._runsLocally) {
                process.stderr.write(`Incorrect parcel version detected: ${version} <> ${PARCEL_VERSION}. Switching to Docker bundling.\n`);
            }
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : PARCEL_VERSION,
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.(jsx|tsx?)$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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