"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.exec = exports.findUp = exports.nodeMajorVersion = exports.parseStackTrace = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
// From https://github.com/errwischt/stacktrace-parser/blob/master/src/stack-trace-parser.js
const STACK_RE = /^\s*at (?:((?:\[object object\])?[^\\/]+(?: \[as \S+\])?) )?\(?(.*?):(\d+)(?::(\d+))?\)?\s*$/i;
/**
 * Parses the stack trace of an error
 */
function parseStackTrace(error) {
    const err = error || new Error();
    if (!err.stack) {
        return [];
    }
    const lines = err.stack.split('\n');
    const stackTrace = [];
    for (const line of lines) {
        const results = STACK_RE.exec(line);
        if (results) {
            stackTrace.push({
                file: results[2],
                methodName: results[1],
                lineNumber: parseInt(results[3], 10),
                column: parseInt(results[4], 10),
            });
        }
    }
    return stackTrace;
}
exports.parseStackTrace = parseStackTrace;
/**
 * Returns the major version of node installation
 */
function nodeMajorVersion() {
    return parseInt(process.versions.node.split('.')[0], 10);
}
exports.nodeMajorVersion = nodeMajorVersion;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    if (fs.existsSync(path.join(directory, name))) {
        return directory;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return undefined;
    }
    return findUp(name, path.dirname(absoluteDirectory));
}
exports.findUp = findUp;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
//# sourceMappingURL=data:application/json;base64,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