"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process = require("child_process");
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const bundlers_1 = require("../lib/bundlers");
const bundling_1 = require("../lib/bundling");
const util = require("../lib/util");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
const originalFindUp = util.findUp;
const fromAssetMock = jest.spyOn(core_1.BundlingDockerImage, 'fromAsset');
let findUpMock;
beforeEach(() => {
    jest.clearAllMocks();
    bundlers_1.LocalBundler.clearRunsLocallyCache();
    findUpMock = jest.spyOn(util, 'findUp').mockImplementation((name, directory) => {
        if (name === 'package.json') {
            return path.join(__dirname, '..');
        }
        return originalFindUp(name, directory);
    });
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: 'cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            local: {
                props: expect.objectContaining({
                    projectRoot: '/project',
                }),
            },
            environment: {
                KEY: 'value',
            },
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\', { paths: [\'/\'] })") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist --cache-dir /asset-input/cache-dir',
                    'mv /asset-output/entry.js /asset-output/index.js',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
    // Searches for the package.json starting in the directory of the entry file
    expect(findUpMock).toHaveBeenCalledWith('package.json', '/project/folder');
});
test('Parcel bundling with handler named index.ts', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/index.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                '$(node -p "require.resolve(\'parcel\', { paths: [\'/\'] })") build /asset-input/folder/index.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
            ],
        }),
    });
});
test('Parcel bundling with tsx handler', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/handler.tsx',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\', { paths: [\'/\'] })") build /asset-input/folder/handler.tsx --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
                    'mv /asset-output/handler.js /asset-output/index.js',
                ].join(' && '),
            ],
        }),
    });
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\', { paths: [\'/\'] })") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
                    'mv /asset-output/entry.js /asset-output/index.js',
                    `echo \'{\"dependencies\":{\"delay\":\"${package_json_1.version}\"}}\' > /asset-output/package.json`,
                    'cd /asset-output',
                    'npm install',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
test('with Docker build args', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        buildArgs: {
            HELLO: 'WORLD',
        },
        forceDockerBundling: true,
    });
    expect(fromAssetMock).toHaveBeenCalledWith(expect.stringMatching(/parcel$/), expect.objectContaining({
        buildArgs: expect.objectContaining({
            HELLO: 'WORLD',
        }),
    }));
});
test('Local bundling', () => {
    const spawnSyncMock = jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('2.0.0-beta.1'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    const bundler = new bundlers_1.LocalBundler({
        installer: bundlers_1.Installer.NPM,
        projectRoot: __dirname,
        relativeEntryPath: 'folder/entry.ts',
        dependencies: {
            dep: 'version',
        },
        environment: {
            KEY: 'value',
        },
        lockFile: bundlers_1.LockFile.NPM,
    });
    bundler.tryBundle('/outdir');
    expect(spawnSyncMock).toHaveBeenCalledWith('bash', expect.arrayContaining(['-c', expect.stringContaining(__dirname)]), expect.objectContaining({
        env: expect.objectContaining({ KEY: 'value' }),
        cwd: expect.stringContaining(path.join(__dirname, 'folder')),
    }));
    // Docker image is not built
    expect(fromAssetMock).not.toHaveBeenCalled();
});
test('LocalBundler.runsLocally checks parcel version and caches results', () => {
    const spawnSyncMock = jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('2.0.0-beta.1'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    expect(bundlers_1.LocalBundler.runsLocally(__dirname)).toBe(true);
    expect(bundlers_1.LocalBundler.runsLocally(__dirname)).toBe(true);
    expect(spawnSyncMock).toHaveBeenCalledTimes(1);
    expect(spawnSyncMock).toHaveBeenCalledWith(expect.stringContaining('parcel'), ['--version']);
});
test('LocalBundler.runsLocally with incorrect parcel version', () => {
    jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('3.5.1'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    expect(bundlers_1.LocalBundler.runsLocally(__dirname)).toBe(false);
});
test('Project root detection', () => {
    findUpMock.mockImplementation(() => undefined);
    expect(() => bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    })).toThrow(/Cannot find project root/);
    expect(findUpMock).toHaveBeenNthCalledWith(1, `.git${path.sep}`);
    expect(findUpMock).toHaveBeenNthCalledWith(2, bundlers_1.LockFile.YARN);
    expect(findUpMock).toHaveBeenNthCalledWith(3, bundlers_1.LockFile.NPM);
    expect(findUpMock).toHaveBeenNthCalledWith(4, 'package.json');
});
test('Custom bundling docker image', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        projectRoot: '/project',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        bundlingDockerImage: core_1.BundlingDockerImage.fromRegistry('my-custom-image'),
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            image: { image: 'my-custom-image' },
        }),
    });
});
//# sourceMappingURL=data:application/json;base64,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