"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const ESBUILD_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        this.props = props;
        Bundling.runsLocally = (_c = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : (_b = util_1.getEsBuildVersion()) === null || _b === void 0 ? void 0 : _b.startsWith(ESBUILD_VERSION)) !== null && _c !== void 0 ? _c : false;
        const projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(projectRoot, path.resolve(props.entry));
        this.externals = [
            ...(_d = this.props.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'],
            ...(_e = this.props.nodeModules) !== null && _e !== void 0 ? _e : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_f = props.bundlingDockerImage) !== null && _f !== void 0 ? _f : cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_g = props.buildArgs) !== null && _g !== void 0 ? _g : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    ESBUILD_VERSION: (_h = props.esbuildVersion) !== null && _h !== void 0 ? _h : ESBUILD_VERSION,
                },
            }) : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.bundlingEnvironment;
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    var _a;
                    if (Bundling.runsLocally === false) {
                        process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...(_a = props.bundlingEnvironment) !== null && _a !== void 0 ? _a : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.entry),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b;
        const pathJoin = osPathJoin(osPlatform);
        const npx = osPlatform === 'win32' ? 'npx.cmd' : 'npx';
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const esbuildCommand = [
            npx, 'esbuild',
            '--bundle', pathJoin(inputDir, this.relativeEntryPath),
            `--target=${(_b = this.props.target) !== null && _b !== void 0 ? _b : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile=${pathJoin(outputDir, 'index.js')}`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
        ].join(' ');
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            let installer = Installer.NPM;
            let lockFile = util_1.LockFile.NPM;
            if (this.props.depsLockFilePath.endsWith(util_1.LockFile.YARN)) {
                lockFile = util_1.LockFile.YARN;
                installer = Installer.YARN;
            }
            const osCommand = new OsCommand(osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(inputDir, lockFile), pathJoin(outputDir, lockFile)),
                osCommand.changeDirectory(outputDir),
                `${installer} install`,
            ]);
        }
        return chain([esbuildCommand, depsCommand]);
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,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