"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * (experimental) A Node.js Lambda function bundled using esbuild.
 *
 * @experimental
 */
class NodejsFunction extends lambda.Function {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Find lock file
        let depsLockFilePath;
        if (props.depsLockFilePath) {
            if (!fs.existsSync(props.depsLockFilePath)) {
                throw new Error(`Lock file at ${props.depsLockFilePath} doesn't exist`);
            }
            if (!fs.statSync(props.depsLockFilePath).isFile()) {
                throw new Error('`depsLockFilePath` should point to a file');
            }
            depsLockFilePath = path.resolve(props.depsLockFilePath);
        }
        else {
            const lockFile = (_a = util_1.findUp(util_1.LockFile.YARN)) !== null && _a !== void 0 ? _a : util_1.findUp(util_1.LockFile.NPM);
            if (!lockFile) {
                throw new Error('Cannot find a package lock file (`yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
            }
            depsLockFilePath = lockFile;
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_b = props.handler) !== null && _b !== void 0 ? _b : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_c = props.runtime) !== null && _c !== void 0 ? _c : defaultRunTime;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(_d = props.bundling) !== null && _d !== void 0 ? _d : {},
                entry,
                runtime,
                depsLockFilePath,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if ((_e = props.awsSdkConnectionReuse) !== null && _e !== void 0 ? _e : true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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