"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
const ESBUILD_VERSION = '0';
/**
 * Bundling with esbuild
 */
class Bundling {
    constructor(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        this.props = props;
        Bundling.runsLocally = (_c = (_a = Bundling.runsLocally) !== null && _a !== void 0 ? _a : (_b = util_1.getEsBuildVersion()) === null || _b === void 0 ? void 0 : _b.startsWith(ESBUILD_VERSION)) !== null && _c !== void 0 ? _c : false;
        const projectRoot = path.dirname(props.depsLockFilePath);
        this.relativeEntryPath = path.relative(projectRoot, path.resolve(props.entry));
        if (props.tsconfig) {
            this.relativeTsconfigPath = path.relative(projectRoot, path.resolve(props.tsconfig));
        }
        this.externals = [
            ...(_d = props.externalModules) !== null && _d !== void 0 ? _d : ['aws-sdk'],
            ...(_e = props.nodeModules) !== null && _e !== void 0 ? _e : [],
        ];
        // Docker bundling
        const shouldBuildImage = props.forceDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? (_f = props.dockerImage) !== null && _f !== void 0 ? _f : cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...(_g = props.buildArgs) !== null && _g !== void 0 ? _g : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    ESBUILD_VERSION: (_h = props.esbuildVersion) !== null && _h !== void 0 ? _h : ESBUILD_VERSION,
                },
            }) : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = props.environment;
        // Bundling sets the working directory to cdk.AssetStaging.BUNDLING_INPUT_DIR
        // and we want to force npx to use the globally installed esbuild.
        this.workingDirectory = '/';
        // Local bundling
        if (!props.forceDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    var _a;
                    if (Bundling.runsLocally === false) {
                        process.stderr.write('esbuild cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    util_1.exec(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...(_a = props.environment) !== null && _a !== void 0 ? _a : {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit',
                        ],
                        cwd: path.dirname(props.entry),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    /**
     * esbuild bundled Lambda asset code
     */
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(path.dirname(options.depsLockFilePath), {
            assetHashType: cdk.AssetHashType.OUTPUT,
            bundling: new Bundling(options),
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        const pathJoin = osPathJoin(osPlatform);
        const npx = osPlatform === 'win32' ? 'npx.cmd' : 'npx';
        const loaders = Object.entries((_a = this.props.loader) !== null && _a !== void 0 ? _a : {});
        const defines = Object.entries((_b = this.props.define) !== null && _b !== void 0 ? _b : {});
        const esbuildCommand = [
            npx, 'esbuild',
            '--bundle', pathJoin(inputDir, this.relativeEntryPath),
            `--target=${(_c = this.props.target) !== null && _c !== void 0 ? _c : toTarget(this.props.runtime)}`,
            '--platform=node',
            `--outfile=${pathJoin(outputDir, 'index.js')}`,
            ...this.props.minify ? ['--minify'] : [],
            ...this.props.sourceMap ? ['--sourcemap'] : [],
            ...this.externals.map(external => `--external:${external}`),
            ...loaders.map(([ext, name]) => `--loader:${ext}=${name}`),
            ...defines.map(([key, value]) => `--define:${key}=${value}`),
            ...this.props.logLevel ? [`--log-level=${this.props.logLevel}`] : [],
            ...this.props.keepNames ? ['--keep-names'] : [],
            ...this.relativeTsconfigPath ? [`--tsconfig=${pathJoin(inputDir, this.relativeTsconfigPath)}`] : [],
            ...this.props.metafile ? [`--metafile=${pathJoin(outputDir, 'index.meta.json')}`] : [],
            ...this.props.banner ? [`--banner='${this.props.banner}'`] : [],
            ...this.props.footer ? [`--footer='${this.props.footer}'`] : [],
        ].join(' ');
        let depsCommand = '';
        if (this.props.nodeModules) {
            // Find 'package.json' closest to entry folder, we are going to extract the
            // modules versions from it.
            const pkgPath = util_1.findUp('package.json', path.dirname(this.props.entry));
            if (!pkgPath) {
                throw new Error('Cannot find a `package.json` in this project. Using `nodeModules` requires a `package.json`.');
            }
            // Determine dependencies versions, lock file and installer
            const dependencies = util_1.extractDependencies(pkgPath, this.props.nodeModules);
            let installer = Installer.NPM;
            let lockFile = util_1.LockFile.NPM;
            if (this.props.depsLockFilePath.endsWith(util_1.LockFile.YARN)) {
                lockFile = util_1.LockFile.YARN;
                installer = Installer.YARN;
            }
            const osCommand = new OsCommand(osPlatform);
            // Create dummy package.json, copy lock file if any and then install
            depsCommand = chain([
                osCommand.writeJson(pathJoin(outputDir, 'package.json'), { dependencies }),
                osCommand.copy(pathJoin(inputDir, lockFile), pathJoin(outputDir, lockFile)),
                osCommand.changeDirectory(outputDir),
                `${installer} install`,
            ]);
        }
        return chain([
            ...(_e = (_d = this.props.commandHooks) === null || _d === void 0 ? void 0 : _d.beforeBundling(inputDir, outputDir)) !== null && _e !== void 0 ? _e : [],
            esbuildCommand,
            ...(_g = (this.props.nodeModules && ((_f = this.props.commandHooks) === null || _f === void 0 ? void 0 : _f.beforeInstall(inputDir, outputDir)))) !== null && _g !== void 0 ? _g : [],
            depsCommand,
            ...(_j = (_h = this.props.commandHooks) === null || _h === void 0 ? void 0 : _h.afterBundling(inputDir, outputDir)) !== null && _j !== void 0 ? _j : [],
        ]);
    }
}
exports.Bundling = Bundling;
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer || (Installer = {}));
/**
 * OS agnostic command
 */
class OsCommand {
    constructor(osPlatform) {
        this.osPlatform = osPlatform;
    }
    writeJson(filePath, data) {
        const stringifiedData = JSON.stringify(data);
        if (this.osPlatform === 'win32') {
            return `echo ^${stringifiedData}^ > ${filePath}`;
        }
        return `echo '${stringifiedData}' > ${filePath}`;
    }
    copy(src, dest) {
        if (this.osPlatform === 'win32') {
            return `copy ${src} ${dest}`;
        }
        return `cp ${src} ${dest}`;
    }
    changeDirectory(dir) {
        return `cd ${dir}`;
    }
}
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
/**
 * Converts a runtime to an esbuild node target
 */
function toTarget(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return `node${match[1]}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLG9EQUErRDtBQUMvRCxxQ0FBcUM7QUFFckMsaUNBQXdGO0FBRXhGLE1BQU0sZUFBZSxHQUFHLEdBQUcsQ0FBQztBQXNCNUI7O0dBRUc7QUFDSCxNQUFhLFFBQVE7SUE0Qm5CLFlBQTZCLEtBQW9COztRQUFwQixVQUFLLEdBQUwsS0FBSyxDQUFlO1FBQy9DLFFBQVEsQ0FBQyxXQUFXLGVBQUcsUUFBUSxDQUFDLFdBQVcseUNBQ3RDLHdCQUFpQixFQUFFLDBDQUFFLFVBQVUsQ0FBQyxlQUFlLG9DQUMvQyxLQUFLLENBQUM7UUFFWCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRS9FLElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNsQixJQUFJLENBQUMsb0JBQW9CLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztTQUN0RjtRQUVELElBQUksQ0FBQyxTQUFTLEdBQUc7WUFDZixTQUFHLEtBQUssQ0FBQyxlQUFlLG1DQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRTtTQUMzQixDQUFDO1FBRUYsa0JBQWtCO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztRQUM1RSxJQUFJLENBQUMsS0FBSyxHQUFHLGdCQUFnQjtZQUMzQixDQUFDLE9BQUMsS0FBSyxDQUFDLFdBQVcsbUNBQUksR0FBRyxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsRUFBRTtnQkFDdkYsU0FBUyxFQUFFO29CQUNULFNBQUcsS0FBSyxDQUFDLFNBQVMsbUNBQUksRUFBRTtvQkFDeEIsS0FBSyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsS0FBSztvQkFDOUMsZUFBZSxRQUFFLEtBQUssQ0FBQyxjQUFjLG1DQUFJLGVBQWU7aUJBQ3pEO2FBQ0YsQ0FBQyxDQUNGLENBQUMsQ0FBQyxHQUFHLENBQUMsbUJBQW1CLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsbUNBQW1DO1FBRXRGLE1BQU0sZUFBZSxHQUFHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM5SCxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDckMsNkVBQTZFO1FBQzdFLGtFQUFrRTtRQUNsRSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsR0FBRyxDQUFDO1FBRTVCLGlCQUFpQjtRQUNqQixJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFLEVBQUUsK0JBQStCO1lBQy9ELE1BQU0sVUFBVSxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztZQUNqQyxNQUFNLGtCQUFrQixHQUFHLENBQUMsU0FBaUIsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLFdBQVcsRUFBRSxTQUFTLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFFakgsSUFBSSxDQUFDLEtBQUssR0FBRztnQkFDWCxTQUFTLENBQUMsU0FBaUI7O29CQUN6QixJQUFJLFFBQVEsQ0FBQyxXQUFXLEtBQUssS0FBSyxFQUFFO3dCQUNsQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO3dCQUNwRixPQUFPLEtBQUssQ0FBQztxQkFDZDtvQkFFRCxNQUFNLFlBQVksR0FBRyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztvQkFFbkQsV0FBSSxDQUNGLFVBQVUsS0FBSyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsTUFBTSxFQUN2Qzt3QkFDRSxVQUFVLEtBQUssT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUk7d0JBQ3BDLFlBQVk7cUJBQ2IsRUFDRDt3QkFDRSxHQUFHLEVBQUUsRUFBRSxHQUFHLE9BQU8sQ0FBQyxHQUFHLEVBQUUsU0FBRyxLQUFLLENBQUMsV0FBVyxtQ0FBSSxFQUFFLEVBQUU7d0JBQ25ELEtBQUssRUFBRTs0QkFDTCxRQUFROzRCQUNSLE9BQU8sQ0FBQyxNQUFNOzRCQUNkLFNBQVM7eUJBQ1Y7d0JBQ0QsR0FBRyxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQzt3QkFDOUIsd0JBQXdCLEVBQUUsVUFBVSxLQUFLLE9BQU87cUJBQ2pELENBQUMsQ0FBQztvQkFFTCxPQUFPLElBQUksQ0FBQztnQkFDZCxDQUFDO2FBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQWxHRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsT0FBc0I7UUFDekMsT0FBTyxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO1lBQzVELGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE1BQU07WUFDdkMsUUFBUSxFQUFFLElBQUksUUFBUSxDQUFDLE9BQU8sQ0FBQztTQUNoQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sTUFBTSxDQUFDLHFCQUFxQjtRQUNqQyxJQUFJLENBQUMsV0FBVyxHQUFHLFNBQVMsQ0FBQztJQUMvQixDQUFDO0lBd0ZNLHFCQUFxQixDQUFDLFFBQWdCLEVBQUUsU0FBaUIsRUFBRSxhQUE4QixPQUFPOztRQUNyRyxNQUFNLFFBQVEsR0FBRyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFeEMsTUFBTSxHQUFHLEdBQUcsVUFBVSxLQUFLLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUM7UUFDdkQsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sbUNBQUksRUFBRSxDQUFDLENBQUM7UUFDeEQsTUFBTSxPQUFPLEdBQUcsTUFBTSxDQUFDLE9BQU8sT0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sbUNBQUksRUFBRSxDQUFDLENBQUM7UUFFeEQsTUFBTSxjQUFjLEdBQVc7WUFDN0IsR0FBRyxFQUFFLFNBQVM7WUFDZCxVQUFVLEVBQUUsUUFBUSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUM7WUFDdEQsWUFBWSxNQUFBLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxRQUFRLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUMvRCxpQkFBaUI7WUFDakIsYUFBYSxRQUFRLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxFQUFFO1lBQzlDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDeEMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM5QyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsY0FBYyxRQUFRLEVBQUUsQ0FBQztZQUMzRCxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsWUFBWSxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7WUFDMUQsR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUFDLFlBQVksR0FBRyxJQUFJLEtBQUssRUFBRSxDQUFDO1lBQzVELEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDcEUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUMvQyxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLFFBQVEsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ25HLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxRQUFRLENBQUMsU0FBUyxFQUFFLGlCQUFpQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3RGLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDL0QsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxhQUFhLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtTQUNoRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUVaLElBQUksV0FBVyxHQUFHLEVBQUUsQ0FBQztRQUNyQixJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQzFCLDJFQUEyRTtZQUMzRSw0QkFBNEI7WUFDNUIsTUFBTSxPQUFPLEdBQUcsYUFBTSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUN2RSxJQUFJLENBQUMsT0FBTyxFQUFFO2dCQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsOEZBQThGLENBQUMsQ0FBQzthQUNqSDtZQUVELDJEQUEyRDtZQUMzRCxNQUFNLFlBQVksR0FBRywwQkFBbUIsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUMxRSxJQUFJLFNBQVMsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDO1lBQzlCLElBQUksUUFBUSxHQUFHLGVBQVEsQ0FBQyxHQUFHLENBQUM7WUFDNUIsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUU7Z0JBQ3ZELFFBQVEsR0FBRyxlQUFRLENBQUMsSUFBSSxDQUFDO2dCQUN6QixTQUFTLEdBQUcsU0FBUyxDQUFDLElBQUksQ0FBQzthQUM1QjtZQUVELE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBRTVDLG9FQUFvRTtZQUNwRSxXQUFXLEdBQUcsS0FBSyxDQUFDO2dCQUNsQixTQUFTLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsY0FBYyxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQztnQkFDMUUsU0FBUyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFLFFBQVEsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQUM7Z0JBQzNFLFNBQVMsQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO2dCQUNwQyxHQUFHLFNBQVMsVUFBVTthQUN2QixDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sS0FBSyxDQUFDO1lBQ1gsZUFBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsY0FBYyxDQUFDLFFBQVEsRUFBRSxTQUFTLG9DQUFLLEVBQUU7WUFDckUsY0FBYztZQUNkLFNBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVcsV0FBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksMENBQUUsYUFBYSxDQUFDLFFBQVEsRUFBRSxTQUFTLEVBQUMsQ0FBQyxtQ0FBSSxFQUFFO1lBQ2hHLFdBQVc7WUFDWCxlQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSwwQ0FBRSxhQUFhLENBQUMsUUFBUSxFQUFFLFNBQVMsb0NBQUssRUFBRTtTQUNyRSxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFwS0QsNEJBb0tDO0FBRUQsSUFBSyxTQUdKO0FBSEQsV0FBSyxTQUFTO0lBQ1osd0JBQVcsQ0FBQTtJQUNYLDBCQUFhLENBQUE7QUFDZixDQUFDLEVBSEksU0FBUyxLQUFULFNBQVMsUUFHYjtBQUVEOztHQUVHO0FBQ0gsTUFBTSxTQUFTO0lBQ2IsWUFBNkIsVUFBMkI7UUFBM0IsZUFBVSxHQUFWLFVBQVUsQ0FBaUI7SUFBRyxDQUFDO0lBRXJELFNBQVMsQ0FBQyxRQUFnQixFQUFFLElBQVM7UUFDMUMsTUFBTSxlQUFlLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM3QyxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssT0FBTyxFQUFFO1lBQy9CLE9BQU8sU0FBUyxlQUFlLE9BQU8sUUFBUSxFQUFFLENBQUM7U0FDbEQ7UUFFRCxPQUFPLFNBQVMsZUFBZSxPQUFPLFFBQVEsRUFBRSxDQUFDO0lBQ25ELENBQUM7SUFFTSxJQUFJLENBQUMsR0FBVyxFQUFFLElBQVk7UUFDbkMsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLE9BQU8sRUFBRTtZQUMvQixPQUFPLFFBQVEsR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDO1NBQzlCO1FBRUQsT0FBTyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUUsQ0FBQztJQUM3QixDQUFDO0lBRU0sZUFBZSxDQUFDLEdBQVc7UUFDaEMsT0FBTyxNQUFNLEdBQUcsRUFBRSxDQUFDO0lBQ3JCLENBQUM7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsU0FBUyxLQUFLLENBQUMsUUFBa0I7SUFDL0IsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNoRCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFVBQVUsQ0FBQyxRQUF5QjtJQUMzQyxPQUFPLFVBQVMsR0FBRyxLQUFlO1FBQ2hDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUMsQ0FBQztRQUNuQyxnREFBZ0Q7UUFDaEQsSUFBSSxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssT0FBTyxJQUFJLFFBQVEsS0FBSyxPQUFPLEVBQUU7WUFDckQsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztTQUNuQztRQUNELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsUUFBUSxDQUFDLE9BQWdCO0lBQ2hDLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBRWhELElBQUksQ0FBQyxLQUFLLEVBQUU7UUFDVixNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxDQUFDLENBQUM7S0FDekQ7SUFFRCxPQUFPLE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7QUFDM0IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG9zIGZyb20gJ29zJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgeyBBc3NldENvZGUsIENvZGUsIFJ1bnRpbWUgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEJ1bmRsaW5nT3B0aW9ucyB9IGZyb20gJy4vdHlwZXMnO1xuaW1wb3J0IHsgZXhlYywgZXh0cmFjdERlcGVuZGVuY2llcywgZmluZFVwLCBnZXRFc0J1aWxkVmVyc2lvbiwgTG9ja0ZpbGUgfSBmcm9tICcuL3V0aWwnO1xuXG5jb25zdCBFU0JVSUxEX1ZFUlNJT04gPSAnMCc7XG5cbi8qKlxuICogQnVuZGxpbmcgcHJvcGVydGllc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1bmRsaW5nUHJvcHMgZXh0ZW5kcyBCdW5kbGluZ09wdGlvbnMge1xuICAvKipcbiAgICogUGF0aCB0byBsb2NrIGZpbGVcbiAgICovXG4gIHJlYWRvbmx5IGRlcHNMb2NrRmlsZVBhdGg6IHN0cmluZztcblxuICAvKipcbiAgICogRW50cnkgZmlsZVxuICAgKi9cbiAgcmVhZG9ubHkgZW50cnk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJ1bnRpbWUgb2YgdGhlIGxhbWJkYSBmdW5jdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgcnVudGltZTogUnVudGltZTtcbn1cblxuLyoqXG4gKiBCdW5kbGluZyB3aXRoIGVzYnVpbGRcbiAqL1xuZXhwb3J0IGNsYXNzIEJ1bmRsaW5nIGltcGxlbWVudHMgY2RrLkJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBlc2J1aWxkIGJ1bmRsZWQgTGFtYmRhIGFzc2V0IGNvZGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYnVuZGxlKG9wdGlvbnM6IEJ1bmRsaW5nUHJvcHMpOiBBc3NldENvZGUge1xuICAgIHJldHVybiBDb2RlLmZyb21Bc3NldChwYXRoLmRpcm5hbWUob3B0aW9ucy5kZXBzTG9ja0ZpbGVQYXRoKSwge1xuICAgICAgYXNzZXRIYXNoVHlwZTogY2RrLkFzc2V0SGFzaFR5cGUuT1VUUFVULFxuICAgICAgYnVuZGxpbmc6IG5ldyBCdW5kbGluZyhvcHRpb25zKSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBzdGF0aWMgY2xlYXJSdW5zTG9jYWxseUNhY2hlKCk6IHZvaWQge1xuICAgIHRoaXMucnVuc0xvY2FsbHkgPSB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIHN0YXRpYyBydW5zTG9jYWxseT86IGJvb2xlYW47XG5cbiAgLy8gQ29yZSBidW5kbGluZyBvcHRpb25zXG4gIHB1YmxpYyByZWFkb25seSBpbWFnZTogY2RrLkJ1bmRsaW5nRG9ja2VySW1hZ2U7XG4gIHB1YmxpYyByZWFkb25seSBjb21tYW5kOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IGVudmlyb25tZW50PzogeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgcHVibGljIHJlYWRvbmx5IHdvcmtpbmdEaXJlY3Rvcnk6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGxvY2FsPzogY2RrLklMb2NhbEJ1bmRsaW5nO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgcmVsYXRpdmVFbnRyeVBhdGg6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSByZWxhdGl2ZVRzY29uZmlnUGF0aD86IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBleHRlcm5hbHM6IHN0cmluZ1tdO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEJ1bmRsaW5nUHJvcHMpIHtcbiAgICBCdW5kbGluZy5ydW5zTG9jYWxseSA9IEJ1bmRsaW5nLnJ1bnNMb2NhbGx5XG4gICAgICA/PyBnZXRFc0J1aWxkVmVyc2lvbigpPy5zdGFydHNXaXRoKEVTQlVJTERfVkVSU0lPTilcbiAgICAgID8/IGZhbHNlO1xuXG4gICAgY29uc3QgcHJvamVjdFJvb3QgPSBwYXRoLmRpcm5hbWUocHJvcHMuZGVwc0xvY2tGaWxlUGF0aCk7XG4gICAgdGhpcy5yZWxhdGl2ZUVudHJ5UGF0aCA9IHBhdGgucmVsYXRpdmUocHJvamVjdFJvb3QsIHBhdGgucmVzb2x2ZShwcm9wcy5lbnRyeSkpO1xuXG4gICAgaWYgKHByb3BzLnRzY29uZmlnKSB7XG4gICAgICB0aGlzLnJlbGF0aXZlVHNjb25maWdQYXRoID0gcGF0aC5yZWxhdGl2ZShwcm9qZWN0Um9vdCwgcGF0aC5yZXNvbHZlKHByb3BzLnRzY29uZmlnKSk7XG4gICAgfVxuXG4gICAgdGhpcy5leHRlcm5hbHMgPSBbXG4gICAgICAuLi5wcm9wcy5leHRlcm5hbE1vZHVsZXMgPz8gWydhd3Mtc2RrJ10sIC8vIE1hcmsgYXdzLXNkayBhcyBleHRlcm5hbCBieSBkZWZhdWx0IChhdmFpbGFibGUgaW4gdGhlIHJ1bnRpbWUpXG4gICAgICAuLi5wcm9wcy5ub2RlTW9kdWxlcyA/PyBbXSwgLy8gTWFyayB0aGUgbW9kdWxlcyB0aGF0IHdlIGFyZSBnb2luZyB0byBpbnN0YWxsIGFzIGV4dGVybmFscyBhbHNvXG4gICAgXTtcblxuICAgIC8vIERvY2tlciBidW5kbGluZ1xuICAgIGNvbnN0IHNob3VsZEJ1aWxkSW1hZ2UgPSBwcm9wcy5mb3JjZURvY2tlckJ1bmRsaW5nIHx8ICFCdW5kbGluZy5ydW5zTG9jYWxseTtcbiAgICB0aGlzLmltYWdlID0gc2hvdWxkQnVpbGRJbWFnZVxuICAgICAgPyBwcm9wcy5kb2NrZXJJbWFnZSA/PyBjZGsuQnVuZGxpbmdEb2NrZXJJbWFnZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJy4uL2xpYicpLCB7XG4gICAgICAgIGJ1aWxkQXJnczoge1xuICAgICAgICAgIC4uLnByb3BzLmJ1aWxkQXJncyA/PyB7fSxcbiAgICAgICAgICBJTUFHRTogcHJvcHMucnVudGltZS5idW5kbGluZ0RvY2tlckltYWdlLmltYWdlLFxuICAgICAgICAgIEVTQlVJTERfVkVSU0lPTjogcHJvcHMuZXNidWlsZFZlcnNpb24gPz8gRVNCVUlMRF9WRVJTSU9OLFxuICAgICAgICB9LFxuICAgICAgfSlcbiAgICAgIDogY2RrLkJ1bmRsaW5nRG9ja2VySW1hZ2UuZnJvbVJlZ2lzdHJ5KCdkdW1teScpOyAvLyBEbyBub3QgYnVpbGQgaWYgd2UgZG9uJ3QgbmVlZCB0b1xuXG4gICAgY29uc3QgYnVuZGxpbmdDb21tYW5kID0gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQoY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19JTlBVVF9ESVIsIGNkay5Bc3NldFN0YWdpbmcuQlVORExJTkdfT1VUUFVUX0RJUik7XG4gICAgdGhpcy5jb21tYW5kID0gWydiYXNoJywgJy1jJywgYnVuZGxpbmdDb21tYW5kXTtcbiAgICB0aGlzLmVudmlyb25tZW50ID0gcHJvcHMuZW52aXJvbm1lbnQ7XG4gICAgLy8gQnVuZGxpbmcgc2V0cyB0aGUgd29ya2luZyBkaXJlY3RvcnkgdG8gY2RrLkFzc2V0U3RhZ2luZy5CVU5ETElOR19JTlBVVF9ESVJcbiAgICAvLyBhbmQgd2Ugd2FudCB0byBmb3JjZSBucHggdG8gdXNlIHRoZSBnbG9iYWxseSBpbnN0YWxsZWQgZXNidWlsZC5cbiAgICB0aGlzLndvcmtpbmdEaXJlY3RvcnkgPSAnLyc7XG5cbiAgICAvLyBMb2NhbCBidW5kbGluZ1xuICAgIGlmICghcHJvcHMuZm9yY2VEb2NrZXJCdW5kbGluZykgeyAvLyBvbmx5IGlmIERvY2tlciBpcyBub3QgZm9yY2VkXG4gICAgICBjb25zdCBvc1BsYXRmb3JtID0gb3MucGxhdGZvcm0oKTtcbiAgICAgIGNvbnN0IGNyZWF0ZUxvY2FsQ29tbWFuZCA9IChvdXRwdXREaXI6IHN0cmluZykgPT4gdGhpcy5jcmVhdGVCdW5kbGluZ0NvbW1hbmQocHJvamVjdFJvb3QsIG91dHB1dERpciwgb3NQbGF0Zm9ybSk7XG5cbiAgICAgIHRoaXMubG9jYWwgPSB7XG4gICAgICAgIHRyeUJ1bmRsZShvdXRwdXREaXI6IHN0cmluZykge1xuICAgICAgICAgIGlmIChCdW5kbGluZy5ydW5zTG9jYWxseSA9PT0gZmFsc2UpIHtcbiAgICAgICAgICAgIHByb2Nlc3Muc3RkZXJyLndyaXRlKCdlc2J1aWxkIGNhbm5vdCBydW4gbG9jYWxseS4gU3dpdGNoaW5nIHRvIERvY2tlciBidW5kbGluZy5cXG4nKTtcbiAgICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBjb25zdCBsb2NhbENvbW1hbmQgPSBjcmVhdGVMb2NhbENvbW1hbmQob3V0cHV0RGlyKTtcblxuICAgICAgICAgIGV4ZWMoXG4gICAgICAgICAgICBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ2NtZCcgOiAnYmFzaCcsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIG9zUGxhdGZvcm0gPT09ICd3aW4zMicgPyAnL2MnIDogJy1jJyxcbiAgICAgICAgICAgICAgbG9jYWxDb21tYW5kLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgZW52OiB7IC4uLnByb2Nlc3MuZW52LCAuLi5wcm9wcy5lbnZpcm9ubWVudCA/PyB7fSB9LFxuICAgICAgICAgICAgICBzdGRpbzogWyAvLyBzaG93IG91dHB1dFxuICAgICAgICAgICAgICAgICdpZ25vcmUnLCAvLyBpZ25vcmUgc3RkaW9cbiAgICAgICAgICAgICAgICBwcm9jZXNzLnN0ZGVyciwgLy8gcmVkaXJlY3Qgc3Rkb3V0IHRvIHN0ZGVyclxuICAgICAgICAgICAgICAgICdpbmhlcml0JywgLy8gaW5oZXJpdCBzdGRlcnJcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgY3dkOiBwYXRoLmRpcm5hbWUocHJvcHMuZW50cnkpLFxuICAgICAgICAgICAgICB3aW5kb3dzVmVyYmF0aW1Bcmd1bWVudHM6IG9zUGxhdGZvcm0gPT09ICd3aW4zMicsXG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgY3JlYXRlQnVuZGxpbmdDb21tYW5kKGlucHV0RGlyOiBzdHJpbmcsIG91dHB1dERpcjogc3RyaW5nLCBvc1BsYXRmb3JtOiBOb2RlSlMuUGxhdGZvcm0gPSAnbGludXgnKTogc3RyaW5nIHtcbiAgICBjb25zdCBwYXRoSm9pbiA9IG9zUGF0aEpvaW4ob3NQbGF0Zm9ybSk7XG5cbiAgICBjb25zdCBucHggPSBvc1BsYXRmb3JtID09PSAnd2luMzInID8gJ25weC5jbWQnIDogJ25weCc7XG4gICAgY29uc3QgbG9hZGVycyA9IE9iamVjdC5lbnRyaWVzKHRoaXMucHJvcHMubG9hZGVyID8/IHt9KTtcbiAgICBjb25zdCBkZWZpbmVzID0gT2JqZWN0LmVudHJpZXModGhpcy5wcm9wcy5kZWZpbmUgPz8ge30pO1xuXG4gICAgY29uc3QgZXNidWlsZENvbW1hbmQ6IHN0cmluZyA9IFtcbiAgICAgIG5weCwgJ2VzYnVpbGQnLFxuICAgICAgJy0tYnVuZGxlJywgcGF0aEpvaW4oaW5wdXREaXIsIHRoaXMucmVsYXRpdmVFbnRyeVBhdGgpLFxuICAgICAgYC0tdGFyZ2V0PSR7dGhpcy5wcm9wcy50YXJnZXQgPz8gdG9UYXJnZXQodGhpcy5wcm9wcy5ydW50aW1lKX1gLFxuICAgICAgJy0tcGxhdGZvcm09bm9kZScsXG4gICAgICBgLS1vdXRmaWxlPSR7cGF0aEpvaW4ob3V0cHV0RGlyLCAnaW5kZXguanMnKX1gLFxuICAgICAgLi4udGhpcy5wcm9wcy5taW5pZnkgPyBbJy0tbWluaWZ5J10gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMuc291cmNlTWFwID8gWyctLXNvdXJjZW1hcCddIDogW10sXG4gICAgICAuLi50aGlzLmV4dGVybmFscy5tYXAoZXh0ZXJuYWwgPT4gYC0tZXh0ZXJuYWw6JHtleHRlcm5hbH1gKSxcbiAgICAgIC4uLmxvYWRlcnMubWFwKChbZXh0LCBuYW1lXSkgPT4gYC0tbG9hZGVyOiR7ZXh0fT0ke25hbWV9YCksXG4gICAgICAuLi5kZWZpbmVzLm1hcCgoW2tleSwgdmFsdWVdKSA9PiBgLS1kZWZpbmU6JHtrZXl9PSR7dmFsdWV9YCksXG4gICAgICAuLi50aGlzLnByb3BzLmxvZ0xldmVsID8gW2AtLWxvZy1sZXZlbD0ke3RoaXMucHJvcHMubG9nTGV2ZWx9YF0gOiBbXSxcbiAgICAgIC4uLnRoaXMucHJvcHMua2VlcE5hbWVzID8gWyctLWtlZXAtbmFtZXMnXSA6IFtdLFxuICAgICAgLi4udGhpcy5yZWxhdGl2ZVRzY29uZmlnUGF0aCA/IFtgLS10c2NvbmZpZz0ke3BhdGhKb2luKGlucHV0RGlyLCB0aGlzLnJlbGF0aXZlVHNjb25maWdQYXRoKX1gXSA6IFtdLFxuICAgICAgLi4udGhpcy5wcm9wcy5tZXRhZmlsZSA/IFtgLS1tZXRhZmlsZT0ke3BhdGhKb2luKG91dHB1dERpciwgJ2luZGV4Lm1ldGEuanNvbicpfWBdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmJhbm5lciA/IFtgLS1iYW5uZXI9JyR7dGhpcy5wcm9wcy5iYW5uZXJ9J2BdIDogW10sXG4gICAgICAuLi50aGlzLnByb3BzLmZvb3RlciA/IFtgLS1mb290ZXI9JyR7dGhpcy5wcm9wcy5mb290ZXJ9J2BdIDogW10sXG4gICAgXS5qb2luKCcgJyk7XG5cbiAgICBsZXQgZGVwc0NvbW1hbmQgPSAnJztcbiAgICBpZiAodGhpcy5wcm9wcy5ub2RlTW9kdWxlcykge1xuICAgICAgLy8gRmluZCAncGFja2FnZS5qc29uJyBjbG9zZXN0IHRvIGVudHJ5IGZvbGRlciwgd2UgYXJlIGdvaW5nIHRvIGV4dHJhY3QgdGhlXG4gICAgICAvLyBtb2R1bGVzIHZlcnNpb25zIGZyb20gaXQuXG4gICAgICBjb25zdCBwa2dQYXRoID0gZmluZFVwKCdwYWNrYWdlLmpzb24nLCBwYXRoLmRpcm5hbWUodGhpcy5wcm9wcy5lbnRyeSkpO1xuICAgICAgaWYgKCFwa2dQYXRoKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgYSBgcGFja2FnZS5qc29uYCBpbiB0aGlzIHByb2plY3QuIFVzaW5nIGBub2RlTW9kdWxlc2AgcmVxdWlyZXMgYSBgcGFja2FnZS5qc29uYC4nKTtcbiAgICAgIH1cblxuICAgICAgLy8gRGV0ZXJtaW5lIGRlcGVuZGVuY2llcyB2ZXJzaW9ucywgbG9jayBmaWxlIGFuZCBpbnN0YWxsZXJcbiAgICAgIGNvbnN0IGRlcGVuZGVuY2llcyA9IGV4dHJhY3REZXBlbmRlbmNpZXMocGtnUGF0aCwgdGhpcy5wcm9wcy5ub2RlTW9kdWxlcyk7XG4gICAgICBsZXQgaW5zdGFsbGVyID0gSW5zdGFsbGVyLk5QTTtcbiAgICAgIGxldCBsb2NrRmlsZSA9IExvY2tGaWxlLk5QTTtcbiAgICAgIGlmICh0aGlzLnByb3BzLmRlcHNMb2NrRmlsZVBhdGguZW5kc1dpdGgoTG9ja0ZpbGUuWUFSTikpIHtcbiAgICAgICAgbG9ja0ZpbGUgPSBMb2NrRmlsZS5ZQVJOO1xuICAgICAgICBpbnN0YWxsZXIgPSBJbnN0YWxsZXIuWUFSTjtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3NDb21tYW5kID0gbmV3IE9zQ29tbWFuZChvc1BsYXRmb3JtKTtcblxuICAgICAgLy8gQ3JlYXRlIGR1bW15IHBhY2thZ2UuanNvbiwgY29weSBsb2NrIGZpbGUgaWYgYW55IGFuZCB0aGVuIGluc3RhbGxcbiAgICAgIGRlcHNDb21tYW5kID0gY2hhaW4oW1xuICAgICAgICBvc0NvbW1hbmQud3JpdGVKc29uKHBhdGhKb2luKG91dHB1dERpciwgJ3BhY2thZ2UuanNvbicpLCB7IGRlcGVuZGVuY2llcyB9KSxcbiAgICAgICAgb3NDb21tYW5kLmNvcHkocGF0aEpvaW4oaW5wdXREaXIsIGxvY2tGaWxlKSwgcGF0aEpvaW4ob3V0cHV0RGlyLCBsb2NrRmlsZSkpLFxuICAgICAgICBvc0NvbW1hbmQuY2hhbmdlRGlyZWN0b3J5KG91dHB1dERpciksXG4gICAgICAgIGAke2luc3RhbGxlcn0gaW5zdGFsbGAsXG4gICAgICBdKTtcbiAgICB9XG5cbiAgICByZXR1cm4gY2hhaW4oW1xuICAgICAgLi4udGhpcy5wcm9wcy5jb21tYW5kSG9va3M/LmJlZm9yZUJ1bmRsaW5nKGlucHV0RGlyLCBvdXRwdXREaXIpID8/IFtdLFxuICAgICAgZXNidWlsZENvbW1hbmQsXG4gICAgICAuLi4odGhpcy5wcm9wcy5ub2RlTW9kdWxlcyAmJiB0aGlzLnByb3BzLmNvbW1hbmRIb29rcz8uYmVmb3JlSW5zdGFsbChpbnB1dERpciwgb3V0cHV0RGlyKSkgPz8gW10sXG4gICAgICBkZXBzQ29tbWFuZCxcbiAgICAgIC4uLnRoaXMucHJvcHMuY29tbWFuZEhvb2tzPy5hZnRlckJ1bmRsaW5nKGlucHV0RGlyLCBvdXRwdXREaXIpID8/IFtdLFxuICAgIF0pO1xuICB9XG59XG5cbmVudW0gSW5zdGFsbGVyIHtcbiAgTlBNID0gJ25wbScsXG4gIFlBUk4gPSAneWFybicsXG59XG5cbi8qKlxuICogT1MgYWdub3N0aWMgY29tbWFuZFxuICovXG5jbGFzcyBPc0NvbW1hbmQge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IG9zUGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge31cblxuICBwdWJsaWMgd3JpdGVKc29uKGZpbGVQYXRoOiBzdHJpbmcsIGRhdGE6IGFueSk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RyaW5naWZpZWREYXRhID0gSlNPTi5zdHJpbmdpZnkoZGF0YSk7XG4gICAgaWYgKHRoaXMub3NQbGF0Zm9ybSA9PT0gJ3dpbjMyJykge1xuICAgICAgcmV0dXJuIGBlY2hvIF4ke3N0cmluZ2lmaWVkRGF0YX1eID4gJHtmaWxlUGF0aH1gO1xuICAgIH1cblxuICAgIHJldHVybiBgZWNobyAnJHtzdHJpbmdpZmllZERhdGF9JyA+ICR7ZmlsZVBhdGh9YDtcbiAgfVxuXG4gIHB1YmxpYyBjb3B5KHNyYzogc3RyaW5nLCBkZXN0OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLm9zUGxhdGZvcm0gPT09ICd3aW4zMicpIHtcbiAgICAgIHJldHVybiBgY29weSAke3NyY30gJHtkZXN0fWA7XG4gICAgfVxuXG4gICAgcmV0dXJuIGBjcCAke3NyY30gJHtkZXN0fWA7XG4gIH1cblxuICBwdWJsaWMgY2hhbmdlRGlyZWN0b3J5KGRpcjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gYGNkICR7ZGlyfWA7XG4gIH1cbn1cblxuLyoqXG4gKiBDaGFpbiBjb21tYW5kc1xuICovXG5mdW5jdGlvbiBjaGFpbihjb21tYW5kczogc3RyaW5nW10pOiBzdHJpbmcge1xuICByZXR1cm4gY29tbWFuZHMuZmlsdGVyKGMgPT4gISFjKS5qb2luKCcgJiYgJyk7XG59XG5cbi8qKlxuICogUGxhdGZvcm0gc3BlY2lmaWMgcGF0aCBqb2luXG4gKi9cbmZ1bmN0aW9uIG9zUGF0aEpvaW4ocGxhdGZvcm06IE5vZGVKUy5QbGF0Zm9ybSkge1xuICByZXR1cm4gZnVuY3Rpb24oLi4ucGF0aHM6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgICBjb25zdCBqb2luZWQgPSBwYXRoLmpvaW4oLi4ucGF0aHMpO1xuICAgIC8vIElmIHdlIGFyZSBvbiB3aW4zMiBidXQgbmVlZCBwb3NpeCBzdHlsZSBwYXRoc1xuICAgIGlmIChvcy5wbGF0Zm9ybSgpID09PSAnd2luMzInICYmIHBsYXRmb3JtICE9PSAnd2luMzInKSB7XG4gICAgICByZXR1cm4gam9pbmVkLnJlcGxhY2UoL1xcXFwvZywgJy8nKTtcbiAgICB9XG4gICAgcmV0dXJuIGpvaW5lZDtcbiAgfTtcbn1cblxuLyoqXG4gKiBDb252ZXJ0cyBhIHJ1bnRpbWUgdG8gYW4gZXNidWlsZCBub2RlIHRhcmdldFxuICovXG5mdW5jdGlvbiB0b1RhcmdldChydW50aW1lOiBSdW50aW1lKTogc3RyaW5nIHtcbiAgY29uc3QgbWF0Y2ggPSBydW50aW1lLm5hbWUubWF0Y2goL25vZGVqcyhcXGQrKS8pO1xuXG4gIGlmICghbWF0Y2gpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBleHRyYWN0IHZlcnNpb24gZnJvbSBydW50aW1lLicpO1xuICB9XG5cbiAgcmV0dXJuIGBub2RlJHttYXRjaFsxXX1gO1xufVxuIl19