"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, poetryWithoutUrls, commandHooks, assetExcludes = [], } = props;
        const outputPath = path.posix.join(core_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            poetryWithoutUrls,
            commandHooks,
            assetExcludes,
        });
        this.image = image ?? core_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes, options.poetryWithoutUrls);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        const exclusionStr = options.assetExcludes?.map(item => `--exclude='${item}'`).join(' ');
        bundlingCommands.push([
            'rsync', '-rLv', exclusionStr ?? '', `${options.inputDir}/`, options.outputDir,
        ].filter(item => item).join(' '));
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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