"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    image;
    entrypoint;
    command;
    volumes;
    volumesFrom;
    environment;
    workingDirectory;
    user;
    securityOpt;
    network;
    bundlingFileAccess;
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, poetryWithoutUrls, commandHooks, assetExcludes = [], } = props;
        const outputPath = path.posix.join(core_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            poetryWithoutUrls,
            commandHooks,
            assetExcludes,
        });
        this.image = image ?? core_1.DockerImage.fromBuild(path.join(__dirname, '..', 'lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
            network: props.network,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes, options.poetryWithoutUrls);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        const excludes = options.assetExcludes ?? [];
        if (packaging.dependenciesFile == packaging_1.DependenciesFile.UV && !excludes.includes('.python-version')) {
            excludes.push('.python-version');
        }
        const exclusionStr = excludes.map(item => `--exclude='${item}'`).join(' ');
        bundlingCommands.push([
            'rsync', '-rLv', exclusionStr ?? '', `${options.inputDir}/`, options.outputDir,
        ].filter(item => item).join(' '));
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile == packaging_1.DependenciesFile.UV) {
            bundlingCommands.push(`uv pip install -r ${packaging_1.DependenciesFile.PIP} --target ${options.outputDir}`);
        }
        else if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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