"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, } = props;
        const outputPath = path.posix.join(aws_cdk_lib_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: aws_cdk_lib_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
        });
        this.image = image ?? aws_cdk_lib_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = ['bash', '-c', chain(bundlingCommands)];
        this.environment = props.environment;
    }
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry);
        let bundlingCommands = [];
        bundlingCommands.push(`cp -rTL ${options.inputDir}/ ${options.outputDir}`);
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVuZGxpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidW5kbGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsdURBQWdGO0FBQ2hGLDZDQUErRjtBQUMvRiwyQ0FBMEQ7QUFHMUQ7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7QUFFN0M7O0dBRUc7QUFDVSxRQUFBLDBCQUEwQixHQUFHLG1CQUFtQixDQUFDO0FBK0I5RDs7R0FFRztBQUNILE1BQWEsUUFBUTtJQWNuQixZQUFZLEtBQW9CO1FBQzlCLE1BQU0sRUFDSixLQUFLLEVBQ0wsT0FBTyxFQUNQLFlBQVksR0FBRyx5QkFBWSxDQUFDLE1BQU0sRUFDbEMsZ0JBQWdCLEdBQUcsRUFBRSxFQUNyQixLQUFLLEdBQ04sR0FBRyxLQUFLLENBQUM7UUFFVixNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQywwQkFBWSxDQUFDLG1CQUFtQixFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFFdkYsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMscUJBQXFCLENBQUM7WUFDbEQsS0FBSztZQUNMLFFBQVEsRUFBRSwwQkFBWSxDQUFDLGtCQUFrQjtZQUN6QyxTQUFTLEVBQUUsVUFBVTtTQUN0QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssSUFBSSx5QkFBVyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsRUFBRTtZQUMxRSxTQUFTLEVBQUU7Z0JBQ1QsR0FBRyxLQUFLLENBQUMsU0FBUztnQkFDbEIsS0FBSyxFQUFFLE9BQU8sQ0FBQyxhQUFhLENBQUMsS0FBSzthQUNuQztZQUNELFFBQVEsRUFBRSxZQUFZLENBQUMsY0FBYztTQUN0QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztLQUN0QztJQXZDTSxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQXNCO1FBQ3pDLE9BQU8saUJBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRTtZQUNuQyxTQUFTLEVBQUUsT0FBTyxDQUFDLFNBQVM7WUFDNUIsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhO1lBQ3BDLE9BQU8sRUFBRSwyQkFBbUI7WUFDNUIsUUFBUSxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDO1NBQzNELENBQUMsQ0FBQztLQUNKO0lBa0NPLHFCQUFxQixDQUFDLE9BQStCO1FBQzNELE1BQU0sU0FBUyxHQUFHLHFCQUFTLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNyRCxJQUFJLGdCQUFnQixHQUFhLEVBQUUsQ0FBQztRQUNwQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsV0FBVyxPQUFPLENBQUMsUUFBUSxLQUFLLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1FBQzNFLGdCQUFnQixDQUFDLElBQUksQ0FBQyxNQUFNLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1FBQ2pELGdCQUFnQixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3JELElBQUksU0FBUyxDQUFDLGdCQUFnQixFQUFFO1lBQzlCLGdCQUFnQixDQUFDLElBQUksQ0FBQyw0QkFBNEIsNEJBQWdCLENBQUMsR0FBRyxPQUFPLE9BQU8sQ0FBQyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1NBQ25HO1FBQ0QsT0FBTyxnQkFBZ0IsQ0FBQztLQUN6QjtDQUNGO0FBckRELDRCQXFEQztBQVFEOztHQUVHO0FBQ0gsU0FBUyxLQUFLLENBQUMsUUFBa0I7SUFDL0IsT0FBTyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNoRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IEFyY2hpdGVjdHVyZSwgQXNzZXRDb2RlLCBDb2RlLCBSdW50aW1lIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBBc3NldFN0YWdpbmcsIEJ1bmRsaW5nT3B0aW9ucyBhcyBDZGtCdW5kbGluZ09wdGlvbnMsIERvY2tlckltYWdlIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgUGFja2FnaW5nLCBEZXBlbmRlbmNpZXNGaWxlIH0gZnJvbSAnLi9wYWNrYWdpbmcnO1xuaW1wb3J0IHsgQnVuZGxpbmdPcHRpb25zIH0gZnJvbSAnLi90eXBlcyc7XG5cbi8qKlxuICogRGVwZW5kZW5jeSBmaWxlcyB0byBleGNsdWRlIGZyb20gdGhlIGFzc2V0IGhhc2guXG4gKi9cbmV4cG9ydCBjb25zdCBERVBFTkRFTkNZX0VYQ0xVREVTID0gWycqLnB5YyddO1xuXG4vKipcbiAqIFRoZSBsb2NhdGlvbiBpbiB0aGUgaW1hZ2UgdGhhdCB0aGUgYnVuZGxlciBpbWFnZSBjYWNoZXMgZGVwZW5kZW5jaWVzLlxuICovXG5leHBvcnQgY29uc3QgQlVORExFUl9ERVBFTkRFTkNJRVNfQ0FDSEUgPSAnL3Zhci9kZXBlbmRlbmNpZXMnO1xuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGJ1bmRsaW5nXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQnVuZGxpbmdQcm9wcyBleHRlbmRzIEJ1bmRsaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBFbnRyeSBwYXRoXG4gICAqL1xuICByZWFkb25seSBlbnRyeTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcnVudGltZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU6IFJ1bnRpbWU7XG5cbiAgLyoqXG4gICAqIFRoZSBzeXN0ZW0gYXJjaGl0ZWN0dXJlIG9mIHRoZSBsYW1iZGEgZnVuY3Rpb25cbiAgICpcbiAgICogQGRlZmF1bHQgQXJjaGl0ZWN0dXJlLlg4Nl82NFxuICAgKi9cbiAgcmVhZG9ubHkgYXJjaGl0ZWN0dXJlPzogQXJjaGl0ZWN0dXJlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0aGUgYnVuZGxpbmcgcHJvY2VzcyBzaG91bGQgYmUgc2tpcHBlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIERvZXMgbm90IHNraXAgYnVuZGxpbmdcbiAgICovXG4gIHJlYWRvbmx5IHNraXA/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFByb2R1Y2UgYnVuZGxlZCBMYW1iZGEgYXNzZXQgY29kZVxuICovXG5leHBvcnQgY2xhc3MgQnVuZGxpbmcgaW1wbGVtZW50cyBDZGtCdW5kbGluZ09wdGlvbnMge1xuICBwdWJsaWMgc3RhdGljIGJ1bmRsZShvcHRpb25zOiBCdW5kbGluZ1Byb3BzKTogQXNzZXRDb2RlIHtcbiAgICByZXR1cm4gQ29kZS5mcm9tQXNzZXQob3B0aW9ucy5lbnRyeSwge1xuICAgICAgYXNzZXRIYXNoOiBvcHRpb25zLmFzc2V0SGFzaCxcbiAgICAgIGFzc2V0SGFzaFR5cGU6IG9wdGlvbnMuYXNzZXRIYXNoVHlwZSxcbiAgICAgIGV4Y2x1ZGU6IERFUEVOREVOQ1lfRVhDTFVERVMsXG4gICAgICBidW5kbGluZzogb3B0aW9ucy5za2lwID8gdW5kZWZpbmVkIDogbmV3IEJ1bmRsaW5nKG9wdGlvbnMpLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGltYWdlOiBEb2NrZXJJbWFnZTtcbiAgcHVibGljIHJlYWRvbmx5IGNvbW1hbmQ6IHN0cmluZ1tdO1xuICBwdWJsaWMgcmVhZG9ubHkgZW52aXJvbm1lbnQ/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBCdW5kbGluZ1Byb3BzKSB7XG4gICAgY29uc3Qge1xuICAgICAgZW50cnksXG4gICAgICBydW50aW1lLFxuICAgICAgYXJjaGl0ZWN0dXJlID0gQXJjaGl0ZWN0dXJlLlg4Nl82NCxcbiAgICAgIG91dHB1dFBhdGhTdWZmaXggPSAnJyxcbiAgICAgIGltYWdlLFxuICAgIH0gPSBwcm9wcztcblxuICAgIGNvbnN0IG91dHB1dFBhdGggPSBwYXRoLnBvc2l4LmpvaW4oQXNzZXRTdGFnaW5nLkJVTkRMSU5HX09VVFBVVF9ESVIsIG91dHB1dFBhdGhTdWZmaXgpO1xuXG4gICAgY29uc3QgYnVuZGxpbmdDb21tYW5kcyA9IHRoaXMuY3JlYXRlQnVuZGxpbmdDb21tYW5kKHtcbiAgICAgIGVudHJ5LFxuICAgICAgaW5wdXREaXI6IEFzc2V0U3RhZ2luZy5CVU5ETElOR19JTlBVVF9ESVIsXG4gICAgICBvdXRwdXREaXI6IG91dHB1dFBhdGgsXG4gICAgfSk7XG5cbiAgICB0aGlzLmltYWdlID0gaW1hZ2UgPz8gRG9ja2VySW1hZ2UuZnJvbUJ1aWxkKHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9saWInKSwge1xuICAgICAgYnVpbGRBcmdzOiB7XG4gICAgICAgIC4uLnByb3BzLmJ1aWxkQXJncyxcbiAgICAgICAgSU1BR0U6IHJ1bnRpbWUuYnVuZGxpbmdJbWFnZS5pbWFnZSxcbiAgICAgIH0sXG4gICAgICBwbGF0Zm9ybTogYXJjaGl0ZWN0dXJlLmRvY2tlclBsYXRmb3JtLFxuICAgIH0pO1xuICAgIHRoaXMuY29tbWFuZCA9IFsnYmFzaCcsICctYycsIGNoYWluKGJ1bmRsaW5nQ29tbWFuZHMpXTtcbiAgICB0aGlzLmVudmlyb25tZW50ID0gcHJvcHMuZW52aXJvbm1lbnQ7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUJ1bmRsaW5nQ29tbWFuZChvcHRpb25zOiBCdW5kbGluZ0NvbW1hbmRPcHRpb25zKTogc3RyaW5nW10ge1xuICAgIGNvbnN0IHBhY2thZ2luZyA9IFBhY2thZ2luZy5mcm9tRW50cnkob3B0aW9ucy5lbnRyeSk7XG4gICAgbGV0IGJ1bmRsaW5nQ29tbWFuZHM6IHN0cmluZ1tdID0gW107XG4gICAgYnVuZGxpbmdDb21tYW5kcy5wdXNoKGBjcCAtclRMICR7b3B0aW9ucy5pbnB1dERpcn0vICR7b3B0aW9ucy5vdXRwdXREaXJ9YCk7XG4gICAgYnVuZGxpbmdDb21tYW5kcy5wdXNoKGBjZCAke29wdGlvbnMub3V0cHV0RGlyfWApO1xuICAgIGJ1bmRsaW5nQ29tbWFuZHMucHVzaChwYWNrYWdpbmcuZXhwb3J0Q29tbWFuZCA/PyAnJyk7XG4gICAgaWYgKHBhY2thZ2luZy5kZXBlbmRlbmNpZXNGaWxlKSB7XG4gICAgICBidW5kbGluZ0NvbW1hbmRzLnB1c2goYHB5dGhvbiAtbSBwaXAgaW5zdGFsbCAtciAke0RlcGVuZGVuY2llc0ZpbGUuUElQfSAtdCAke29wdGlvbnMub3V0cHV0RGlyfWApO1xuICAgIH1cbiAgICByZXR1cm4gYnVuZGxpbmdDb21tYW5kcztcbiAgfVxufVxuXG5pbnRlcmZhY2UgQnVuZGxpbmdDb21tYW5kT3B0aW9ucyB7XG4gIHJlYWRvbmx5IGVudHJ5OiBzdHJpbmc7XG4gIHJlYWRvbmx5IGlucHV0RGlyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IG91dHB1dERpcjogc3RyaW5nO1xufVxuXG4vKipcbiAqIENoYWluIGNvbW1hbmRzXG4gKi9cbmZ1bmN0aW9uIGNoYWluKGNvbW1hbmRzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gIHJldHVybiBjb21tYW5kcy5maWx0ZXIoYyA9PiAhIWMpLmpvaW4oJyAmJiAnKTtcbn1cbiJdfQ==