"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile = exports.DependenciesFile || (exports.DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv lock -r > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else {
            return this.withNoPackaging();
        }
    }
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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