"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stageDependencies = exports.bundle = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
function bundle(options) {
    const { entry, runtime, outputPathSuffix } = options;
    const stagedir = cdk.FileSystem.mkdtemp('python-bundling-');
    const hasDeps = stageDependencies(entry, stagedir);
    const depsCommand = chain([
        hasDeps ? `rsync -r ${exports.BUNDLER_DEPENDENCIES_CACHE}/. ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}` : '',
        `rsync -r . ${cdk.AssetStaging.BUNDLING_OUTPUT_DIR}/${outputPathSuffix}`,
    ]);
    // Determine which dockerfile to use. When dependencies are present, we use a
    // Dockerfile that can create a cacheable layer. We can't use this Dockerfile
    // if there aren't dependencies or the Dockerfile will complain about missing
    // sources.
    const dockerfile = hasDeps
        ? 'Dockerfile.dependencies'
        : 'Dockerfile';
    // copy Dockerfile to workdir
    fs.copyFileSync(path.join(__dirname, dockerfile), path.join(stagedir, dockerfile));
    const image = cdk.BundlingDockerImage.fromAsset(stagedir, {
        buildArgs: {
            IMAGE: runtime.bundlingDockerImage.image,
        },
        file: dockerfile,
    });
    return lambda.Code.fromAsset(entry, {
        assetHashType: options.assetHashType,
        assetHash: options.assetHash,
        exclude: exports.DEPENDENCY_EXCLUDES,
        bundling: {
            image,
            command: ['bash', '-c', depsCommand],
        },
    });
}
exports.bundle = bundle;
/**
 * Checks to see if the `entry` directory contains a type of dependency that
 * we know how to install.
 */
function stageDependencies(entry, stagedir) {
    const prefixes = [
        'Pipfile',
        'pyproject',
        'poetry',
        'requirements.txt',
    ];
    let found = false;
    for (const file of fs.readdirSync(entry)) {
        for (const prefix of prefixes) {
            if (file.startsWith(prefix)) {
                fs.copyFileSync(path.join(entry, file), path.join(stagedir, file));
                found = true;
            }
        }
    }
    return found;
}
exports.stageDependencies = stageDependencies;
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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