"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const api = require("aws-cdk-lib/aws-apigateway");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
test('Error on existingLambdaObj=undefined', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {};
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
test('Test with lambdaFunctionProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                OVERRIDE_STATUS: 'true'
            }
        },
        apiGatewayProps: {
            description: "sampleApiProp"
        }
    };
    new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Environment: {
            Variables: {
                OVERRIDE_STATUS: "true",
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1"
            }
        }
    });
    template.resourceCountIs('AWS::ApiGateway::UsagePlan', 1);
});
test('Test properties', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    };
    const app = new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction).toBeDefined();
    // Assertion 2
    expect(app.apiGateway).toBeDefined();
    expect(app.apiGatewayCloudWatchRole).toBeDefined();
    expect(app.apiGatewayLogGroup).toBeDefined();
});
test('Error on lambdaFunctionProps=undefined', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {};
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion 1
    expect(app).toThrowError();
});
test('Test deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToLambda(stack, 'api-gateway-lambda', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        },
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ApiGateway::Method", {
        HttpMethod: "ANY",
        AuthorizationType: "NONE"
    });
});
test('Test deployment ApiGateway override cloudWatchRole = false', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToLambda(stack, 'api-gateway-lambda', {
        apiGatewayProps: {
            cloudWatchRole: false
        },
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    defaults.expectNonexistence(stack, "AWS::ApiGateway::Account", {});
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                OVERRIDE_STATUS: 'true'
            }
        },
        existingLambdaObj: lambdaFunction,
        apiGatewayProps: {
            description: "sampleApiProp"
        }
    };
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
test('Confirm call to CheckApiProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        apiGatewayProps: {
            defaultMethodOptions: {
                apiKeyRequired: true
            },
        },
        createUsagePlan: false,
    };
    const app = () => {
        new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    };
    // Assertion
    expect(app).toThrowError('Error - if API key is required, then the Usage plan must be created\n');
});
test('Confirm suppression of Usage Plan', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        lambdaFunctionProps: {
            code: new lambda.InlineCode('exports.handler = async (event) => { console.log(event); return {\'statusCode\': 200, \'body\': \'\'}; }'),
            runtime: lambda.Runtime.NODEJS_20_X,
            handler: 'index.handler',
        },
        createUsagePlan: false
    };
    new lib_1.ApiGatewayToLambda(stack, 'test-apigateway-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ApiGateway::UsagePlan', 0);
});
//# sourceMappingURL=data:application/json;base64,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