"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const apigateway = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const core_2 = require("@aws-cdk/core");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const core_3 = require("@aws-solutions-constructs/core");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_3.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: core_2.Duration.hours(8),
    minTtl: core_2.Duration.hours(5),
    maxTtl: core_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: core_2.Duration.minutes(0),
    minTtl: core_2.Duration.minutes(0),
    maxTtl: core_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.NONE,
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    },
    cloudFrontLoggingBucketProps: {
        removalPolicy: core_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    },
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
core_3.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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