"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontToApiGatewayToLambda = void 0;
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const aws_cloudfront_apigateway_1 = require("@aws-solutions-constructs/aws-cloudfront-apigateway");
class CloudFrontToApiGatewayToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the CloudFrontToApiGatewayToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] =
            defaults.RegionalLambdaRestApi(this, this.lambdaFunction, props.apiGatewayProps);
        this.apiGateway.methods.forEach((apiMethod) => {
            // Override the API Gateway Authorization Type from AWS_IAM to NONE
            const child = apiMethod.node.findChild('Resource');
            if (child.authorizationType === 'AWS_IAM') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
                child.cfnOptions.metadata = {
                    cfn_nag: {
                        rules_to_suppress: [{
                                id: 'W59',
                                reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                            }]
                    }
                };
            }
        });
        const apiCloudfront = new aws_cloudfront_apigateway_1.CloudFrontToApiGateway(this, 'CloudFrontToApiGateway', {
            existingApiGatewayObj: this.apiGateway,
            cloudFrontDistributionProps: props.cloudFrontDistributionProps,
            insertHttpSecurityHeaders: props.insertHttpSecurityHeaders
        });
        this.cloudFrontWebDistribution = apiCloudfront.cloudFrontWebDistribution;
        this.edgeLambdaFunctionVersion = apiCloudfront.edgeLambdaFunctionVersion;
        this.cloudFrontLoggingBucket = apiCloudfront.cloudFrontLoggingBucket;
    }
}
exports.CloudFrontToApiGatewayToLambda = CloudFrontToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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