"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const apigateway = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const duration_1 = require("@aws-cdk/core/lib/duration");
const origins = require("@aws-cdk/aws-cloudfront-origins");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, 'test-cf-api-lambda-override-stack');
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_10_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: duration_1.Duration.hours(8),
    minTtl: duration_1.Duration.hours(5),
    maxTtl: duration_1.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: duration_1.Duration.minutes(0),
    minTtl: duration_1.Duration.minutes(0),
    maxTtl: duration_1.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.NONE,
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    }
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWcub3ZlcnJpZGUtYmVoYXZpb3IuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnRlZy5vdmVycmlkZS1iZWhhdmlvci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsZ0JBQWdCO0FBQ2hCLHdDQUEyQztBQUMzQyxnQ0FBd0Q7QUFDeEQsOENBQThDO0FBQzlDLHNEQUFzRDtBQUN0RCxzREFBc0Q7QUFDdEQscUNBQXFDO0FBQ3JDLHlEQUFzRDtBQUN0RCwyREFBMkQ7QUFFM0QsUUFBUTtBQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksVUFBRyxFQUFFLENBQUM7QUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLG1DQUFtQyxDQUFDLENBQUM7QUFDbEUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxXQUFXLEdBQUcsdURBQXVELENBQUM7QUFFNUYsTUFBTSxXQUFXLEdBQXlCO0lBQ3RDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO0lBQ2xELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7SUFDbkMsT0FBTyxFQUFFLGVBQWU7Q0FDM0IsQ0FBQztBQUVGLGtDQUFrQztBQUNsQyxNQUFNLGVBQWUsR0FBRyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFO0lBQ3pFLGVBQWUsRUFBRSxpQkFBaUI7SUFDbEMsVUFBVSxFQUFFLG1CQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztJQUM3QixNQUFNLEVBQUUsbUJBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO0lBQ3pCLE1BQU0sRUFBRSxtQkFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7Q0FDN0IsQ0FBQyxDQUFDO0FBRUgsc0NBQXNDO0FBQ3RDLE1BQU0sYUFBYSxHQUFHLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO0lBQ3JFLGVBQWUsRUFBRSxlQUFlO0lBQ2hDLFVBQVUsRUFBRSxtQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDL0IsTUFBTSxFQUFFLG1CQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUMzQixNQUFNLEVBQUUsbUJBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0NBQzlCLENBQUMsQ0FBQztBQUVILE1BQU0sU0FBUyxHQUFHLElBQUksb0NBQThCLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFO0lBQ2xGLG1CQUFtQixFQUFFLFdBQVc7SUFDaEMsZUFBZSxFQUFFO1FBQ2IsS0FBSyxFQUFFLEtBQUs7UUFDWixvQkFBb0IsRUFBRTtZQUNsQixpQkFBaUIsRUFBRSxVQUFVLENBQUMsaUJBQWlCLENBQUMsSUFBSTtTQUN2RDtLQUNKO0lBQ0QsMkJBQTJCLEVBQUU7UUFDekIsZUFBZSxFQUFFO1lBQ2IsV0FBVyxFQUFFLGVBQWU7U0FDL0I7S0FDSjtDQUNKLENBQUMsQ0FBQztBQUVILE1BQU0sV0FBVyxHQUFHLFNBQVMsQ0FBQyxVQUFVLENBQUM7QUFDekMsTUFBTSw2QkFBNkIsR0FBRyxHQUFHLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsR0FBRyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0FBQzdGLE1BQU0scUJBQXFCLEdBQUcsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSw2QkFBNkIsQ0FBQyxDQUFDLENBQUM7QUFFakcsTUFBTSxjQUFjLEdBQUcsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0FBQ3ZFLE1BQU0sWUFBWSxHQUFHLGNBQWMsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksVUFBVSxDQUFDLGVBQWUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUM7QUFFMUcsTUFBTSxlQUFlLEdBQUcsU0FBUyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0FBQ3pFLE1BQU0sYUFBYSxHQUFHLGVBQWUsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7QUFFdkQsZUFBZTtBQUNmLFNBQVMsQ0FBQyx5QkFBeUIsQ0FBQyxXQUFXLENBQUMsVUFBVSxFQUFFLElBQUksT0FBTyxDQUFDLFVBQVUsQ0FBQyxxQkFBcUIsRUFBRTtJQUN0RyxVQUFVLEVBQUUsSUFBSSxXQUFXLENBQUMsZUFBZSxDQUFDLFNBQVMsVUFBVTtDQUM5RCxDQUFDLEVBQUU7SUFDSixXQUFXLEVBQUUsYUFBYTtDQUM3QixDQUFDLENBQUM7QUFDSCw0QkFBNEI7QUFDNUIsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLENBQUM7QUFDL0IsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUM7QUFFaEMsU0FBUyxnQkFBZ0IsQ0FBQyxNQUF5QjtJQUMvQyxNQUFNLEtBQUssR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQXlCLENBQUM7SUFDeEUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEdBQUc7UUFDeEIsT0FBTyxFQUFFO1lBQ0wsaUJBQWlCLEVBQUUsQ0FBQztvQkFDaEIsRUFBRSxFQUFFLEtBQUs7b0JBQ1QsTUFBTSxFQUFFLDBJQUEwSTtpQkFDckosQ0FBQztTQUNMO0tBQ0osQ0FBQztBQUNOLENBQUM7QUFFRCxRQUFRO0FBQ1IsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMjAgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLy8gIWNkay1pbnRlZyAqXG5pbXBvcnQgeyBBcHAsIFN0YWNrIH0gZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcbmltcG9ydCB7IENsb3VkRnJvbnRUb0FwaUdhdGV3YXlUb0xhbWJkYSB9IGZyb20gXCIuLi9saWJcIjtcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgYXBpZ2F0ZXdheSBmcm9tICdAYXdzLWNkay9hd3MtYXBpZ2F0ZXdheSc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBEdXJhdGlvbiB9IGZyb20gXCJAYXdzLWNkay9jb3JlL2xpYi9kdXJhdGlvblwiO1xuaW1wb3J0ICogYXMgb3JpZ2lucyBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmcm9udC1vcmlnaW5zJztcblxuLy8gU2V0dXBcbmNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbmNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ3Rlc3QtY2YtYXBpLWxhbWJkYS1vdmVycmlkZS1zdGFjaycpO1xuc3RhY2sudGVtcGxhdGVPcHRpb25zLmRlc2NyaXB0aW9uID0gJ0ludGVncmF0aW9uIFRlc3QgZm9yIGF3cy1jbG91ZGZyb250LWFwaWdhdGV3YXktbGFtYmRhJztcblxuY29uc3QgbGFtYmRhUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMF9YLFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xufTtcblxuLy8gU29tZSBDYWNoaW5nIGZvciBzdGF0aWMgY29udGVudFxuY29uc3Qgc29tZUNhY2hlUG9saWN5ID0gbmV3IGNsb3VkZnJvbnQuQ2FjaGVQb2xpY3koc3RhY2ssICdTb21lQ2FjaGVQb2xpY3knLCB7XG4gICAgY2FjaGVQb2xpY3lOYW1lOiAnU29tZUNhY2hlUG9saWN5JyxcbiAgICBkZWZhdWx0VHRsOiBEdXJhdGlvbi5ob3Vycyg4KSxcbiAgICBtaW5UdGw6IER1cmF0aW9uLmhvdXJzKDUpLFxuICAgIG1heFR0bDogRHVyYXRpb24uaG91cnMoMTApLFxufSk7XG5cbi8vIERpc2FibGUgQ2FjaGluZyBmb3IgZHluYW1pYyBjb250ZW50XG5jb25zdCBub0NhY2hlUG9saWN5ID0gbmV3IGNsb3VkZnJvbnQuQ2FjaGVQb2xpY3koc3RhY2ssICdOb0NhY2hlUG9saWN5Jywge1xuICAgIGNhY2hlUG9saWN5TmFtZTogJ05vQ2FjaGVQb2xpY3knLFxuICAgIGRlZmF1bHRUdGw6IER1cmF0aW9uLm1pbnV0ZXMoMCksXG4gICAgbWluVHRsOiBEdXJhdGlvbi5taW51dGVzKDApLFxuICAgIG1heFR0bDogRHVyYXRpb24ubWludXRlcygwKSxcbn0pO1xuXG5jb25zdCBjb25zdHJ1Y3QgPSBuZXcgQ2xvdWRGcm9udFRvQXBpR2F0ZXdheVRvTGFtYmRhKHN0YWNrLCAnY2YtYXBpLWxhbWJkYS1vdmVycmlkZScsIHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGFQcm9wcyxcbiAgICBhcGlHYXRld2F5UHJvcHM6IHtcbiAgICAgICAgcHJveHk6IGZhbHNlLFxuICAgICAgICBkZWZhdWx0TWV0aG9kT3B0aW9uczoge1xuICAgICAgICAgICAgYXV0aG9yaXphdGlvblR5cGU6IGFwaWdhdGV3YXkuQXV0aG9yaXphdGlvblR5cGUuTk9ORSxcbiAgICAgICAgfSxcbiAgICB9LFxuICAgIGNsb3VkRnJvbnREaXN0cmlidXRpb25Qcm9wczoge1xuICAgICAgICBkZWZhdWx0QmVoYXZpb3I6IHtcbiAgICAgICAgICAgIGNhY2hlUG9saWN5OiBzb21lQ2FjaGVQb2xpY3lcbiAgICAgICAgfVxuICAgIH1cbn0pO1xuXG5jb25zdCBhcGlFbmRQb2ludCA9IGNvbnN0cnVjdC5hcGlHYXRld2F5O1xuY29uc3QgYXBpRW5kUG9pbnRVcmxXaXRob3V0UHJvdG9jb2wgPSBjZGsuRm4uc2VsZWN0KDEsIGNkay5Gbi5zcGxpdChcIjovL1wiLCBhcGlFbmRQb2ludC51cmwpKTtcbmNvbnN0IGFwaUVuZFBvaW50RG9tYWluTmFtZSA9IGNkay5Gbi5zZWxlY3QoMCwgY2RrLkZuLnNwbGl0KFwiL1wiLCBhcGlFbmRQb2ludFVybFdpdGhvdXRQcm90b2NvbCkpO1xuXG5jb25zdCBzdGF0aWNSZXNvdXJjZSA9IGNvbnN0cnVjdC5hcGlHYXRld2F5LnJvb3QuYWRkUmVzb3VyY2UoJ3N0YXRpYycpO1xuY29uc3Qgc3RhdGljTWV0aG9kID0gc3RhdGljUmVzb3VyY2UuYWRkTWV0aG9kKCdHRVQnLCBuZXcgYXBpZ2F0ZXdheS5IdHRwSW50ZWdyYXRpb24oJ2h0dHA6Ly9hbWF6b24uY29tJykpO1xuXG5jb25zdCBkeW5hbWljUmVzb3VyY2UgPSBjb25zdHJ1Y3QuYXBpR2F0ZXdheS5yb290LmFkZFJlc291cmNlKCdkeW5hbWljJyk7XG5jb25zdCBkeW5hbWljTWV0aG9kID0gZHluYW1pY1Jlc291cmNlLmFkZE1ldGhvZCgnR0VUJyk7XG5cbi8vIEFkZCBiZWhhdmlvclxuY29uc3RydWN0LmNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24uYWRkQmVoYXZpb3IoJy9keW5hbWljJywgbmV3IG9yaWdpbnMuSHR0cE9yaWdpbihhcGlFbmRQb2ludERvbWFpbk5hbWUsIHtcbiAgICBvcmlnaW5QYXRoOiBgLyR7YXBpRW5kUG9pbnQuZGVwbG95bWVudFN0YWdlLnN0YWdlTmFtZX0vZHluYW1pY2BcbiAgICB9KSwge1xuICAgIGNhY2hlUG9saWN5OiBub0NhY2hlUG9saWN5XG59KTtcbi8vIFN1cHByZXNzIENGTl9OQUcgd2FybmluZ3NcbnN1cHByZXNzV2FybmluZ3Moc3RhdGljTWV0aG9kKTtcbnN1cHByZXNzV2FybmluZ3MoZHluYW1pY01ldGhvZCk7XG5cbmZ1bmN0aW9uIHN1cHByZXNzV2FybmluZ3MobWV0aG9kOiBhcGlnYXRld2F5Lk1ldGhvZCkge1xuICAgIGNvbnN0IGNoaWxkID0gbWV0aG9kLm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGFwaWdhdGV3YXkuQ2ZuTWV0aG9kO1xuICAgIGNoaWxkLmNmbk9wdGlvbnMubWV0YWRhdGEgPSB7XG4gICAgICAgIGNmbl9uYWc6IHtcbiAgICAgICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICAgICAgICAgIGlkOiAnVzU5JyxcbiAgICAgICAgICAgICAgICByZWFzb246IGBBV1M6OkFwaUdhdGV3YXk6Ok1ldGhvZCBBdXRob3JpemF0aW9uVHlwZSBpcyBzZXQgdG8gJ05PTkUnIGJlY2F1c2UgQVBJIEdhdGV3YXkgYmVoaW5kIENsb3VkRnJvbnQgZG9lcyBub3Qgc3VwcG9ydCBBV1NfSUFNIGF1dGhlbnRpY2F0aW9uYFxuICAgICAgICAgICAgfV1cbiAgICAgICAgfVxuICAgIH07XG59XG5cbi8vIFN5bnRoXG5hcHAuc3ludGgoKTtcbiJdfQ==