"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const core_1 = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.hours(8),
    minTtl: aws_cdk_lib_2.Duration.hours(5),
    maxTtl: aws_cdk_lib_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.minutes(0),
    minTtl: aws_cdk_lib_2.Duration.minutes(0),
    maxTtl: aws_cdk_lib_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.NONE,
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    }
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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