"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const defaults = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.hours(8),
    minTtl: aws_cdk_lib_2.Duration.hours(5),
    maxTtl: aws_cdk_lib_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.minutes(0),
    minTtl: aws_cdk_lib_2.Duration.minutes(0),
    maxTtl: aws_cdk_lib_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.CUSTOM,
            authorizer: core_1.CreateApiAuthorizer(stack, `${core_1.generateIntegStackName(__filename)}-authorizer`)
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    },
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    },
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
core_1.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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