"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const defaults = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, (0, core_1.generateIntegStackName)(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.hours(8),
    minTtl: aws_cdk_lib_2.Duration.hours(5),
    maxTtl: aws_cdk_lib_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.minutes(0),
    minTtl: aws_cdk_lib_2.Duration.minutes(0),
    maxTtl: aws_cdk_lib_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.CUSTOM,
            authorizer: (0, core_1.CreateApiAuthorizer)(stack, `${(0, core_1.generateIntegStackName)(__filename)}-authorizer`)
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    },
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    },
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
(0, core_1.suppressCustomHandlerCfnNagWarnings)(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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