"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
class LambdaToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableProps);
        }
        this.lambdaFunction.addEnvironment('DDB_TABLE_NAME', this.dynamoTable.tableName);
        this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        // Conditional metadata for cfn_nag
        if (((_a = props.dynamoTableProps) === null || _a === void 0 ? void 0 : _a.billingMode) === dynamodb.BillingMode.PROVISIONED) {
            const cfnTable = this.dynamoTable.node.findChild('Resource');
            cfnTable.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W73',
                            reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`
                        }]
                }
            };
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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