"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class LambdaToDynamoDB extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckDynamoDBProps(props);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        // Other permissions for constructs are accepted as arrays, turning tablePermissions into
        // an array to use the same validation function.
        if (props.tablePermissions) {
            defaults.CheckListValues(['All', 'Read', 'ReadWrite', 'Write'], [props.tablePermissions], 'table permission');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.DYNAMODB);
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc
        });
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can safely cast away the optional aspect of the type.
        const buildDynamoDBTableResponse = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.dynamoTable = buildDynamoDBTableResponse.tableObject;
        // Configure environment variables
        const tableEnvironmentVariableName = props.tableEnvironmentVariableName || 'DDB_TABLE_NAME';
        this.lambdaFunction.addEnvironment(tableEnvironmentVariableName, this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const permissions = props.tablePermissions.toUpperCase();
            if (permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (props.dynamoTableProps?.billingMode === dynamodb.BillingMode.PROVISIONED) {
            defaults.addCfnSuppressRules(this.dynamoTable, [
                {
                    id: "W73",
                    reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`,
                },
            ]);
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
LambdaToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-dynamodb.LambdaToDynamoDB", version: "2.97.0" };
//# sourceMappingURL=data:application/json;base64,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