"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
class LambdaToDynamoDB extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToDynamoDBProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckDynamoDBProps(props);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        // Other permissions for constructs are accepted as arrays, turning tablePermissions into
        // an array to use the same validation function.
        if (props.tablePermissions) {
            defaults.CheckListValues(['All', 'Read', 'ReadWrite', 'Write'], [props.tablePermissions], 'table permission');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.DYNAMODB);
        }
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc
        });
        // Since we are only invoking this function with an existing Table or tableProps,
        // (not a table interface), we know that the implementation will always return
        // a Table object and we can safely cast away the optional aspect of the type.
        const buildDynamoDBTableResponse = defaults.buildDynamoDBTable(this, {
            dynamoTableProps: props.dynamoTableProps,
            existingTableObj: props.existingTableObj
        });
        this.dynamoTable = buildDynamoDBTableResponse.tableObject;
        // Configure environment variables
        const tableEnvironmentVariableName = props.tableEnvironmentVariableName || 'DDB_TABLE_NAME';
        this.lambdaFunction.addEnvironment(tableEnvironmentVariableName, this.dynamoTable.tableName);
        // Add the requested or default table permissions
        if (props.tablePermissions) {
            const permissions = props.tablePermissions.toUpperCase();
            if (permissions === 'ALL') {
                this.dynamoTable.grantFullAccess(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READ') {
                this.dynamoTable.grantReadData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'READWRITE') {
                this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
            }
            else if (permissions === 'WRITE') {
                this.dynamoTable.grantWriteData(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.dynamoTable.grantReadWriteData(this.lambdaFunction.grantPrincipal);
        }
        // Conditional metadata for cfn_nag
        if (props.dynamoTableProps?.billingMode === dynamodb.BillingMode.PROVISIONED) {
            defaults.addCfnSuppressRules(this.dynamoTable, [
                {
                    id: "W73",
                    reason: `PROVISIONED billing mode is a default and is not explicitly applied as a setting.`,
                },
            ]);
        }
    }
}
exports.LambdaToDynamoDB = LambdaToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
LambdaToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-dynamodb.LambdaToDynamoDB", version: "2.98.0" };
//# sourceMappingURL=data:application/json;base64,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