"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sagemaker = require("@aws-cdk/aws-sagemaker");
const iam = require("@aws-cdk/aws-iam");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with new Lambda function
// --------------------------------------------------------------
test('Test deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSagemaker(stack, 'lambda-to-sagemaker-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deployed-function'
            }
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deployed-function'
            }
        }
    });
    // Assertion 3
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment with override
// --------------------------------------------------------------
test('Test deployment with override', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSagemaker(stack, 'lambda-to-sagemaker-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        deployInsideVpc: true,
        sagemakerNotebookProps: {
            subnetId: 'subnet-deadbeef',
            securityGroupIds: ['sg-deadbeef']
        }
    });
    expect(stack).toHaveResource("AWS::SageMaker::NotebookInstance", {
        DirectInternetAccess: "Disabled",
        SecurityGroupIds: [
            "sg-deadbeef"
        ],
        SubnetId: "subnet-deadbeef"
    });
});
// --------------------------------------------------------------
// Test deployment with existingNotebookObj
// --------------------------------------------------------------
test('Test deployment with existingNotebookObj', () => {
    // Stack
    const stack = new core_1.Stack();
    const sagemakerRole = new iam.Role(stack, "SagemakerRole", {
        assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
    });
    const _sagemaker = new sagemaker.CfnNotebookInstance(stack, 'MyNotebookInstance', {
        instanceType: 'ml.c4.2xlarge',
        roleArn: sagemakerRole.roleArn
    });
    const construct = new lib_1.LambdaToSagemaker(stack, 'lambda-to-sagemaker-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingNotebookObj: _sagemaker
    });
    expect(stack).toHaveResource("AWS::SageMaker::NotebookInstance", {
        InstanceType: "ml.c4.2xlarge",
    });
    expect(construct.vpc).toBeUndefined();
    expect(construct.securityGroup).toBeUndefined();
});
// --------------------------------------------------------------
// Test the properties
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSagemaker(stack, 'lambda-to-sagemaker-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        }
    });
    // Assertion 1
    expect(pattern.lambdaFunction !== null);
    // Assertion 2
    expect(pattern.sagemakerNotebook !== null);
    // Assertion 3
    expect(pattern.sagemakerRole !== null);
    // Assertion 4
    expect(pattern.vpc !== null);
    // Assertion 5
    expect(pattern.securityGroup !== null);
});
//# sourceMappingURL=data:application/json;base64,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