"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToSagemaker = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
/**
 * @summary The LambdaToSagemaker class.
 */
class LambdaToSagemaker extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the LambdaToSagemaker class.
     * @since 1.69.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Set up Sagemaker role
        this.sagemakerRole = new iam.Role(this, "SagemakerRole", {
            assumedBy: new iam.ServicePrincipal("sagemaker.amazonaws.com"),
        });
        // Build notebook instance
        [this.sagemakerNotebook, this.vpc, this.securityGroup] = defaults.buildSagemakerNotebook(this, {
            role: this.sagemakerRole,
            sagemakerNotebookProps: props.sagemakerNotebookProps,
            deployInsideVpc: props.deployInsideVpc,
            existingNotebookObj: props.existingNotebookObj
        });
        // Set up Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
        });
        // Attach Sagemaker policy to lambda role
        const policy = new iam.Policy(this, 'LambdaFunctionPolicy');
        const lambdaFunctionRole = this.lambdaFunction.role;
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/*`],
            actions: ['sagemaker:InvokeEndpoint']
        }));
        policy.attachToRole(lambdaFunctionRole);
    }
}
exports.LambdaToSagemaker = LambdaToSagemaker;
//# sourceMappingURL=data:application/json;base64,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