"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const ec2 = require("@aws-cdk/aws-ec2");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Test minimal deployment with new Lambda function
// --------------------------------------------------------------
test('Test minimal deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const secret = pattern.secret;
    expect(secret).toBeDefined();
});
// --------------------------------------------------------------
// Test deployment w/ existing secret
// --------------------------------------------------------------
test('Test deployment w/ existing secret', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const existingSecret = new aws_secretsmanager_1.Secret(stack, 'secret', {});
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        existingSecretObj: existingSecret
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
    // Assertion 2
    expect(pattern.secret).toBe(existingSecret);
});
// --------------------------------------------------------------
// Test deployment w/ existing function
// --------------------------------------------------------------
test('Test deployment w/ existing function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const existingFunction = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    const pattern = new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        existingLambdaObj: existingFunction,
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
    // Assertion 2
    expect(pattern.lambdaFunction).toBe(existingFunction);
});
// --------------------------------------------------------------
// Test minimal deployment with write access to Secret
// --------------------------------------------------------------
test('Test minimal deployment write access to Secret', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        grantWriteAccess: 'ReadWrite'
    });
    // Assertion 1
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
    });
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "192.68.0.0/16",
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: "index.handler",
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        existingVpc: testVpc,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosecretsmanagerstackReplaceDefaultSecurityGroupsecuritygroupED420155",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPC
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
            secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new core_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSecretsmanager(stack, "lambda-to-secretsmanager-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            }
        },
        secretProps: { removalPolicy: core_1.RemovalPolicy.DESTROY },
        secretEnvironmentVariableName: 'CUSTOM_SECRET_NAME'
    });
    // Assertion
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_SECRET_NAME: {
                    Ref: 'lambdatosecretsmanagerstacksecretBA684E34'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Test overriding secretProps to pass a customer provided CMK
// --------------------------------------------------------------
test('Test overriding secretProps to pass a customer provided CMK', () => {
    // Stack
    const stack = new core_1.Stack();
    const encryptionKey = defaults.buildEncryptionKey(stack, {
        description: 'secret-key'
    });
    // Helper declaration
    new lib_1.LambdaToSecretsmanager(stack, 'lambda-to-secretsmanager-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            }
        },
        secretProps: {
            encryptionKey
        }
    });
    // Assertion 1
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                SECRET_ARN: {
                    Ref: 'lambdatosecretsmanagerstacksecretBA684E34'
                }
            }
        }
    });
    // Assertion 2
    expect(stack).toHaveResource("AWS::SecretsManager::Secret", {
        GenerateSecretString: {},
        KmsKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        "kms:Create*",
                        "kms:Describe*",
                        "kms:Enable*",
                        "kms:List*",
                        "kms:Put*",
                        "kms:Update*",
                        "kms:Revoke*",
                        "kms:Disable*",
                        "kms:Get*",
                        "kms:Delete*",
                        "kms:ScheduleKeyDeletion",
                        "kms:CancelKeyDeletion",
                        "kms:GenerateDataKey",
                        "kms:TagResource",
                        "kms:UntagResource"
                    ],
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:Decrypt",
                        "kms:Encrypt",
                        "kms:ReEncrypt*",
                        "kms:GenerateDataKey*"
                    ],
                    Condition: {
                        StringEquals: {
                            "kms:ViaService": {
                                "Fn::Join": [
                                    "",
                                    [
                                        "secretsmanager.",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ".amazonaws.com"
                                    ]
                                ]
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: [
                        "kms:CreateGrant",
                        "kms:DescribeKey"
                    ],
                    Condition: {
                        StringEquals: {
                            "kms:ViaService": {
                                "Fn::Join": [
                                    "",
                                    [
                                        "secretsmanager.",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ".amazonaws.com"
                                    ]
                                ]
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: "*"
                },
                {
                    Action: "kms:Decrypt",
                    Condition: {
                        StringEquals: {
                            "kms:ViaService": {
                                "Fn::Join": [
                                    "",
                                    [
                                        "secretsmanager.",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ".amazonaws.com"
                                    ]
                                ]
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "lambdatosecretsmanagerstackLambdaFunctionServiceRole035B2C55",
                                "Arn"
                            ]
                        }
                    },
                    Resource: "*"
                }
            ],
            Version: "2012-10-17"
        },
        Description: "secret-key",
        EnableKeyRotation: true
    });
});
//# sourceMappingURL=data:application/json;base64,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