"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const defaults = require("@aws-solutions-constructs/core");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
// --------------------------------------------------------------
// Test minimal deployment
// --------------------------------------------------------------
test('Test minimal deployment', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for an producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'producer-function'
    });
    // Assertion 3: test for a consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'consumer-function'
    });
    // Assertion 4: test for a queue (queue and DLQ)
    template.resourceCountIs('AWS::SQS::Queue', 2);
    // Assertion 5: test for send-message permissions (only) on the producer function
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:SendMessage",
                        "sqs:GetQueueAttributes",
                        "sqs:GetQueueUrl"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
    // Assertion 6: test for consume-message permissions (only) on the consumer function
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "sqs:ReceiveMessage",
                        "sqs:ChangeMessageVisibility",
                        "sqs:GetQueueUrl",
                        "sqs:DeleteMessage",
                        "sqs:GetQueueAttributes"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "lambdasqslambdalambdatosqsqueue49588D68",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing producer function
// --------------------------------------------------------------
test('Test deployment w/ existing producer function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const existingProducerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'existing-producer-function'
        }
    });
    // Helper declaration
    const props = {
        existingProducerLambdaObj: existingProducerFn,
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'deployed-consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the existing producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'existing-producer-function'
    });
    // Assertion 3: test for the deployed consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'deployed-consumer-function'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing consumer function
// --------------------------------------------------------------
test('Test deployment w/ existing consumer function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const existingConsumerFn = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'existing-consumer-function'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'deployed-producer-function'
        },
        existingConsumerLambdaObj: existingConsumerFn
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the deployed producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'deployed-producer-function'
    });
    // Assertion 3: test for the existing consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'existing-consumer-function'
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Define existing resources
    const buildQueueResponse = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        existingQueueObj: buildQueueResponse.queue
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for the existing queue
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SQS::Queue', {
        QueueName: 'existing-queue'
    });
});
// --------------------------------------------------------------
// Test deployment w/ DLQ explicitly disabled
// --------------------------------------------------------------
test('Test deployment w/ DLQ explicitly disabled', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: false,
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for a non-existing DLQ
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::SQS::Queue', 1);
    defaults.expectNonexistence(stack, 'AWS::SQS::Queue', {
        RedrivePolicy: {}
    });
});
// --------------------------------------------------------------
// Test deployment w/ DLQ explicitly enabled and w/ MRC override
// --------------------------------------------------------------
test('Test deployment w/ DLQ explicitly enabled and w/ MRC override', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`)
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`)
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 6
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    // Assertion 2: test for an existing DLQ
    template.resourceCountIs('AWS::SQS::Queue', 2);
    template.hasResourceProperties('AWS::SQS::Queue', {
        RedrivePolicy: {}
    });
    // Assertion 3: test for the overridden max receive count
    template.hasResourceProperties('AWS::SQS::Queue', {
        RedrivePolicy: {
            maxReceiveCount: 6
        }
    });
});
// --------------------------------------------------------------
// Test overrides for producer and consumer functions
// --------------------------------------------------------------
test('Test overrides for producer and consumer functions', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 2: test for updated runtime on producer function
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Runtime: "nodejs14.x"
    });
    // Assertion 3: test for updated runtime on consumer function
    template.hasResourceProperties('AWS::Lambda::Function', {
        Runtime: "nodejs14.x"
    });
});
// --------------------------------------------------------------
// Test the public pattern props
// --------------------------------------------------------------
test('Test the public pattern props', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        }
    };
    const pattern = new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion 1: get the producer function
    expect(pattern.producerLambdaFunction).toBeDefined();
    // Assertion 2: get the queue
    expect(pattern.sqsQueue).toBeDefined();
    // Assertion 3: get the dead letter queue
    expect(pattern.deadLetterQueue).toBeDefined();
    // Assertion 4: get the consumer function
    expect(pattern.consumerLambdaFunction).toBeDefined();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        queueEnvironmentVariableName: 'CUSTOM_QUEUE_NAME'
    };
    new lib_1.LambdaToSqsToLambda(stack, 'lambda-sqs-lambda', props);
    // Assertion
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'producer-function',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_QUEUE_NAME: {
                    Ref: 'lambdasqslambdalambdatosqsqueue49588D68'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Pattern deployment w/ batch size
// --------------------------------------------------------------
test('Pattern deployment w/ batch size', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        sqsEventSourceProps: {
            batchSize: 5
        }
    };
    new lib_1.LambdaToSqsToLambda(stack, 'test-lambda-sqs-lambda', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::EventSourceMapping', {
        BatchSize: 5
    });
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet1Subnet70F24179",
                },
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet2Subnet1D6A3FAF",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet1Subnet70F24179",
                },
                {
                    Ref: "lambdatosqstolambdastackVpcisolatedSubnet2Subnet1D6A3FAF",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
        producerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
            functionName: 'producer-function'
        },
        consumerLambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
            functionName: 'consumer-function'
        },
        existingVpc: testVpc,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatosqstolambdastacklambdatosqsReplaceDefaultSecurityGroupsecuritygroup90A497DF",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSqsToLambda(stack, "lambda-to-sqs-to-lambda-stack", {
            producerLambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda/producer-function`),
                functionName: 'producer-function'
            },
            consumerLambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_14_X,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda/consumer-function`),
                functionName: 'consumer-function'
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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