"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const ec2 = require("aws-cdk-lib/aws-ec2");
const lib_1 = require("../lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const defaults = require("@aws-solutions-constructs/core");
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
            }
        },
        stringParameterProps: { stringValue: "test-string-value" },
        stringParameterEnvironmentVariableName: 'CUSTOM_SSM_PARAMETER_NAME'
    });
    // Assertion
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_SSM_PARAMETER_NAME: {
                    Ref: 'lambdatossmstackstringParameterA6E27D57'
                }
            }
        }
    });
});
test('Test the properties', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const pattern = new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stringParameterProps: { stringValue: "test-string-value" },
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const stringParam = pattern.stringParameter;
    expect(stringParam).toBeDefined();
});
test('Test deployment w/ existing String Parameter', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const existingStringParam = new aws_ssm_1.StringParameter(stack, 'myNewStringParameter', { stringValue: "test-string-value" });
    const pattern = new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        existingStringParameterObj: existingStringParam
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SSM::Parameter", {
        Type: "String",
        Value: "test-string-value"
    });
    // Assertion 2
    expect(pattern.stringParameter).toBe(existingStringParam);
});
test('Test deployment w/ existing function', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const lambdaFunctionProps = {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const existingFunction = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    const pattern = new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        existingLambdaObj: existingFunction,
        stringParameterProps: { stringValue: "test-string-value" },
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SSM::Parameter", {
        Type: "String",
        Value: "test-string-value"
    });
    // Assertion 2
    expect(pattern.lambdaFunction).toBe(existingFunction);
});
test('Test minimal deployment write access to String Parameter ', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        stringParameterProps: { stringValue: "test-string-value" },
        stringParameterPermissions: 'ReadWrite'
    });
    // Assertion 1
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::SSM::Parameter", {
        Type: "String",
        Value: "test-string-value"
    });
});
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stringParameterProps: { stringValue: "test-string-value" },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatossmstackReplaceDefaultSecurityGroupsecuritygroupD1E88D13",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stringParameterProps: { stringValue: "test-string-value" },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            ipAddresses: ec2.IpAddresses.cidr("192.68.0.0/16"),
        },
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatossmstackReplaceDefaultSecurityGroupsecuritygroupD1E88D13",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    template.resourceCountIs("AWS::EC2::Subnet", 2);
    template.resourceCountIs("AWS::EC2::InternetGateway", 0);
});
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToSsmstringparameter(stack, 'lambda-to-ssm-stack', {
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stringParameterProps: { stringValue: "test-string-value" },
        existingVpc: testVpc,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatossmstackReplaceDefaultSecurityGroupsecuritygroupD1E88D13",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // buildLambdaFunction should throw an error if the Lambda function is not
        // attached to a VPC
        new lib_1.LambdaToSsmstringparameter(stack, "lambda-to-ssm-stack", {
            existingLambdaObj: testLambdaFunction,
            existingVpc: testVpc,
            stringParameterProps: { stringValue: "test-string-value" }
        });
    };
    // Assertion
    expect(app).toThrowError();
});
test("Confirm that CheckVpcProps is called", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSsmstringparameter(stack, "lambda-to-ssm-stack", {
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            stringParameterProps: { stringValue: "test-string-value" },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test("Test bad call with invalid string parameter permission", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToSsmstringparameter(stack, "lambda-to-ssm-stack", {
            lambdaFunctionProps: {
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: "index.handler",
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            },
            stringParameterProps: { stringValue: "test-string-value" },
            deployVpc: true,
            stringParameterPermissions: 'Reed',
        });
    };
    // Assertion
    expect(app).toThrowError('Invalid String Parameter permission submitted - Reed');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        stringParameterProps: { stringValue: "test-string-value" },
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToSsmstringparameter(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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