"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const stepfunctions = require("@aws-cdk/aws-stepfunctions");
const ec2 = require("@aws-cdk/aws-ec2");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test deployment with new Lambda function
// --------------------------------------------------------------
test('Test deployment with new Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    new lib_1.LambdaToStepfunctions(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'deploy-function'
            }
        },
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'deploy-function',
                STATE_MACHINE_ARN: {
                    Ref: 'lambdatostepfunctionstackStateMachine98EE8EFB'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Test deployment with existing Lambda function
// --------------------------------------------------------------
test('Test deployment with existing Lambda function', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        environment: {
            LAMBDA_NAME: 'existing-function'
        }
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    // Add the pattern
    new lib_1.LambdaToStepfunctions(stack, 'test-lambda-step-function-construct', {
        existingLambdaObj: fn,
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                LAMBDA_NAME: 'existing-function'
            }
        }
    });
});
// --------------------------------------------------------------
// Test invocation permissions
// --------------------------------------------------------------
test('Test invocation permissions', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        environment: {
            LAMBDA_NAME: 'existing-function'
        }
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    // Add the pattern
    new lib_1.LambdaToStepfunctions(stack, 'test-lambda-step-function-stack', {
        existingLambdaObj: fn,
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testlambdastepfunctionstackStateMachine373C0BB9"
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
// --------------------------------------------------------------
// Test the properties
// --------------------------------------------------------------
test('Test the properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const pattern = new lib_1.LambdaToStepfunctions(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'existing-function'
            }
        },
        stateMachineProps: {
            definition: startState
        }
    });
    // Assertion 1
    const func = pattern.lambdaFunction;
    expect(func).toBeDefined();
    // Assertion 2
    const stateMachine = pattern.stateMachine;
    expect(stateMachine).toBeDefined();
    // Assertion 3
    const cwAlarm = pattern.cloudwatchAlarms;
    expect(cwAlarm).toBeDefined();
    expect(pattern.stateMachineLogGroup).toBeDefined();
});
// --------------------------------------------------------------
// Test the properties
// --------------------------------------------------------------
test('Test the properties with no CW Alarms', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const pattern = new lib_1.LambdaToStepfunctions(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            environment: {
                LAMBDA_NAME: 'existing-function'
            }
        },
        stateMachineProps: {
            definition: startState
        },
        createCloudWatchAlarms: false
    });
    // Assertion 1
    expect(pattern.lambdaFunction).toBeDefined();
    // Assertion 2
    expect(pattern.stateMachine).toBeDefined();
    // Assertion 3
    expect(pattern.cloudwatchAlarms).toBeUndefined();
    expect(pattern.stateMachineLogGroup).toBeDefined();
});
// --------------------------------------------------------------
// Test lambda function custom environment variable
// --------------------------------------------------------------
test('Test lambda function custom environment variable', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const startState = new stepfunctions.Pass(stack, 'StartState');
    new lib_1.LambdaToStepfunctions(stack, 'lambda-to-step-function-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stateMachineProps: {
            definition: startState
        },
        stateMachineEnvironmentVariableName: 'CUSTOM_STATE_MAHINCE'
    });
    // Assertion
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_STATE_MAHINCE: {
                    Ref: 'lambdatostepfunctionstackStateMachine98EE8EFB'
                }
            }
        }
    });
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC without vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC without vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    const startState = new stepfunctions.Pass(stack, 'StartState');
    // Helper declaration
    new lib_1.LambdaToStepfunctions(stack, "lambda-to-stepfunctions-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stateMachineProps: {
            definition: startState
        },
        deployVpc: true
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatostepfunctionsstackReplaceDefaultSecurityGroupsecuritygroup0F25B19B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment that deploys a VPC w/vpcProps
// --------------------------------------------------------------
test("Test minimal deployment that deploys a VPC w/vpcProps", () => {
    // Stack
    const stack = new core_1.Stack();
    const startState = new stepfunctions.Pass(stack, 'StartState');
    // Helper declaration
    new lib_1.LambdaToStepfunctions(stack, "lambda-to-stepfunctions-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stateMachineProps: {
            definition: startState
        },
        vpcProps: {
            enableDnsHostnames: false,
            enableDnsSupport: false,
            cidr: "192.68.0.0/16",
        },
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatostepfunctionsstackReplaceDefaultSecurityGroupsecuritygroup0F25B19B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        CidrBlock: "192.68.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toCountResources("AWS::EC2::Subnet", 2);
    expect(stack).toCountResources("AWS::EC2::InternetGateway", 0);
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    new lib_1.LambdaToStepfunctions(stack, "lambda-to-stepfunctions-stack", {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`)
        },
        stateMachineProps: {
            definition: startState
        },
        existingVpc: testVpc,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdatostepfunctionsstackReplaceDefaultSecurityGroupsecuritygroup0F25B19B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "testvpcPrivateSubnet1Subnet865FB50A",
                },
                {
                    Ref: "testvpcPrivateSubnet2Subnet23D3396F",
                },
            ],
        },
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
});
// --------------------------------------------------------------
// Test minimal deployment with an existing VPC and existing Lambda function not in a VPC
//
// buildLambdaFunction should throw an error if the Lambda function is not
// attached to a VPC
// --------------------------------------------------------------
test("Test minimal deployment with an existing VPC and existing Lambda function not in a VPC", () => {
    // Stack
    const stack = new core_1.Stack();
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const testLambdaFunction = new lambda.Function(stack, 'test-lamba', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: "index.handler",
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToStepfunctions(stack, "lambda-to-stepfunctions-stack", {
            existingLambdaObj: testLambdaFunction,
            stateMachineProps: {
                definition: startState
            },
            existingVpc: testVpc,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Test bad call with existingVpc and deployVpc
// --------------------------------------------------------------
test("Test bad call with existingVpc and deployVpc", () => {
    // Stack
    const stack = new core_1.Stack();
    const startState = new stepfunctions.Pass(stack, 'StartState');
    const testVpc = new ec2.Vpc(stack, "test-vpc", {});
    const app = () => {
        // Helper declaration
        new lib_1.LambdaToStepfunctions(stack, "lambda-to-stepfunctions-stack", {
            lambdaFunctionProps: {
                runtime: lambda.Runtime.NODEJS_10_X,
                handler: 'index.handler',
                code: lambda.Code.fromAsset(`${__dirname}/lambda`)
            },
            stateMachineProps: {
                definition: startState
            },
            existingVpc: testVpc,
            deployVpc: true,
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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